"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.isJobHandler = exports.JobState = exports.JobOutboundMessageKind = exports.JobInboundMessageKind = void 0;
/**
 * Messages that can be sent TO a job. The job needs to listen to those.
 */
var JobInboundMessageKind;
(function (JobInboundMessageKind) {
    JobInboundMessageKind["Ping"] = "ip";
    JobInboundMessageKind["Stop"] = "is";
    // Channel specific messages.
    JobInboundMessageKind["Input"] = "in";
    // Input channel does not allow completion / error. Erroring this will just close the Subject
    // but not notify the job.
})(JobInboundMessageKind || (exports.JobInboundMessageKind = JobInboundMessageKind = {}));
/**
 * Kind of messages that can be outputted from a job.
 */
var JobOutboundMessageKind;
(function (JobOutboundMessageKind) {
    // Lifecycle specific messages.
    JobOutboundMessageKind["OnReady"] = "c";
    JobOutboundMessageKind["Start"] = "s";
    JobOutboundMessageKind["End"] = "e";
    JobOutboundMessageKind["Pong"] = "p";
    // Feedback messages.
    JobOutboundMessageKind["Output"] = "o";
    // Channel specific messages.
    JobOutboundMessageKind["ChannelCreate"] = "cn";
    JobOutboundMessageKind["ChannelMessage"] = "cm";
    JobOutboundMessageKind["ChannelError"] = "ce";
    JobOutboundMessageKind["ChannelComplete"] = "cc";
})(JobOutboundMessageKind || (exports.JobOutboundMessageKind = JobOutboundMessageKind = {}));
/**
 * The state of a job. These are changed as the job reports a new state through its messages.
 */
var JobState;
(function (JobState) {
    /**
     * The job was queued and is waiting to start.
     */
    JobState["Queued"] = "queued";
    /**
     * The job description was found, its dependencies (see "Synchronizing and Dependencies")
     * are done running, and the job's argument is validated and the job's code will be executed.
     */
    JobState["Ready"] = "ready";
    /**
     * The job has been started. The job implementation is expected to send this as soon as its
     * work is starting.
     */
    JobState["Started"] = "started";
    /**
     * The job has ended and is done running.
     */
    JobState["Ended"] = "ended";
    /**
     * An error occured and the job stopped because of internal state.
     */
    JobState["Errored"] = "errored";
})(JobState || (exports.JobState = JobState = {}));
function isJobHandler(value) {
    const job = value;
    return (typeof job == 'function' && typeof job.jobDescription == 'object' && job.jobDescription !== null);
}
exports.isJobHandler = isJobHandler;
//# sourceMappingURL=data:application/json;base64,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