"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.executeBuild = void 0;
const node_assert_1 = __importDefault(require("node:assert"));
const compiler_plugin_1 = require("../../tools/esbuild/angular/compiler-plugin");
const application_code_bundle_1 = require("../../tools/esbuild/application-code-bundle");
const bundler_context_1 = require("../../tools/esbuild/bundler-context");
const bundler_execution_result_1 = require("../../tools/esbuild/bundler-execution-result");
const commonjs_checker_1 = require("../../tools/esbuild/commonjs-checker");
const global_scripts_1 = require("../../tools/esbuild/global-scripts");
const global_styles_1 = require("../../tools/esbuild/global-styles");
const index_html_generator_1 = require("../../tools/esbuild/index-html-generator");
const license_extractor_1 = require("../../tools/esbuild/license-extractor");
const utils_1 = require("../../tools/esbuild/utils");
const copy_assets_1 = require("../../utils/copy-assets");
const environment_options_1 = require("../../utils/environment-options");
const prerender_1 = require("../../utils/server-rendering/prerender");
const service_worker_1 = require("../../utils/service-worker");
const supported_browsers_1 = require("../../utils/supported-browsers");
// eslint-disable-next-line max-lines-per-function
async function executeBuild(options, context, rebuildState) {
    const startTime = process.hrtime.bigint();
    const { projectRoot, workspaceRoot, serviceWorker, optimizationOptions, serverEntryPoint, assets, indexHtmlOptions, cacheOptions, prerenderOptions, appShellOptions, ssrOptions, } = options;
    const browsers = (0, supported_browsers_1.getSupportedBrowsers)(projectRoot, context.logger);
    const target = (0, utils_1.transformSupportedBrowsersToTargets)(browsers);
    // Reuse rebuild state or create new bundle contexts for code and global stylesheets
    let bundlerContexts = rebuildState?.rebuildContexts;
    const codeBundleCache = rebuildState?.codeBundleCache ??
        new compiler_plugin_1.SourceFileCache(cacheOptions.enabled ? cacheOptions.path : undefined);
    if (bundlerContexts === undefined) {
        bundlerContexts = [];
        // Browser application code
        bundlerContexts.push(new bundler_context_1.BundlerContext(workspaceRoot, !!options.watch, (0, application_code_bundle_1.createBrowserCodeBundleOptions)(options, target, codeBundleCache)));
        // Global Stylesheets
        if (options.globalStyles.length > 0) {
            for (const initial of [true, false]) {
                const bundleOptions = (0, global_styles_1.createGlobalStylesBundleOptions)(options, target, initial, codeBundleCache?.loadResultCache);
                if (bundleOptions) {
                    bundlerContexts.push(new bundler_context_1.BundlerContext(workspaceRoot, !!options.watch, bundleOptions, () => initial));
                }
            }
        }
        // Global Scripts
        if (options.globalScripts.length > 0) {
            for (const initial of [true, false]) {
                const bundleOptions = (0, global_scripts_1.createGlobalScriptsBundleOptions)(options, initial);
                if (bundleOptions) {
                    bundlerContexts.push(new bundler_context_1.BundlerContext(workspaceRoot, !!options.watch, bundleOptions, () => initial));
                }
            }
        }
        // Server application code
        if (serverEntryPoint) {
            bundlerContexts.push(new bundler_context_1.BundlerContext(workspaceRoot, !!options.watch, (0, application_code_bundle_1.createServerCodeBundleOptions)(options, 
            // NOTE: earlier versions of Node.js are not supported due to unsafe promise patching.
            // See: https://github.com/angular/angular/pull/50552#issue-1737967592
            [...target, 'node18.13'], codeBundleCache), () => false));
        }
    }
    const bundlingResult = await bundler_context_1.BundlerContext.bundleAll(bundlerContexts);
    // Log all warnings and errors generated during bundling
    await (0, utils_1.logMessages)(context, bundlingResult);
    const executionResult = new bundler_execution_result_1.ExecutionResult(bundlerContexts, codeBundleCache);
    // Return if the bundling has errors
    if (bundlingResult.errors) {
        return executionResult;
    }
    const { metafile, initialFiles, outputFiles } = bundlingResult;
    executionResult.outputFiles.push(...outputFiles);
    // Check metafile for CommonJS module usage if optimizing scripts
    if (optimizationOptions.scripts) {
        const messages = (0, commonjs_checker_1.checkCommonJSModules)(metafile, options.allowedCommonJsDependencies);
        await (0, utils_1.logMessages)(context, { warnings: messages });
    }
    /**
     * Index HTML content without CSS inlining to be used for server rendering (AppShell, SSG and SSR).
     *
     * NOTE: we don't perform critical CSS inlining as this will be done during server rendering.
     */
    let indexContentOutputNoCssInlining;
    // Generate index HTML file
    if (indexHtmlOptions) {
        const { content, contentWithoutCriticalCssInlined, errors, warnings } = await (0, index_html_generator_1.generateIndexHtml)(initialFiles, executionResult, {
            ...options,
            optimizationOptions,
        });
        indexContentOutputNoCssInlining = contentWithoutCriticalCssInlined;
        printWarningsAndErrorsToConsole(context, warnings, errors);
        executionResult.addOutputFile(indexHtmlOptions.output, content);
        if (ssrOptions) {
            executionResult.addOutputFile('index.server.html', contentWithoutCriticalCssInlined);
        }
    }
    // Pre-render (SSG) and App-shell
    if (prerenderOptions || appShellOptions) {
        (0, node_assert_1.default)(indexContentOutputNoCssInlining, 'The "index" option is required when using the "ssg" or "appShell" options.');
        const { output, warnings, errors } = await (0, prerender_1.prerenderPages)(workspaceRoot, options.tsconfig, appShellOptions, prerenderOptions, executionResult.outputFiles, indexContentOutputNoCssInlining, optimizationOptions.styles.inlineCritical, environment_options_1.maxWorkers);
        printWarningsAndErrorsToConsole(context, warnings, errors);
        for (const [path, content] of Object.entries(output)) {
            executionResult.addOutputFile(path, content);
        }
    }
    // Copy assets
    if (assets) {
        // The webpack copy assets helper is used with no base paths defined. This prevents the helper
        // from directly writing to disk. This should eventually be replaced with a more optimized helper.
        executionResult.assetFiles.push(...(await (0, copy_assets_1.copyAssets)(assets, [], workspaceRoot)));
    }
    // Write metafile if stats option is enabled
    if (options.stats) {
        executionResult.addOutputFile('stats.json', JSON.stringify(metafile, null, 2));
    }
    // Extract and write licenses for used packages
    if (options.extractLicenses) {
        executionResult.addOutputFile('3rdpartylicenses.txt', await (0, license_extractor_1.extractLicenses)(metafile, workspaceRoot));
    }
    // Augment the application with service worker support
    if (serviceWorker) {
        try {
            const serviceWorkerResult = await (0, service_worker_1.augmentAppWithServiceWorkerEsbuild)(workspaceRoot, serviceWorker, options.baseHref || '/', executionResult.outputFiles, executionResult.assetFiles);
            executionResult.addOutputFile('ngsw.json', serviceWorkerResult.manifest);
            executionResult.assetFiles.push(...serviceWorkerResult.assetFiles);
        }
        catch (error) {
            context.logger.error(error instanceof Error ? error.message : `${error}`);
            return executionResult;
        }
    }
    // Calculate estimated transfer size if scripts are optimized
    let estimatedTransferSizes;
    if (optimizationOptions.scripts || optimizationOptions.styles.minify) {
        estimatedTransferSizes = await (0, utils_1.calculateEstimatedTransferSizes)(executionResult.outputFiles);
    }
    (0, utils_1.logBuildStats)(context, metafile, initialFiles, estimatedTransferSizes);
    const buildTime = Number(process.hrtime.bigint() - startTime) / 10 ** 9;
    context.logger.info(`Application bundle generation complete. [${buildTime.toFixed(3)} seconds]`);
    return executionResult;
}
exports.executeBuild = executeBuild;
function printWarningsAndErrorsToConsole(context, warnings, errors) {
    for (const error of errors) {
        context.logger.error(error);
    }
    for (const warning of warnings) {
        context.logger.warn(warning);
    }
}
//# sourceMappingURL=data:application/json;base64,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