"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createAngularCompilerHost = exports.ensureSourceFileVersions = void 0;
const typescript_1 = __importDefault(require("typescript"));
// Temporary deep import for host augmentation support.
// TODO: Move these to a private exports location or move the implementation into this package.
const { augmentHostWithCaching, augmentHostWithReplacements, augmentProgramWithVersioning, } = require('@ngtools/webpack/src/ivy/host');
/**
 * Patches in-place the `getSourceFiles` function on an instance of a TypeScript
 * `Program` to ensure that all returned SourceFile instances have a `version`
 * field. The `version` field is required when used with a TypeScript BuilderProgram.
 * @param program The TypeScript Program instance to patch.
 */
function ensureSourceFileVersions(program) {
    augmentProgramWithVersioning(program);
}
exports.ensureSourceFileVersions = ensureSourceFileVersions;
function createAngularCompilerHost(compilerOptions, hostOptions) {
    // Create TypeScript compiler host
    const host = typescript_1.default.createIncrementalCompilerHost(compilerOptions);
    // The AOT compiler currently requires this hook to allow for a transformResource hook.
    // Once the AOT compiler allows only a transformResource hook, this can be reevaluated.
    host.readResource = async function (filename) {
        return this.readFile(filename) ?? '';
    };
    // Add an AOT compiler resource transform hook
    host.transformResource = async function (data, context) {
        // Only style resources are transformed currently
        if (context.type !== 'style') {
            return null;
        }
        const result = await hostOptions.transformStylesheet(data, context.containingFile, context.resourceFile ?? undefined);
        return result ? { content: result } : null;
    };
    // Allow the AOT compiler to request the set of changed templates and styles
    host.getModifiedResourceFiles = function () {
        return hostOptions.modifiedFiles;
    };
    // Augment TypeScript Host for file replacements option
    if (hostOptions.fileReplacements) {
        // Provide a resolution cache since overriding resolution prevents automatic creation
        const resolutionCache = typescript_1.default.createModuleResolutionCache(host.getCurrentDirectory(), host.getCanonicalFileName.bind(host), compilerOptions);
        host.getModuleResolutionCache = () => resolutionCache;
        augmentHostWithReplacements(host, hostOptions.fileReplacements, resolutionCache);
    }
    // Augment TypeScript Host with source file caching if provided
    if (hostOptions.sourceFileCache) {
        augmentHostWithCaching(host, hostOptions.sourceFileCache);
    }
    return host;
}
exports.createAngularCompilerHost = createAngularCompilerHost;
//# sourceMappingURL=data:application/json;base64,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