"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseJitUri = exports.generateJitInlineUri = exports.generateJitFileUri = exports.JIT_NAMESPACE_REGEXP = exports.JIT_TEMPLATE_NAMESPACE = exports.JIT_STYLE_NAMESPACE = void 0;
/**
 * A string value representing the base namespace for Angular JIT mode related imports.
 */
const JIT_BASE_NAMESPACE = 'angular:jit';
/**
 * A string value representing the namespace for Angular JIT mode related imports for
 * Component styles. This namespace is used for both inline (`styles`) and external
 * (`styleUrls`) styles.
 */
exports.JIT_STYLE_NAMESPACE = `${JIT_BASE_NAMESPACE}:style`;
/**
 * A string value representing the namespace for Angular JIT mode related imports for
 * Component templates. This namespace is currently only used for external (`templateUrl`)
 * templates.
 */
exports.JIT_TEMPLATE_NAMESPACE = `${JIT_BASE_NAMESPACE}:template`;
/**
 * A regular expression that can be used to match a Angular JIT mode namespace URI.
 * It contains capture groups for the type (template/style), origin (file/inline), and specifier.
 * The {@link parseJitUri} function can be used to parse and return an object representation of a JIT URI.
 */
exports.JIT_NAMESPACE_REGEXP = new RegExp(`^${JIT_BASE_NAMESPACE}:(template|style):(file|inline);(.*)$`);
/**
 * Generates an Angular JIT mode namespace URI for a given file.
 * @param file The path of the file to be included.
 * @param type The type of the file (`style` or `template`).
 * @returns A string containing the full JIT namespace URI.
 */
function generateJitFileUri(file, type) {
    return `${JIT_BASE_NAMESPACE}:${type}:file;${file}`;
}
exports.generateJitFileUri = generateJitFileUri;
/**
 * Generates an Angular JIT mode namespace URI for a given inline style or template.
 * The provided content is base64 encoded and included in the URI.
 * @param data The content to encode within the URI.
 * @param type The type of the content (`style` or `template`).
 * @returns A string containing the full JIT namespace URI.
 */
function generateJitInlineUri(data, type) {
    return `${JIT_BASE_NAMESPACE}:${type}:inline;${Buffer.from(data).toString('base64')}`;
}
exports.generateJitInlineUri = generateJitInlineUri;
/**
 * Parses a string containing a JIT namespace URI.
 * JIT namespace URIs are used to encode the information for an Angular component's stylesheets
 * and templates when compiled in JIT mode.
 * @param uri The URI to parse into its underlying components.
 * @returns An object containing the namespace, type, origin, and specifier of the URI;
 * `undefined` if not a JIT namespace URI.
 */
function parseJitUri(uri) {
    const matches = exports.JIT_NAMESPACE_REGEXP.exec(uri);
    if (!matches) {
        return undefined;
    }
    return {
        namespace: `${JIT_BASE_NAMESPACE}:${matches[1]}`,
        type: matches[1],
        origin: matches[2],
        specifier: matches[3],
    };
}
exports.parseJitUri = parseJitUri;
//# sourceMappingURL=data:application/json;base64,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