"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderPage = void 0;
const node_path_1 = require("node:path");
const inline_critical_css_1 = require("../index-file/inline-critical-css");
const load_esm_1 = require("../load-esm");
/**
 * Renders each route in routes and writes them to <outputPath>/<route>/index.html.
 */
async function renderPage({ route, serverContext, document, inlineCriticalCss, outputFiles, loadBundle = load_esm_1.loadEsmModule, }) {
    const { default: bootstrapAppFnOrModule, ɵSERVER_CONTEXT, renderModule, renderApplication, } = await loadBundle('./main.server.mjs');
    const platformProviders = [
        {
            provide: ɵSERVER_CONTEXT,
            useValue: serverContext,
        },
    ];
    let html;
    if (isBootstrapFn(bootstrapAppFnOrModule)) {
        html = await renderApplication(bootstrapAppFnOrModule, {
            document,
            url: route,
            platformProviders,
        });
    }
    else {
        html = await renderModule(bootstrapAppFnOrModule, {
            document,
            url: route,
            extraProviders: platformProviders,
        });
    }
    if (inlineCriticalCss) {
        const inlineCriticalCssProcessor = new inline_critical_css_1.InlineCriticalCssProcessor({
            minify: false,
            readAsset: async (filePath) => {
                filePath = (0, node_path_1.basename)(filePath);
                const content = outputFiles[filePath];
                if (content === undefined) {
                    throw new Error(`Output file does not exist: ${filePath}`);
                }
                return content;
            },
        });
        return inlineCriticalCssProcessor.process(html, { outputPath: '' });
    }
    return {
        content: html,
    };
}
exports.renderPage = renderPage;
function isBootstrapFn(value) {
    // We can differentiate between a module and a bootstrap function by reading `cmp`:
    return typeof value === 'function' && !('ɵmod' in value);
}
//# sourceMappingURL=data:application/json;base64,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