"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.runWebpack = void 0;
const architect_1 = require("@angular-devkit/architect");
const path_1 = require("path");
const rxjs_1 = require("rxjs");
const webpack_1 = __importDefault(require("webpack"));
const utils_1 = require("../utils");
function runWebpack(config, context, options = {}) {
    const { logging: log = (stats, config) => context.logger.info(stats.toString(config.stats)), shouldProvideStats = true, } = options;
    const createWebpack = (c) => {
        if (options.webpackFactory) {
            const result = options.webpackFactory(c);
            if ((0, rxjs_1.isObservable)(result)) {
                return result;
            }
            else {
                return (0, rxjs_1.of)(result);
            }
        }
        else {
            return (0, rxjs_1.of)((0, webpack_1.default)(c));
        }
    };
    return createWebpack({ ...config, watch: false }).pipe((0, rxjs_1.switchMap)((webpackCompiler) => new rxjs_1.Observable((obs) => {
        const callback = (err, stats) => {
            if (err) {
                return obs.error(err);
            }
            if (!stats) {
                return;
            }
            // Log stats.
            log(stats, config);
            const statsOptions = typeof config.stats === 'boolean' ? undefined : config.stats;
            const result = {
                success: !stats.hasErrors(),
                webpackStats: shouldProvideStats ? stats.toJson(statsOptions) : undefined,
                emittedFiles: (0, utils_1.getEmittedFiles)(stats.compilation),
                outputPath: stats.compilation.outputOptions.path,
            };
            if (config.watch) {
                obs.next(result);
            }
            else {
                webpackCompiler.close(() => {
                    obs.next(result);
                    obs.complete();
                });
            }
        };
        try {
            if (config.watch) {
                const watchOptions = config.watchOptions || {};
                const watching = webpackCompiler.watch(watchOptions, callback);
                // Teardown logic. Close the watcher when unsubscribed from.
                return () => {
                    watching.close(() => { });
                    webpackCompiler.close(() => { });
                };
            }
            else {
                webpackCompiler.run(callback);
            }
        }
        catch (err) {
            if (err) {
                context.logger.error(`\nAn error occurred during the build:\n${err instanceof Error ? err.stack : err}`);
            }
            throw err;
        }
    })));
}
exports.runWebpack = runWebpack;
exports.default = (0, architect_1.createBuilder)((options, context) => {
    const configPath = (0, path_1.resolve)(context.workspaceRoot, options.webpackConfig);
    return (0, rxjs_1.from)((0, utils_1.getWebpackConfig)(configPath)).pipe((0, rxjs_1.switchMap)((config) => runWebpack(config, context)));
});
//# sourceMappingURL=data:application/json;base64,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