"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RULE_NAME = void 0;
const bundled_angular_compiler_1 = require("@angular-eslint/bundled-angular-compiler");
const utils_1 = require("@angular-eslint/utils");
const create_eslint_rule_1 = require("../utils/create-eslint-rule");
exports.RULE_NAME = 'attributes-order';
const DEFAULT_ORDER = [
    "STRUCTURAL_DIRECTIVE" /* OrderType.StructuralDirective */,
    "TEMPLATE_REFERENCE" /* OrderType.TemplateReferenceVariable */,
    "ATTRIBUTE_BINDING" /* OrderType.AttributeBinding */,
    "INPUT_BINDING" /* OrderType.InputBinding */,
    "TWO_WAY_BINDING" /* OrderType.TwoWayBinding */,
    "OUTPUT_BINDING" /* OrderType.OutputBinding */,
];
const DEFAULT_OPTIONS = {
    alphabetical: false,
    order: [...DEFAULT_ORDER],
};
exports.default = (0, create_eslint_rule_1.createESLintRule)({
    name: exports.RULE_NAME,
    meta: {
        type: 'layout',
        docs: {
            description: 'Ensures that HTML attributes and Angular bindings are sorted based on an expected order',
            recommended: false,
        },
        fixable: 'code',
        schema: [
            {
                type: 'object',
                properties: {
                    alphabetical: {
                        type: 'boolean',
                        default: DEFAULT_OPTIONS.alphabetical,
                    },
                    order: {
                        type: 'array',
                        items: {
                            enum: DEFAULT_OPTIONS.order,
                        },
                        default: DEFAULT_OPTIONS.order,
                        minItems: DEFAULT_OPTIONS.order.length,
                        uniqueItems: true,
                    },
                },
                additionalProperties: false,
            },
        ],
        messages: {
            attributesOrder: `The element's attributes/bindings did not match the expected order: expected {{expected}} instead of {{actual}}`,
        },
    },
    defaultOptions: [DEFAULT_OPTIONS],
    create(context, [{ alphabetical, order }]) {
        const parserServices = (0, utils_1.getTemplateParserServices)(context);
        function getLocation(attr) {
            const loc = parserServices.convertNodeSourceSpanToLoc(attr.sourceSpan);
            switch (attr.orderType) {
                case "STRUCTURAL_DIRECTIVE" /* OrderType.StructuralDirective */:
                    return {
                        start: {
                            line: loc.start.line,
                            column: loc.start.column - 1,
                        },
                        end: {
                            line: loc.end.line,
                            column: loc.end.column + 1,
                        },
                    };
                default:
                    return loc;
            }
        }
        return {
            ['Element$1, Template'](node) {
                var _a;
                if (isImplicitTemplate(node)) {
                    return;
                }
                const { attributes, inputs, outputs, references } = node;
                const { extractedBananaBoxes, extractedInputs, extractedOutputs } = normalizeInputsOutputs(inputs.map(toInputBindingOrderType), outputs.map(toOutputBindingOrderType));
                const allAttributes = [
                    ...extractTemplateAttrs(node),
                    ...attributes.map(toAttributeBindingOrderType),
                    ...references.map(toTemplateReferenceVariableOrderType),
                    ...extractedBananaBoxes,
                    ...extractedInputs,
                    ...extractedOutputs,
                ];
                if (allAttributes.length < 2) {
                    return;
                }
                const sortedAttributes = [...allAttributes].sort(byLocation);
                const expectedAttributes = [...allAttributes].sort(byOrder(order, alphabetical));
                let errorRange;
                for (let i = 0; i < sortedAttributes.length; i++) {
                    if (sortedAttributes[i] !== expectedAttributes[i]) {
                        errorRange = [(_a = errorRange === null || errorRange === void 0 ? void 0 : errorRange[0]) !== null && _a !== void 0 ? _a : i, i];
                    }
                }
                if (errorRange) {
                    const [startIndex, endIndex] = errorRange;
                    const sourceCode = context.getSourceCode();
                    const { start } = getLocation(sortedAttributes[startIndex]);
                    const { end } = getLocation(sortedAttributes[endIndex]);
                    const loc = { start, end };
                    const range = [
                        getStartPos(sortedAttributes[startIndex]),
                        getEndPos(sortedAttributes[endIndex]),
                    ];
                    let replacementText = '';
                    let lastPos = range[0];
                    for (let i = startIndex; i <= endIndex; i++) {
                        const oldAttr = sortedAttributes[i];
                        const oldStart = getStartPos(oldAttr);
                        const oldEnd = getEndPos(oldAttr);
                        const newAttr = expectedAttributes[i];
                        const newStart = getStartPos(newAttr);
                        const newEnd = getEndPos(newAttr);
                        replacementText += sourceCode.text.slice(lastPos, oldStart);
                        replacementText += sourceCode.text.slice(newStart, newEnd);
                        lastPos = oldEnd;
                    }
                    context.report({
                        loc,
                        messageId: 'attributesOrder',
                        data: {
                            expected: expectedAttributes
                                .slice(startIndex, endIndex + 1)
                                .map((a) => `\`${getMessageName(a)}\``)
                                .join(', '),
                            actual: sortedAttributes
                                .slice(startIndex, endIndex + 1)
                                .map((a) => `\`${getMessageName(a)}\``)
                                .join(', '),
                        },
                        fix: (fixer) => fixer.replaceTextRange(range, replacementText),
                    });
                }
            },
        };
    },
});
function byLocation(one, other) {
    return one.sourceSpan.start.line === other.sourceSpan.start.line
        ? one.sourceSpan.start.col - other.sourceSpan.start.col
        : one.sourceSpan.start.line - other.sourceSpan.start.line;
}
function byOrder(order, alphabetical) {
    return function (one, other) {
        var _a, _b, _c, _d;
        const orderComparison = getOrderIndex(one, order) - getOrderIndex(other, order);
        if (alphabetical && orderComparison === 0) {
            return ((_b = (_a = one.keySpan) === null || _a === void 0 ? void 0 : _a.details) !== null && _b !== void 0 ? _b : one.name) >
                ((_d = (_c = other.keySpan) === null || _c === void 0 ? void 0 : _c.details) !== null && _d !== void 0 ? _d : other.name)
                ? 1
                : -1;
        }
        return orderComparison;
    };
}
function getOrderIndex(attr, order) {
    return order.indexOf(attr.orderType);
}
function toAttributeBindingOrderType(attribute) {
    return Object.assign(Object.assign({}, attribute), { orderType: "ATTRIBUTE_BINDING" /* OrderType.AttributeBinding */ });
}
function toInputBindingOrderType(input) {
    return Object.assign(Object.assign({}, input), { orderType: "INPUT_BINDING" /* OrderType.InputBinding */ });
}
function toStructuralDirectiveOrderType(attributeOrInput) {
    return Object.assign(Object.assign({}, attributeOrInput), { orderType: "STRUCTURAL_DIRECTIVE" /* OrderType.StructuralDirective */ });
}
function toOutputBindingOrderType(output) {
    return Object.assign(Object.assign({}, output), { orderType: "OUTPUT_BINDING" /* OrderType.OutputBinding */ });
}
function toTwoWayBindingOrderType(output) {
    return Object.assign(Object.assign({}, output), { orderType: "TWO_WAY_BINDING" /* OrderType.TwoWayBinding */ });
}
function toTemplateReferenceVariableOrderType(reference) {
    return Object.assign(Object.assign({}, reference), { orderType: "TEMPLATE_REFERENCE" /* OrderType.TemplateReferenceVariable */ });
}
function isImplicitTemplate(node) {
    return isTmplAstTemplate(node) && node.tagName !== 'ng-template';
}
function extractTemplateAttrs(node) {
    if (isTmplAstTemplate(node)) {
        return node.templateAttrs.map(toStructuralDirectiveOrderType).concat(node.variables.map((x) => {
            return Object.assign(Object.assign({}, toAttributeBindingOrderType(x)), { 
                // `let-` is excluded from the keySpan and name - add it back in
                keySpan: new bundled_angular_compiler_1.ParseSourceSpan(x.keySpan.start.moveBy(-4), x.keySpan.end), name: 'let-' + x.name });
        }));
    }
    if (!isImplicitTemplate(node.parent)) {
        return [];
    }
    /*
     * There may be multiple "attributes" for a structural directive even though
     * there is only a single HTML attribute:
     * e.g. `<ng-container *ngFor="let foo of bar"></ng-container>`
     * will parsed as two attributes (`ngFor` and `ngForOf`)
     */
    const attrs = node.parent.templateAttrs.map(toStructuralDirectiveOrderType);
    // Pick up on any subsequent `let` bindings, e.g. `index as i`
    let sourceEnd = attrs[attrs.length - 1].sourceSpan.end;
    node.parent.variables.forEach((v) => {
        if (v.sourceSpan.start.offset <= sourceEnd.offset &&
            sourceEnd.offset < v.sourceSpan.end.offset) {
            sourceEnd = v.sourceSpan.end;
        }
    });
    return [
        Object.assign(Object.assign({}, attrs[0]), { sourceSpan: new bundled_angular_compiler_1.ParseSourceSpan(attrs[0].sourceSpan.start, sourceEnd) }),
    ];
}
function normalizeInputsOutputs(inputs, outputs) {
    const extractedInputs = inputs
        .filter((input) => !outputs.some((output) => isOnSameLocation(input, output)))
        .map(toInputBindingOrderType);
    const { extractedBananaBoxes, extractedOutputs } = outputs.reduce(({ extractedBananaBoxes, extractedOutputs }, output) => {
        const boundInput = inputs.find((input) => isOnSameLocation(input, output));
        return {
            extractedBananaBoxes: extractedBananaBoxes.concat(boundInput ? toTwoWayBindingOrderType(boundInput) : []),
            extractedOutputs: extractedOutputs.concat(boundInput ? [] : toOutputBindingOrderType(output)),
        };
    }, { extractedBananaBoxes: [], extractedOutputs: [] });
    return { extractedBananaBoxes, extractedInputs, extractedOutputs };
}
function isTmplAstTemplate(node) {
    return node instanceof bundled_angular_compiler_1.TmplAstTemplate;
}
function isOnSameLocation(input, output) {
    return (input.sourceSpan.start === output.sourceSpan.start &&
        input.sourceSpan.end === output.sourceSpan.end);
}
function getMessageName(expected) {
    var _a, _b;
    const fullName = (_b = (_a = expected.keySpan) === null || _a === void 0 ? void 0 : _a.details) !== null && _b !== void 0 ? _b : expected.name;
    switch (expected.orderType) {
        case "STRUCTURAL_DIRECTIVE" /* OrderType.StructuralDirective */:
            return `*${fullName}`;
        case "TEMPLATE_REFERENCE" /* OrderType.TemplateReferenceVariable */:
            return `#${fullName}`;
        case "INPUT_BINDING" /* OrderType.InputBinding */:
            return `[${fullName}]`;
        case "OUTPUT_BINDING" /* OrderType.OutputBinding */:
            return `(${fullName})`;
        case "TWO_WAY_BINDING" /* OrderType.TwoWayBinding */:
            return `[(${fullName})]`;
        default:
            return fullName;
    }
}
function getStartPos(expected) {
    switch (expected.orderType) {
        case "STRUCTURAL_DIRECTIVE" /* OrderType.StructuralDirective */:
            return expected.sourceSpan.start.offset - 1;
        default:
            return expected.sourceSpan.start.offset;
    }
}
function getEndPos(expected) {
    switch (expected.orderType) {
        case "STRUCTURAL_DIRECTIVE" /* OrderType.StructuralDirective */:
            return expected.sourceSpan.end.offset + 1;
        default:
            return expected.sourceSpan.end.offset;
    }
}
