"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.INVALID_TYPE_DATA_KEY = exports.RULE_NAME = void 0;
const bundled_angular_compiler_1 = require("@angular-eslint/bundled-angular-compiler");
const utils_1 = require("@angular-eslint/utils");
const create_eslint_rule_1 = require("../utils/create-eslint-rule");
exports.RULE_NAME = 'button-has-type';
exports.INVALID_TYPE_DATA_KEY = 'type';
// https://www.w3.org/TR/html401/interact/forms.html#adef-type-BUTTON
const VALID_BUTTON_TYPES = ['button', 'submit', 'reset'];
const TYPE_ATTRIBUTE_NAME = 'type';
exports.default = (0, create_eslint_rule_1.createESLintRule)({
    name: exports.RULE_NAME,
    meta: {
        type: 'suggestion',
        docs: {
            description: 'Ensures that a button has a valid type specified',
            recommended: false,
        },
        schema: [],
        messages: {
            missingType: 'Type for <button> is missing',
            invalidType: `"{{${exports.INVALID_TYPE_DATA_KEY}}}" can not be used as a type for <button>`,
        },
    },
    defaultOptions: [],
    create(context) {
        const parserServices = (0, utils_1.getTemplateParserServices)(context);
        return {
            [`Element$1[name=button]`](element) {
                if (!isTypeAttributePresentInElement(element)) {
                    context.report({
                        loc: parserServices.convertNodeSourceSpanToLoc(element.sourceSpan),
                        messageId: 'missingType',
                    });
                }
                const invalidTypeInfo = getInvalidButtonTypeIfPresent(element);
                if (invalidTypeInfo != null) {
                    context.report({
                        loc: parserServices.convertNodeSourceSpanToLoc(invalidTypeInfo.sourceSpan),
                        messageId: 'invalidType',
                        data: {
                            [exports.INVALID_TYPE_DATA_KEY]: invalidTypeInfo.value,
                        },
                    });
                }
            },
        };
    },
});
function isTypeAttributePresentInElement({ inputs, attributes, }) {
    return [...inputs, ...attributes].some(({ name }) => name === TYPE_ATTRIBUTE_NAME);
}
function getInvalidButtonTypeIfPresent(element) {
    const invalidTextAttribute = element.attributes.find(({ name, value }) => name === TYPE_ATTRIBUTE_NAME && !VALID_BUTTON_TYPES.includes(value));
    if (invalidTextAttribute) {
        return {
            sourceSpan: invalidTextAttribute.sourceSpan,
            value: invalidTextAttribute.value,
        };
    }
    for (const { name, value, sourceSpan } of element.inputs) {
        // Intentionally ignore the property binding by looking for literal primitives only
        // in order to avoid type-checking for the property, e.g. check that it's 'submit', 'button', etc.
        if (name === TYPE_ATTRIBUTE_NAME &&
            value instanceof bundled_angular_compiler_1.ASTWithSource &&
            value.ast instanceof bundled_angular_compiler_1.LiteralPrimitive &&
            !VALID_BUTTON_TYPES.includes(value.ast.value)) {
            return {
                value: value.ast.value,
                sourceSpan,
            };
        }
    }
    return null;
}
