"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RULE_NAME = exports.MESSAGE_ID = void 0;
const utils_1 = require("@angular-eslint/utils");
const create_eslint_rule_1 = require("../utils/create-eslint-rule");
const get_dom_elements_1 = require("../utils/get-dom-elements");
exports.MESSAGE_ID = 'preferSelfClosingTags';
exports.RULE_NAME = 'prefer-self-closing-tags';
exports.default = (0, create_eslint_rule_1.createESLintRule)({
    name: exports.RULE_NAME,
    meta: {
        type: 'layout',
        docs: {
            description: 'Ensures that self-closing tags are used for elements with a closing tag but no content.',
            recommended: false,
        },
        fixable: 'code',
        schema: [],
        messages: {
            [exports.MESSAGE_ID]: 'Use self-closing tags for elements with a closing tag but no content.',
        },
    },
    defaultOptions: [],
    create(context) {
        const parserServices = (0, utils_1.getTemplateParserServices)(context);
        return {
            Element$1({ children, name, startSourceSpan, endSourceSpan, }) {
                // Ignore native elements.
                if ((0, get_dom_elements_1.getDomElements)().has(name)) {
                    return;
                }
                const noContent = !children.length ||
                    children.every((node) => {
                        const text = node.value;
                        // If the node has no value, or only whitespace,
                        // we can consider it empty.
                        return (typeof text === 'string' && text.replace(/\n/g, '').trim() === '');
                    });
                const noCloseTag = !endSourceSpan ||
                    (startSourceSpan.start.offset === endSourceSpan.start.offset &&
                        startSourceSpan.end.offset === endSourceSpan.end.offset);
                if (!noContent || noCloseTag) {
                    return;
                }
                // HTML tags always have more than two characters
                // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                const openingTagLastChar = startSourceSpan.toString().at(-2);
                const hasOwnWhitespace = openingTagLastChar.trim() === '';
                const closingTagPrefix = hasOwnWhitespace ? '' : ' ';
                context.report({
                    loc: parserServices.convertNodeSourceSpanToLoc(endSourceSpan),
                    messageId: exports.MESSAGE_ID,
                    fix: (fixer) => fixer.replaceTextRange([startSourceSpan.end.offset - 1, endSourceSpan.end.offset], closingTagPrefix + '/>'),
                });
            },
        };
    },
});
