"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RULE_NAME = void 0;
const utils_1 = require("@angular-eslint/utils");
const utils_2 = require("@typescript-eslint/utils");
const create_eslint_rule_1 = require("../utils/create-eslint-rule");
exports.RULE_NAME = 'no-output-rename';
const STYLE_GUIDE_LINK = 'https://angular.io/guide/styleguide#style-05-13';
exports.default = (0, create_eslint_rule_1.createESLintRule)({
    name: exports.RULE_NAME,
    meta: {
        type: 'suggestion',
        docs: {
            description: 'Ensures that output bindings are not aliased',
            recommended: 'error',
        },
        fixable: 'code',
        hasSuggestions: true,
        schema: [],
        messages: {
            noOutputRename: `Output bindings should not be aliased (${STYLE_GUIDE_LINK})`,
            suggestRemoveAliasName: 'Remove alias name',
            suggestReplaceOriginalNameWithAliasName: 'Remove alias name and use it as the original name',
        },
    },
    defaultOptions: [],
    create(context) {
        let selectors = new Set();
        return {
            [utils_1.Selectors.COMPONENT_OR_DIRECTIVE_SELECTOR_LITERAL](node) {
                selectors = new Set((0, utils_1.withoutBracketsAndWhitespaces)(utils_1.ASTUtils.getRawText(node)).split(','));
            },
            [utils_1.Selectors.OUTPUT_ALIAS](node) {
                const propertyOrMethodDefinition = utils_1.ASTUtils.getNearestNodeFrom(node, utils_1.ASTUtils.isPropertyOrMethodDefinition);
                if (!propertyOrMethodDefinition ||
                    !utils_2.ASTUtils.isIdentifier(propertyOrMethodDefinition.key)) {
                    return;
                }
                const aliasName = utils_1.ASTUtils.getRawText(node);
                const propertyName = utils_1.ASTUtils.getRawText(propertyOrMethodDefinition.key);
                if (aliasName === propertyName) {
                    context.report({
                        node,
                        messageId: 'noOutputRename',
                        fix: (fixer) => fixer.remove(node),
                    });
                }
                else if (!isAliasNameAllowed(selectors, propertyName, aliasName)) {
                    context.report({
                        node,
                        messageId: 'noOutputRename',
                        suggest: [
                            {
                                messageId: 'suggestRemoveAliasName',
                                fix: (fixer) => fixer.remove(node),
                            },
                            {
                                messageId: 'suggestReplaceOriginalNameWithAliasName',
                                fix: (fixer) => [
                                    fixer.remove(node),
                                    fixer.replaceText(propertyOrMethodDefinition.key, aliasName.includes('-') ? `'${aliasName}'` : aliasName),
                                ],
                            },
                        ],
                    });
                }
            },
            [utils_1.Selectors.OUTPUTS_METADATA_PROPERTY_LITERAL](node) {
                var _a, _b, _c, _d;
                const ancestorMaybeHostDirectiveAPI = (_d = (_c = (_b = (_a = node.parent) === null || _a === void 0 ? void 0 : _a.parent) === null || _b === void 0 ? void 0 : _b.parent) === null || _c === void 0 ? void 0 : _c.parent) === null || _d === void 0 ? void 0 : _d.parent;
                if (ancestorMaybeHostDirectiveAPI &&
                    utils_1.ASTUtils.isProperty(ancestorMaybeHostDirectiveAPI)) {
                    /**
                     * Angular v15 introduced the directive composition API: https://angular.io/guide/directive-composition-api
                     * Renaming host directive outputs using this API is not a bad practice and should not be reported
                     */
                    const hostDirectiveAPIPropertyName = 'hostDirectives';
                    if ((utils_1.ASTUtils.isLiteral(ancestorMaybeHostDirectiveAPI.key) &&
                        ancestorMaybeHostDirectiveAPI.key.value ===
                            hostDirectiveAPIPropertyName) ||
                        (utils_2.ASTUtils.isIdentifier(ancestorMaybeHostDirectiveAPI.key) &&
                            ancestorMaybeHostDirectiveAPI.key.name ===
                                hostDirectiveAPIPropertyName)) {
                        return;
                    }
                }
                const [propertyName, aliasName] = (0, utils_1.withoutBracketsAndWhitespaces)(utils_1.ASTUtils.getRawText(node)).split(':');
                if (!aliasName)
                    return;
                if (aliasName === propertyName) {
                    context.report({
                        node,
                        messageId: 'noOutputRename',
                        fix: (fixer) => fixer.replaceText(node, utils_1.ASTUtils.getReplacementText(node, propertyName)),
                    });
                }
                else if (!isAliasNameAllowed(selectors, propertyName, aliasName)) {
                    context.report({
                        node,
                        messageId: 'noOutputRename',
                        suggest: [
                            ['suggestRemoveAliasName', propertyName],
                            ['suggestReplaceOriginalNameWithAliasName', aliasName],
                        ].map(([messageId, name]) => ({
                            messageId,
                            fix: (fixer) => fixer.replaceText(node, utils_1.ASTUtils.getReplacementText(node, name)),
                        })),
                    });
                }
            },
            'ClassDeclaration:exit'() {
                selectors = new Set();
            },
        };
    },
});
function composedName(selector, propertyName) {
    return `${selector}${(0, utils_1.capitalize)(propertyName)}`;
}
function isAliasNameAllowed(selectors, propertyName, aliasName) {
    return [...selectors].some((selector) => {
        return (selector === aliasName ||
            composedName(selector, propertyName) === aliasName);
    });
}
