"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateSchematicDefaults = exports.updateSchematicCollections = exports.determineTargetProjectName = exports.determineTargetProjectName__NX = exports.sortObjectByKeys = exports.createESLintConfigForProject = exports.createESLintConfigForProject__NX = exports.createRootESLintConfig = exports.visitNotIgnoredFiles = exports.addESLintTargetToProject = exports.addESLintTargetToProject__NX = exports.getTargetsConfigFromProject = exports.updateJsonInTree = exports.readJsonInTree = void 0;
const core_1 = require("@angular-devkit/core");
const schematics_1 = require("@angular-devkit/schematics");
const devkit_1 = require("@nx/devkit");
const ignore_1 = __importDefault(require("ignore"));
const strip_json_comments_1 = __importDefault(require("strip-json-comments"));
const DEFAULT_PREFIX = 'app';
/**
 * This method is specifically for reading JSON files in a Tree
 * @param host The host tree
 * @param path The path to the JSON file
 * @returns The JSON data in the file.
 */
// eslint-disable-next-line @typescript-eslint/no-explicit-any
function readJsonInTree(host, path) {
    if (!host.exists(path)) {
        throw new Error(`Cannot find ${path}`);
    }
    const contents = (0, strip_json_comments_1.default)(host.read(path).toString('utf-8'));
    try {
        return JSON.parse(contents);
    }
    catch (e) {
        throw new Error(`Cannot parse ${path}: ${e instanceof Error ? e.message : ''}`);
    }
}
exports.readJsonInTree = readJsonInTree;
/**
 * This method is specifically for updating JSON in a Tree
 * @param path Path of JSON file in the Tree
 * @param callback Manipulation of the JSON data
 * @returns A rule which updates a JSON file file in a Tree
 */
// eslint-disable-next-line @typescript-eslint/no-explicit-any
function updateJsonInTree(path, callback) {
    return (host, context) => {
        if (!host.exists(path)) {
            host.create(path, serializeJson(callback({}, context)));
            return host;
        }
        host.overwrite(path, serializeJson(callback(readJsonInTree(host, path), context)));
        return host;
    };
}
exports.updateJsonInTree = updateJsonInTree;
// eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
function getWorkspacePath(host) {
    const possibleFiles = ['/workspace.json', '/angular.json', '/.angular.json'];
    return possibleFiles.filter((path) => host.exists(path))[0];
}
function getTargetsConfigFromProject(projectConfig) {
    if (!projectConfig) {
        return null;
    }
    if (projectConfig.architect) {
        return projectConfig.architect;
    }
    // "targets" is an undocumented but supported alias of "architect"
    if (projectConfig.targets) {
        return projectConfig.targets;
    }
    return null;
}
exports.getTargetsConfigFromProject = getTargetsConfigFromProject;
function offsetFromRoot(fullPathToSourceDir) {
    const parts = (0, core_1.normalize)(fullPathToSourceDir).split('/');
    let offset = '';
    for (let i = 0; i < parts.length; ++i) {
        offset += '../';
    }
    return offset;
}
function serializeJson(json) {
    return `${JSON.stringify(json, null, 2)}\n`;
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any
function updateWorkspaceInTree(callback) {
    return (host, context) => {
        const path = getWorkspacePath(host);
        host.overwrite(path, serializeJson(callback(readJsonInTree(host, path), context, host)));
        return host;
    };
}
function readProjectConfiguration(tree, projectName) {
    const angularJSON = (0, devkit_1.readJson)(tree, 'angular.json');
    return angularJSON.projects[projectName];
}
function updateProjectConfiguration(tree, projectName, projectConfig) {
    const angularJSON = (0, devkit_1.readJson)(tree, 'angular.json');
    angularJSON.projects[projectName] = projectConfig;
    (0, devkit_1.writeJson)(tree, 'angular.json', angularJSON);
}
function addESLintTargetToProject__NX(tree, projectName, targetName) {
    const existingProjectConfig = readProjectConfiguration(tree, projectName);
    let lintFilePatternsRoot = '';
    // Default Angular CLI project at the root of the workspace
    if (existingProjectConfig.root === '') {
        lintFilePatternsRoot = existingProjectConfig.sourceRoot || 'src';
    }
    else {
        lintFilePatternsRoot = existingProjectConfig.root;
    }
    const eslintTargetConfig = {
        builder: '@angular-eslint/builder:lint',
        options: {
            lintFilePatterns: [
                `${lintFilePatternsRoot}/**/*.ts`,
                `${lintFilePatternsRoot}/**/*.html`,
            ],
        },
    };
    existingProjectConfig.architect = existingProjectConfig.architect || {};
    existingProjectConfig.architect[targetName] = eslintTargetConfig;
    updateProjectConfiguration(tree, projectName, existingProjectConfig);
}
exports.addESLintTargetToProject__NX = addESLintTargetToProject__NX;
function addESLintTargetToProject(projectName, targetName) {
    return updateWorkspaceInTree((workspaceJson) => {
        const existingProjectConfig = workspaceJson.projects[projectName];
        let lintFilePatternsRoot = '';
        // Default Angular CLI project at the root of the workspace
        if (existingProjectConfig.root === '') {
            lintFilePatternsRoot = existingProjectConfig.sourceRoot || 'src';
        }
        else {
            lintFilePatternsRoot = existingProjectConfig.root;
        }
        const eslintTargetConfig = {
            builder: '@angular-eslint/builder:lint',
            options: {
                lintFilePatterns: [
                    `${lintFilePatternsRoot}/**/*.ts`,
                    `${lintFilePatternsRoot}/**/*.html`,
                ],
            },
        };
        existingProjectConfig.architect[targetName] = eslintTargetConfig;
        return workspaceJson;
    });
}
exports.addESLintTargetToProject = addESLintTargetToProject;
/**
 * Utility to act on all files in a tree that are not ignored by git.
 */
function visitNotIgnoredFiles(visitor, dir = (0, core_1.normalize)('')) {
    return (host, context) => {
        let ig;
        if (host.exists('.gitignore')) {
            ig = (0, ignore_1.default)();
            ig.add(host.read('.gitignore').toString());
        }
        function visit(_dir) {
            if (_dir && (ig === null || ig === void 0 ? void 0 : ig.ignores(_dir))) {
                return;
            }
            const dirEntry = host.getDir(_dir);
            dirEntry.subfiles.forEach((file) => {
                if (ig === null || ig === void 0 ? void 0 : ig.ignores((0, core_1.join)(_dir, file))) {
                    return;
                }
                const maybeRule = visitor((0, core_1.join)(_dir, file), host, context);
                if (maybeRule) {
                    (0, schematics_1.callRule)(maybeRule, host, context).subscribe();
                }
            });
            dirEntry.subdirs.forEach((subdir) => {
                visit((0, core_1.join)(_dir, subdir));
            });
        }
        visit(dir);
    };
}
exports.visitNotIgnoredFiles = visitNotIgnoredFiles;
// eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
function setESLintProjectBasedOnProjectType(projectRoot, projectType, hasE2e) {
    let project;
    if (projectType === 'application') {
        project = [`${projectRoot}/tsconfig.(app|spec).json`];
        if (hasE2e) {
            project.push(`${projectRoot}/e2e/tsconfig.json`);
        }
    }
    // Libraries don't have an e2e directory
    if (projectType === 'library') {
        project = [`${projectRoot}/tsconfig.(lib|spec).json`];
    }
    return project;
}
// eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
function createRootESLintConfig(prefix) {
    let codeRules;
    if (prefix) {
        codeRules = {
            '@angular-eslint/directive-selector': [
                'error',
                { type: 'attribute', prefix, style: 'camelCase' },
            ],
            '@angular-eslint/component-selector': [
                'error',
                { type: 'element', prefix, style: 'kebab-case' },
            ],
        };
    }
    else {
        codeRules = {};
    }
    return {
        root: true,
        ignorePatterns: ['projects/**/*'],
        overrides: [
            {
                files: ['*.ts'],
                extends: [
                    'eslint:recommended',
                    'plugin:@typescript-eslint/recommended',
                    'plugin:@angular-eslint/recommended',
                    'plugin:@angular-eslint/template/process-inline-templates',
                ],
                rules: codeRules,
            },
            {
                files: ['*.html'],
                extends: [
                    'plugin:@angular-eslint/template/recommended',
                    'plugin:@angular-eslint/template/accessibility',
                ],
                rules: {},
            },
        ],
    };
}
exports.createRootESLintConfig = createRootESLintConfig;
function createProjectESLintConfig(projectRoot, projectType, prefix, setParserOptionsProject, hasE2e) {
    return {
        extends: `${offsetFromRoot(projectRoot)}.eslintrc.json`,
        ignorePatterns: ['!**/*'],
        overrides: [
            Object.assign(Object.assign({ files: ['*.ts'] }, (setParserOptionsProject
                ? {
                    parserOptions: {
                        project: setESLintProjectBasedOnProjectType(projectRoot, projectType, hasE2e),
                    },
                }
                : null)), { rules: {
                    '@angular-eslint/directive-selector': [
                        'error',
                        { type: 'attribute', prefix, style: 'camelCase' },
                    ],
                    '@angular-eslint/component-selector': [
                        'error',
                        { type: 'element', prefix, style: 'kebab-case' },
                    ],
                } }),
            {
                files: ['*.html'],
                rules: {},
            },
        ],
    };
}
function createProjectESLintConfig__NX(projectRoot, projectType, prefix, setParserOptionsProject, hasE2e) {
    return {
        extends: `${(0, devkit_1.offsetFromRoot)(projectRoot)}.eslintrc.json`,
        ignorePatterns: ['!**/*'],
        overrides: [
            Object.assign(Object.assign({ files: ['*.ts'] }, (setParserOptionsProject
                ? {
                    parserOptions: {
                        project: setESLintProjectBasedOnProjectType(projectRoot, projectType, hasE2e),
                    },
                }
                : null)), { rules: {
                    '@angular-eslint/directive-selector': [
                        'error',
                        { type: 'attribute', prefix, style: 'camelCase' },
                    ],
                    '@angular-eslint/component-selector': [
                        'error',
                        { type: 'element', prefix, style: 'kebab-case' },
                    ],
                } }),
            {
                files: ['*.html'],
                rules: {},
            },
        ],
    };
}
function createESLintConfigForProject__NX(tree, projectName, setParserOptionsProject) {
    const existingProjectConfig = readProjectConfiguration(tree, projectName);
    const targets = existingProjectConfig.architect || existingProjectConfig.targets;
    const { root: projectRoot, projectType, prefix } = existingProjectConfig;
    const hasE2e = !!(targets === null || targets === void 0 ? void 0 : targets.e2e);
    /**
     * If the root is an empty string it must be the initial project created at the
     * root by the Angular CLI's workspace schematic
     */
    if (projectRoot === '') {
        return createRootESLintConfigFile__NX(tree, prefix || DEFAULT_PREFIX);
    }
    // If, for whatever reason, the root .eslintrc.json doesn't exist yet, create it
    if (!tree.exists('.eslintrc.json')) {
        createRootESLintConfigFile__NX(tree, prefix || DEFAULT_PREFIX);
    }
    (0, devkit_1.writeJson)(tree, (0, core_1.join)((0, core_1.normalize)(projectRoot), '.eslintrc.json'), createProjectESLintConfig__NX(projectRoot, projectType || 'library', prefix || DEFAULT_PREFIX, setParserOptionsProject, hasE2e));
}
exports.createESLintConfigForProject__NX = createESLintConfigForProject__NX;
function createESLintConfigForProject(projectName, setParserOptionsProject) {
    return (tree) => {
        const angularJSON = readJsonInTree(tree, 'angular.json');
        const { root: projectRoot, projectType, prefix, } = angularJSON.projects[projectName];
        const hasE2e = determineTargetProjectHasE2E(angularJSON, projectName);
        /**
         * If the root is an empty string it must be the initial project created at the
         * root by the Angular CLI's workspace schematic
         */
        if (projectRoot === '') {
            return createRootESLintConfigFile(projectName);
        }
        return (0, schematics_1.chain)([
            // If, for whatever reason, the root .eslintrc.json doesn't exist yet, create it
            tree.exists('.eslintrc.json')
                ? () => undefined
                : createRootESLintConfigFile(projectName),
            updateJsonInTree((0, core_1.join)((0, core_1.normalize)(projectRoot), '.eslintrc.json'), () => createProjectESLintConfig(projectRoot, projectType, prefix, setParserOptionsProject, hasE2e)),
        ]);
    };
}
exports.createESLintConfigForProject = createESLintConfigForProject;
function createRootESLintConfigFile(projectName) {
    return (tree) => {
        var _a;
        const angularJSON = readJsonInTree(tree, getWorkspacePath(tree));
        let lintPrefix = null;
        if ((_a = angularJSON.projects) === null || _a === void 0 ? void 0 : _a[projectName]) {
            const { prefix } = angularJSON.projects[projectName];
            lintPrefix = prefix;
        }
        return updateJsonInTree('.eslintrc.json', () => createRootESLintConfig(lintPrefix));
    };
}
function createRootESLintConfigFile__NX(tree, prefix) {
    return (0, devkit_1.writeJson)(tree, '.eslintrc.json', createRootESLintConfig(prefix));
}
function sortObjectByKeys(obj) {
    return Object.keys(obj)
        .sort()
        .reduce((result, key) => {
        return Object.assign(Object.assign({}, result), { [key]: obj[key] });
    }, {});
}
exports.sortObjectByKeys = sortObjectByKeys;
/**
 * To make certain schematic usage conversion more ergonomic, if the user does not specify a project
 * and only has a single project in their angular.json we will just go ahead and use that one.
 */
function determineTargetProjectName__NX(tree, maybeProject) {
    if (maybeProject) {
        return maybeProject;
    }
    const workspaceJson = (0, devkit_1.readJson)(tree, 'angular.json');
    const projects = Object.keys(workspaceJson.projects);
    if (projects.length === 1) {
        return projects[0];
    }
    return null;
}
exports.determineTargetProjectName__NX = determineTargetProjectName__NX;
/**
 * To make certain schematic usage conversion more ergonomic, if the user does not specify a project
 * and only has a single project in their angular.json we will just go ahead and use that one.
 */
function determineTargetProjectName(tree, maybeProject) {
    if (maybeProject) {
        return maybeProject;
    }
    const workspaceJson = readJsonInTree(tree, getWorkspacePath(tree));
    const projects = Object.keys(workspaceJson.projects);
    if (projects.length === 1) {
        return projects[0];
    }
    return null;
}
exports.determineTargetProjectName = determineTargetProjectName;
/**
 * Checking if the target project has e2e setup
 * Method will check if angular project architect has e2e configuration to determine if e2e setup
 */
function determineTargetProjectHasE2E(
// eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types, @typescript-eslint/no-explicit-any
angularJSON, projectName) {
    var _a;
    return !!((_a = getTargetsConfigFromProject(angularJSON.projects[projectName])) === null || _a === void 0 ? void 0 : _a.e2e);
}
/**
 * See `schematicCollections` docs here:
 * https://github.com/angular/angular-cli/blob/8431b3f0769b5f95b9e13807a09293d820c4b017/docs/specifications/schematic-collections-config.md
 */
function updateSchematicCollections(angularJson) {
    angularJson.cli = angularJson.cli || {};
    angularJson.cli.schematicCollections =
        angularJson.cli.schematicCollections || [];
    // The first matching schematic will be used, so we unshift rather than push
    angularJson.cli.schematicCollections.unshift('@angular-eslint/schematics');
    // Delete old defaultCollection property if applicable
    delete angularJson.cli.defaultCollection;
    return angularJson;
}
exports.updateSchematicCollections = updateSchematicCollections;
function updateSchematicDefaults(angularJson, schematicFullName, defaultValues) {
    angularJson.schematics = angularJson.schematics || {};
    angularJson.schematics[schematicFullName] =
        angularJson.schematics[schematicFullName] || {};
    angularJson.schematics[schematicFullName] = Object.assign(Object.assign({}, angularJson.schematics[schematicFullName]), defaultValues);
    return angularJson;
}
exports.updateSchematicDefaults = updateSchematicDefaults;
