import { getOrSetDefaultValue } from '../render/shared';
import { copyObj, interpolateParams, iteratorToArray } from '../util';
import { buildAnimationTimelines } from './animation_timeline_builder';
import { createTransitionInstruction } from './animation_transition_instruction';
const EMPTY_OBJECT = {};
export class AnimationTransitionFactory {
    constructor(_triggerName, ast, _stateStyles) {
        this._triggerName = _triggerName;
        this.ast = ast;
        this._stateStyles = _stateStyles;
    }
    match(currentState, nextState, element, params) {
        return oneOrMoreTransitionsMatch(this.ast.matchers, currentState, nextState, element, params);
    }
    buildStyles(stateName, params, errors) {
        let styler = this._stateStyles.get('*');
        if (stateName !== undefined) {
            styler = this._stateStyles.get(stateName?.toString()) || styler;
        }
        return styler ? styler.buildStyles(params, errors) : new Map();
    }
    build(driver, element, currentState, nextState, enterClassName, leaveClassName, currentOptions, nextOptions, subInstructions, skipAstBuild) {
        const errors = [];
        const transitionAnimationParams = this.ast.options && this.ast.options.params || EMPTY_OBJECT;
        const currentAnimationParams = currentOptions && currentOptions.params || EMPTY_OBJECT;
        const currentStateStyles = this.buildStyles(currentState, currentAnimationParams, errors);
        const nextAnimationParams = nextOptions && nextOptions.params || EMPTY_OBJECT;
        const nextStateStyles = this.buildStyles(nextState, nextAnimationParams, errors);
        const queriedElements = new Set();
        const preStyleMap = new Map();
        const postStyleMap = new Map();
        const isRemoval = nextState === 'void';
        const animationOptions = {
            params: applyParamDefaults(nextAnimationParams, transitionAnimationParams),
            delay: this.ast.options?.delay,
        };
        const timelines = skipAstBuild ?
            [] :
            buildAnimationTimelines(driver, element, this.ast.animation, enterClassName, leaveClassName, currentStateStyles, nextStateStyles, animationOptions, subInstructions, errors);
        let totalTime = 0;
        timelines.forEach(tl => {
            totalTime = Math.max(tl.duration + tl.delay, totalTime);
        });
        if (errors.length) {
            return createTransitionInstruction(element, this._triggerName, currentState, nextState, isRemoval, currentStateStyles, nextStateStyles, [], [], preStyleMap, postStyleMap, totalTime, errors);
        }
        timelines.forEach(tl => {
            const elm = tl.element;
            const preProps = getOrSetDefaultValue(preStyleMap, elm, new Set());
            tl.preStyleProps.forEach(prop => preProps.add(prop));
            const postProps = getOrSetDefaultValue(postStyleMap, elm, new Set());
            tl.postStyleProps.forEach(prop => postProps.add(prop));
            if (elm !== element) {
                queriedElements.add(elm);
            }
        });
        if (typeof ngDevMode === 'undefined' || ngDevMode) {
            checkNonAnimatableInTimelines(timelines, this._triggerName, driver);
        }
        const queriedElementsList = iteratorToArray(queriedElements.values());
        return createTransitionInstruction(element, this._triggerName, currentState, nextState, isRemoval, currentStateStyles, nextStateStyles, timelines, queriedElementsList, preStyleMap, postStyleMap, totalTime);
    }
}
/**
 * Checks inside a set of timelines if they try to animate a css property which is not considered
 * animatable, in that case it prints a warning on the console.
 * Besides that the function doesn't have any other effect.
 *
 * Note: this check is done here after the timelines are built instead of doing on a lower level so
 * that we can make sure that the warning appears only once per instruction (we can aggregate here
 * all the issues instead of finding them separately).
 *
 * @param timelines The built timelines for the current instruction.
 * @param triggerName The name of the trigger for the current instruction.
 * @param driver Animation driver used to perform the check.
 *
 */
function checkNonAnimatableInTimelines(timelines, triggerName, driver) {
    if (!driver.validateAnimatableStyleProperty) {
        return;
    }
    const allowedNonAnimatableProps = new Set([
        // 'easing' is a utility/synthetic prop we use to represent
        // easing functions, it represents a property of the animation
        // which is not animatable but different values can be used
        // in different steps
        'easing'
    ]);
    const invalidNonAnimatableProps = new Set();
    timelines.forEach(({ keyframes }) => {
        const nonAnimatablePropsInitialValues = new Map();
        keyframes.forEach(keyframe => {
            const entriesToCheck = Array.from(keyframe.entries()).filter(([prop]) => !allowedNonAnimatableProps.has(prop));
            for (const [prop, value] of entriesToCheck) {
                if (!driver.validateAnimatableStyleProperty(prop)) {
                    if (nonAnimatablePropsInitialValues.has(prop) && !invalidNonAnimatableProps.has(prop)) {
                        const propInitialValue = nonAnimatablePropsInitialValues.get(prop);
                        if (propInitialValue !== value) {
                            invalidNonAnimatableProps.add(prop);
                        }
                    }
                    else {
                        nonAnimatablePropsInitialValues.set(prop, value);
                    }
                }
            }
        });
    });
    if (invalidNonAnimatableProps.size > 0) {
        console.warn(`Warning: The animation trigger "${triggerName}" is attempting to animate the following` +
            ' not animatable properties: ' + Array.from(invalidNonAnimatableProps).join(', ') + '\n' +
            '(to check the list of all animatable properties visit https://developer.mozilla.org/en-US/docs/Web/CSS/CSS_animated_properties)');
    }
}
function oneOrMoreTransitionsMatch(matchFns, currentState, nextState, element, params) {
    return matchFns.some(fn => fn(currentState, nextState, element, params));
}
function applyParamDefaults(userParams, defaults) {
    const result = copyObj(defaults);
    for (const key in userParams) {
        if (userParams.hasOwnProperty(key) && userParams[key] != null) {
            result[key] = userParams[key];
        }
    }
    return result;
}
export class AnimationStateStyles {
    constructor(styles, defaultParams, normalizer) {
        this.styles = styles;
        this.defaultParams = defaultParams;
        this.normalizer = normalizer;
    }
    buildStyles(params, errors) {
        const finalStyles = new Map();
        const combinedParams = copyObj(this.defaultParams);
        Object.keys(params).forEach(key => {
            const value = params[key];
            if (value !== null) {
                combinedParams[key] = value;
            }
        });
        this.styles.styles.forEach(value => {
            if (typeof value !== 'string') {
                value.forEach((val, prop) => {
                    if (val) {
                        val = interpolateParams(val, combinedParams, errors);
                    }
                    const normalizedProp = this.normalizer.normalizePropertyName(prop, errors);
                    val = this.normalizer.normalizeStyleValue(prop, normalizedProp, val, errors);
                    finalStyles.set(prop, val);
                });
            }
        });
        return finalStyles;
    }
}
//# sourceMappingURL=data:application/json;base64,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