/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ElementAttributes } from '../element';
import { OpKind } from '../enums';
import { OpList } from '../operations';
import { TRAIT_CONSUMES_SLOT, TRAIT_USES_SLOT_INDEX } from '../traits';
import { NEW_OP } from './shared';
/**
 * The set of OpKinds that represent the creation of an element or container
 */
const elementContainerOpKinds = new Set([
    OpKind.Element, OpKind.ElementStart, OpKind.Container, OpKind.ContainerStart, OpKind.Template
]);
/**
 * Checks whether the given operation represents the creation of an element or container.
 */
export function isElementOrContainerOp(op) {
    return elementContainerOpKinds.has(op.kind);
}
/**
 * Create an `ElementStartOp`.
 */
export function createElementStartOp(tag, xref, namespace, sourceSpan) {
    return {
        kind: OpKind.ElementStart,
        xref,
        tag,
        attributes: new ElementAttributes(),
        localRefs: [],
        nonBindable: false,
        namespace,
        sourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
/**
 * Create a `TemplateOp`.
 */
export function createTemplateOp(xref, tag, namespace, sourceSpan) {
    return {
        kind: OpKind.Template,
        xref,
        attributes: new ElementAttributes(),
        tag,
        decls: null,
        vars: null,
        localRefs: [],
        nonBindable: false,
        namespace,
        sourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
/**
 * Create an `ElementEndOp`.
 */
export function createElementEndOp(xref, sourceSpan) {
    return {
        kind: OpKind.ElementEnd,
        xref,
        sourceSpan,
        ...NEW_OP,
    };
}
export function createDisableBindingsOp(xref) {
    return {
        kind: OpKind.DisableBindings,
        xref,
        ...NEW_OP,
    };
}
export function createEnableBindingsOp(xref) {
    return {
        kind: OpKind.EnableBindings,
        xref,
        ...NEW_OP,
    };
}
/**
 * Create a `TextOp`.
 */
export function createTextOp(xref, initialValue, sourceSpan) {
    return {
        kind: OpKind.Text,
        xref,
        initialValue,
        sourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
/**
 * Create a `ListenerOp`.
 */
export function createListenerOp(target, name, tag) {
    return {
        kind: OpKind.Listener,
        target,
        tag,
        name,
        handlerOps: new OpList(),
        handlerFnName: null,
        consumesDollarEvent: false,
        isAnimationListener: false,
        animationPhase: null,
        ...NEW_OP,
        ...TRAIT_USES_SLOT_INDEX,
    };
}
/**
 * Create a `ListenerOp` for an animation.
 */
export function createListenerOpForAnimation(target, name, animationPhase, tag) {
    return {
        kind: OpKind.Listener,
        target,
        tag,
        name,
        handlerOps: new OpList(),
        handlerFnName: null,
        consumesDollarEvent: false,
        isAnimationListener: true,
        animationPhase,
        ...NEW_OP,
        ...TRAIT_USES_SLOT_INDEX,
    };
}
export function createPipeOp(xref, name) {
    return {
        kind: OpKind.Pipe,
        xref,
        name,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
    };
}
/**
 * Whether the active namespace is HTML, MathML, or SVG mode.
 */
export var Namespace;
(function (Namespace) {
    Namespace[Namespace["HTML"] = 0] = "HTML";
    Namespace[Namespace["SVG"] = 1] = "SVG";
    Namespace[Namespace["Math"] = 2] = "Math";
})(Namespace || (Namespace = {}));
export function createNamespaceOp(namespace) {
    return {
        kind: OpKind.Namespace,
        active: namespace,
        ...NEW_OP,
    };
}
//# sourceMappingURL=data:application/json;base64,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