/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { OpKind } from '../enums';
import { TRAIT_CONSUMES_VARS, TRAIT_DEPENDS_ON_SLOT_CONTEXT } from '../traits';
import { NEW_OP } from './shared';
/**
 * Create an `InterpolationTextOp`.
 */
export function createInterpolateTextOp(xref, interpolation, sourceSpan) {
    return {
        kind: OpKind.InterpolateText,
        target: xref,
        interpolation,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
export class Interpolation {
    constructor(strings, expressions) {
        this.strings = strings;
        this.expressions = expressions;
    }
}
/**
 * Create a `BindingOp`, not yet transformed into a particular type of binding.
 */
export function createBindingOp(target, kind, name, expression, unit, securityContext, isTemplate, sourceSpan) {
    return {
        kind: OpKind.Binding,
        bindingKind: kind,
        target,
        name,
        expression,
        unit,
        securityContext,
        isTemplate,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Create a `PropertyOp`.
 */
export function createPropertyOp(target, name, expression, isAnimationTrigger, securityContext, isTemplate, sourceSpan) {
    return {
        kind: OpKind.Property,
        target,
        name,
        expression,
        isAnimationTrigger,
        securityContext,
        sanitizer: null,
        isTemplate,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/** Create a `StylePropOp`. */
export function createStylePropOp(xref, name, expression, unit, sourceSpan) {
    return {
        kind: OpKind.StyleProp,
        target: xref,
        name,
        expression,
        unit,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create a `ClassPropOp`.
 */
export function createClassPropOp(xref, name, expression, sourceSpan) {
    return {
        kind: OpKind.ClassProp,
        target: xref,
        name,
        expression,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/** Create a `StyleMapOp`. */
export function createStyleMapOp(xref, expression, sourceSpan) {
    return {
        kind: OpKind.StyleMap,
        target: xref,
        expression,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create a `ClassMapOp`.
 */
export function createClassMapOp(xref, expression, sourceSpan) {
    return {
        kind: OpKind.ClassMap,
        target: xref,
        expression,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create an `AttributeOp`.
 */
export function createAttributeOp(target, name, expression, securityContext, isTemplate, sourceSpan) {
    return {
        kind: OpKind.Attribute,
        target,
        name,
        expression,
        securityContext,
        sanitizer: null,
        isTemplate,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create an `AdvanceOp`.
 */
export function createAdvanceOp(delta, sourceSpan) {
    return {
        kind: OpKind.Advance,
        delta,
        sourceSpan,
        ...NEW_OP,
    };
}
//# sourceMappingURL=data:application/json;base64,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