/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../../../../output/output_ast';
import * as ir from '../../ir';
/**
 * Finds all unresolved safe read expressions, and converts them into the appropriate output AST
 * reads, guarded by null checks.
 */
export function phaseExpandSafeReads(job) {
    for (const unit of job.units) {
        for (const op of unit.ops()) {
            ir.transformExpressionsInOp(op, e => safeTransform(e, { job }), ir.VisitorContextFlag.None);
            ir.transformExpressionsInOp(op, ternaryTransform, ir.VisitorContextFlag.None);
        }
    }
}
// A lookup set of all the expression kinds that require a temporary variable to be generated.
const requiresTemporary = [
    o.InvokeFunctionExpr, o.LiteralArrayExpr, o.LiteralMapExpr, ir.SafeInvokeFunctionExpr,
    ir.PipeBindingExpr
].map(e => e.constructor.name);
function needsTemporaryInSafeAccess(e) {
    // TODO: We probably want to use an expression visitor to recursively visit all descendents.
    // However, that would potentially do a lot of extra work (because it cannot short circuit), so we
    // implement the logic ourselves for now.
    if (e instanceof o.UnaryOperatorExpr) {
        return needsTemporaryInSafeAccess(e.expr);
    }
    else if (e instanceof o.BinaryOperatorExpr) {
        return needsTemporaryInSafeAccess(e.lhs) || needsTemporaryInSafeAccess(e.rhs);
    }
    else if (e instanceof o.ConditionalExpr) {
        if (e.falseCase && needsTemporaryInSafeAccess(e.falseCase))
            return true;
        return needsTemporaryInSafeAccess(e.condition) || needsTemporaryInSafeAccess(e.trueCase);
    }
    else if (e instanceof o.NotExpr) {
        return needsTemporaryInSafeAccess(e.condition);
    }
    else if (e instanceof ir.AssignTemporaryExpr) {
        return needsTemporaryInSafeAccess(e.expr);
    }
    else if (e instanceof o.ReadPropExpr) {
        return needsTemporaryInSafeAccess(e.receiver);
    }
    else if (e instanceof o.ReadKeyExpr) {
        return needsTemporaryInSafeAccess(e.receiver) || needsTemporaryInSafeAccess(e.index);
    }
    // TODO: Switch to a method which is exhaustive of newly added expression subtypes.
    return e instanceof o.InvokeFunctionExpr || e instanceof o.LiteralArrayExpr ||
        e instanceof o.LiteralMapExpr || e instanceof ir.SafeInvokeFunctionExpr ||
        e instanceof ir.PipeBindingExpr;
}
function temporariesIn(e) {
    const temporaries = new Set();
    // TODO: Although it's not currently supported by the transform helper, we should be able to
    // short-circuit exploring the tree to do less work. In particular, we don't have to penetrate
    // into the subexpressions of temporary assignments.
    ir.transformExpressionsInExpression(e, e => {
        if (e instanceof ir.AssignTemporaryExpr) {
            temporaries.add(e.xref);
        }
        return e;
    }, ir.VisitorContextFlag.None);
    return temporaries;
}
function eliminateTemporaryAssignments(e, tmps, ctx) {
    // TODO: We can be more efficient than the transform helper here. We don't need to visit any
    // descendents of temporary assignments.
    ir.transformExpressionsInExpression(e, e => {
        if (e instanceof ir.AssignTemporaryExpr && tmps.has(e.xref)) {
            const read = new ir.ReadTemporaryExpr(e.xref);
            // `TemplateDefinitionBuilder` has the (accidental?) behavior of generating assignments of
            // temporary variables to themselves. This happens because some subexpression that the
            // temporary refers to, possibly through nested temporaries, has a function call. We copy that
            // behavior here.
            return ctx.job.compatibility === ir.CompatibilityMode.TemplateDefinitionBuilder ?
                new ir.AssignTemporaryExpr(read, read.xref) :
                read;
        }
        return e;
    }, ir.VisitorContextFlag.None);
    return e;
}
/**
 * Creates a safe ternary guarded by the input expression, and with a body generated by the provided
 * callback on the input expression. Generates a temporary variable assignment if needed, and
 * deduplicates nested temporary assignments if needed.
 */
function safeTernaryWithTemporary(guard, body, ctx) {
    let result;
    if (needsTemporaryInSafeAccess(guard)) {
        const xref = ctx.job.allocateXrefId();
        result = [new ir.AssignTemporaryExpr(guard, xref), new ir.ReadTemporaryExpr(xref)];
    }
    else {
        result = [guard, guard.clone()];
        // Consider an expression like `a?.[b?.c()]?.d`. The `b?.c()` will be transformed first,
        // introducing a temporary assignment into the key. Then, as part of expanding the `?.d`. That
        // assignment will be duplicated into both the guard and expression sides. We de-duplicate it,
        // by transforming it from an assignment into a read on the expression side.
        eliminateTemporaryAssignments(result[1], temporariesIn(result[0]), ctx);
    }
    return new ir.SafeTernaryExpr(result[0], body(result[1]));
}
function isSafeAccessExpression(e) {
    return e instanceof ir.SafePropertyReadExpr || e instanceof ir.SafeKeyedReadExpr ||
        e instanceof ir.SafeInvokeFunctionExpr;
}
function isUnsafeAccessExpression(e) {
    return e instanceof o.ReadPropExpr || e instanceof o.ReadKeyExpr ||
        e instanceof o.InvokeFunctionExpr;
}
function isAccessExpression(e) {
    return isSafeAccessExpression(e) || isUnsafeAccessExpression(e);
}
function deepestSafeTernary(e) {
    if (isAccessExpression(e) && e.receiver instanceof ir.SafeTernaryExpr) {
        let st = e.receiver;
        while (st.expr instanceof ir.SafeTernaryExpr) {
            st = st.expr;
        }
        return st;
    }
    return null;
}
// TODO: When strict compatibility with TemplateDefinitionBuilder is not required, we can use `&&`
// instead to save some code size.
function safeTransform(e, ctx) {
    if (!isAccessExpression(e)) {
        return e;
    }
    const dst = deepestSafeTernary(e);
    if (dst) {
        if (e instanceof o.InvokeFunctionExpr) {
            dst.expr = dst.expr.callFn(e.args);
            return e.receiver;
        }
        if (e instanceof o.ReadPropExpr) {
            dst.expr = dst.expr.prop(e.name);
            return e.receiver;
        }
        if (e instanceof o.ReadKeyExpr) {
            dst.expr = dst.expr.key(e.index);
            return e.receiver;
        }
        if (e instanceof ir.SafeInvokeFunctionExpr) {
            dst.expr = safeTernaryWithTemporary(dst.expr, (r) => r.callFn(e.args), ctx);
            return e.receiver;
        }
        if (e instanceof ir.SafePropertyReadExpr) {
            dst.expr = safeTernaryWithTemporary(dst.expr, (r) => r.prop(e.name), ctx);
            return e.receiver;
        }
        if (e instanceof ir.SafeKeyedReadExpr) {
            dst.expr = safeTernaryWithTemporary(dst.expr, (r) => r.key(e.index), ctx);
            return e.receiver;
        }
    }
    else {
        if (e instanceof ir.SafeInvokeFunctionExpr) {
            return safeTernaryWithTemporary(e.receiver, (r) => r.callFn(e.args), ctx);
        }
        if (e instanceof ir.SafePropertyReadExpr) {
            return safeTernaryWithTemporary(e.receiver, (r) => r.prop(e.name), ctx);
        }
        if (e instanceof ir.SafeKeyedReadExpr) {
            return safeTernaryWithTemporary(e.receiver, (r) => r.key(e.index), ctx);
        }
    }
    return e;
}
function ternaryTransform(e) {
    if (!(e instanceof ir.SafeTernaryExpr)) {
        return e;
    }
    return new o.ConditionalExpr(new o.BinaryOperatorExpr(o.BinaryOperator.Equals, e.guard, o.NULL_EXPR), o.NULL_EXPR, e.expr);
}
//# sourceMappingURL=data:application/json;base64,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