/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { assertDefined } from '../util/assert';
import { assertLContainer, assertLView, assertTNodeForLView } from './assert';
import { renderView } from './instructions/render';
import { createLView } from './instructions/shared';
import { CONTAINER_HEADER_OFFSET, NATIVE } from './interfaces/container';
import { DECLARATION_LCONTAINER, FLAGS, QUERIES, RENDERER, T_HOST, TVIEW } from './interfaces/view';
import { addViewToDOM, destroyLView, detachView, getBeforeNodeForView, insertView, nativeParentNode } from './node_manipulation';
export function createAndRenderEmbeddedLView(declarationLView, templateTNode, context, options) {
    const embeddedTView = templateTNode.tView;
    ngDevMode && assertDefined(embeddedTView, 'TView must be defined for a template node.');
    ngDevMode && assertTNodeForLView(templateTNode, declarationLView);
    // Embedded views follow the change detection strategy of the view they're declared in.
    const isSignalView = declarationLView[FLAGS] & 4096 /* LViewFlags.SignalView */;
    const viewFlags = isSignalView ? 4096 /* LViewFlags.SignalView */ : 16 /* LViewFlags.CheckAlways */;
    const embeddedLView = createLView(declarationLView, embeddedTView, context, viewFlags, null, templateTNode, null, null, null, options?.injector ?? null, options?.hydrationInfo ?? null);
    const declarationLContainer = declarationLView[templateTNode.index];
    ngDevMode && assertLContainer(declarationLContainer);
    embeddedLView[DECLARATION_LCONTAINER] = declarationLContainer;
    const declarationViewLQueries = declarationLView[QUERIES];
    if (declarationViewLQueries !== null) {
        embeddedLView[QUERIES] = declarationViewLQueries.createEmbeddedView(embeddedTView);
    }
    // execute creation mode of a view
    renderView(embeddedTView, embeddedLView, context);
    return embeddedLView;
}
export function getLViewFromLContainer(lContainer, index) {
    const adjustedIndex = CONTAINER_HEADER_OFFSET + index;
    // avoid reading past the array boundaries
    if (adjustedIndex < lContainer.length) {
        const lView = lContainer[adjustedIndex];
        ngDevMode && assertLView(lView);
        return lView;
    }
    return undefined;
}
export function addLViewToLContainer(lContainer, lView, index, addToDOM = true) {
    const tView = lView[TVIEW];
    // insert to the view tree so the new view can be change-detected
    insertView(tView, lView, lContainer, index);
    // insert to the view to the DOM tree
    if (addToDOM) {
        const beforeNode = getBeforeNodeForView(index, lContainer);
        const renderer = lView[RENDERER];
        const parentRNode = nativeParentNode(renderer, lContainer[NATIVE]);
        if (parentRNode !== null) {
            addViewToDOM(tView, lContainer[T_HOST], renderer, lView, parentRNode, beforeNode);
        }
    }
}
export function removeLViewFromLContainer(lContainer, index) {
    const lView = detachView(lContainer, index);
    if (lView !== undefined) {
        destroyLView(lView[TVIEW], lView);
    }
    return lView;
}
//# sourceMappingURL=data:application/json;base64,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