/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { consumerAfterComputation, consumerBeforeComputation, consumerMarkDirty, consumerPollProducersForChange, REACTIVE_NODE } from './graph';
export function watch(fn, schedule, allowSignalWrites) {
    const node = Object.create(WATCH_NODE);
    if (allowSignalWrites) {
        node.consumerAllowSignalWrites = true;
    }
    node.fn = fn;
    node.schedule = schedule;
    const registerOnCleanup = (cleanupFn) => {
        node.cleanupFn = cleanupFn;
    };
    const run = () => {
        node.dirty = false;
        if (node.hasRun && !consumerPollProducersForChange(node)) {
            return;
        }
        node.hasRun = true;
        const prevConsumer = consumerBeforeComputation(node);
        try {
            node.cleanupFn();
            node.cleanupFn = NOOP_CLEANUP_FN;
            node.fn(registerOnCleanup);
        }
        finally {
            consumerAfterComputation(node, prevConsumer);
        }
    };
    node.ref = {
        notify: () => consumerMarkDirty(node),
        run,
        cleanup: () => node.cleanupFn(),
    };
    return node.ref;
}
const NOOP_CLEANUP_FN = () => { };
// Note: Using an IIFE here to ensure that the spread assignment is not considered
// a side-effect, ending up preserving `COMPUTED_NODE` and `REACTIVE_NODE`.
// TODO: remove when https://github.com/evanw/esbuild/issues/3392 is resolved.
const WATCH_NODE = /* @__PURE__ */ (() => {
    return {
        ...REACTIVE_NODE,
        consumerIsAlwaysLive: true,
        consumerAllowSignalWrites: false,
        consumerMarkedDirty: (node) => {
            node.schedule(node.ref);
        },
        hasRun: false,
        cleanupFn: NOOP_CLEANUP_FN,
    };
})();
//# sourceMappingURL=data:application/json;base64,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