export class MenuController {
    constructor(menuController) {
        this.menuController = menuController;
    }
    /**
     * Programmatically open the Menu.
     * @param [menuId]  Optionally get the menu by its id, or side.
     * @return returns a promise when the menu is fully opened
     */
    open(menuId) {
        return this.menuController.open(menuId);
    }
    /**
     * Programmatically close the Menu. If no `menuId` is given as the first
     * argument then it'll close any menu which is open. If a `menuId`
     * is given then it'll close that exact menu.
     * @param [menuId]  Optionally get the menu by its id, or side.
     * @return returns a promise when the menu is fully closed
     */
    close(menuId) {
        return this.menuController.close(menuId);
    }
    /**
     * Toggle the menu. If it's closed, it will open, and if opened, it
     * will close.
     * @param [menuId]  Optionally get the menu by its id, or side.
     * @return returns a promise when the menu has been toggled
     */
    toggle(menuId) {
        return this.menuController.toggle(menuId);
    }
    /**
     * Used to enable or disable a menu. For example, there could be multiple
     * left menus, but only one of them should be able to be opened at the same
     * time. If there are multiple menus on the same side, then enabling one menu
     * will also automatically disable all the others that are on the same side.
     * @param [menuId]  Optionally get the menu by its id, or side.
     * @return Returns the instance of the menu, which is useful for chaining.
     */
    enable(shouldEnable, menuId) {
        return this.menuController.enable(shouldEnable, menuId);
    }
    /**
     * Used to enable or disable the ability to swipe open the menu.
     * @param shouldEnable  True if it should be swipe-able, false if not.
     * @param [menuId]  Optionally get the menu by its id, or side.
     * @return Returns the instance of the menu, which is useful for chaining.
     */
    swipeGesture(shouldEnable, menuId) {
        return this.menuController.swipeGesture(shouldEnable, menuId);
    }
    /**
     * @param [menuId] Optionally get the menu by its id, or side.
     * @return Returns true if the specified menu is currently open, otherwise false.
     * If the menuId is not specified, it returns true if ANY menu is currenly open.
     */
    isOpen(menuId) {
        return this.menuController.isOpen(menuId);
    }
    /**
     * @param [menuId]  Optionally get the menu by its id, or side.
     * @return Returns true if the menu is currently enabled, otherwise false.
     */
    isEnabled(menuId) {
        return this.menuController.isEnabled(menuId);
    }
    /**
     * Used to get a menu instance. If a `menuId` is not provided then it'll
     * return the first menu found. If a `menuId` is `left` or `right`, then
     * it'll return the enabled menu on that side. Otherwise, if a `menuId` is
     * provided, then it'll try to find the menu using the menu's `id`
     * property. If a menu is not found then it'll return `null`.
     * @param [menuId]  Optionally get the menu by its id, or side.
     * @return Returns the instance of the menu if found, otherwise `null`.
     */
    get(menuId) {
        return this.menuController.get(menuId);
    }
    /**
     * @return Returns the instance of the menu already opened, otherwise `null`.
     */
    getOpen() {
        return this.menuController.getOpen();
    }
    /**
     * @return Returns an array of all menu instances.
     */
    getMenus() {
        return this.menuController.getMenus();
    }
    registerAnimation(name, animation) {
        return this.menuController.registerAnimation(name, animation);
    }
    isAnimating() {
        return this.menuController.isAnimating();
    }
    _getOpenSync() {
        return this.menuController._getOpenSync();
    }
    _createAnimation(type, menuCmp) {
        return this.menuController._createAnimation(type, menuCmp);
    }
    _register(menu) {
        return this.menuController._register(menu);
    }
    _unregister(menu) {
        return this.menuController._unregister(menu);
    }
    _setOpen(menu, shouldOpen, animated) {
        return this.menuController._setOpen(menu, shouldOpen, animated);
    }
}
//# sourceMappingURL=data:application/json;base64,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