import { Injectable, Optional } from '@angular/core';
import { NavigationStart } from '@angular/router';
import * as i0 from "@angular/core";
import * as i1 from "./platform";
import * as i2 from "@angular/common";
import * as i3 from "@angular/router";
export class NavController {
    constructor(platform, location, serializer, router) {
        this.location = location;
        this.serializer = serializer;
        this.router = router;
        this.direction = DEFAULT_DIRECTION;
        this.animated = DEFAULT_ANIMATED;
        this.guessDirection = 'forward';
        this.lastNavId = -1;
        // Subscribe to router events to detect direction
        if (router) {
            router.events.subscribe((ev) => {
                if (ev instanceof NavigationStart) {
                    const id = ev.restoredState ? ev.restoredState.navigationId : ev.id;
                    this.guessDirection = id < this.lastNavId ? 'back' : 'forward';
                    this.guessAnimation = !ev.restoredState ? this.guessDirection : undefined;
                    this.lastNavId = this.guessDirection === 'forward' ? ev.id : id;
                }
            });
        }
        // Subscribe to backButton events
        platform.backButton.subscribeWithPriority(0, (processNextHandler) => {
            this.pop();
            processNextHandler();
        });
    }
    /**
     * This method uses Angular's [Router](https://angular.io/api/router/Router) under the hood,
     * it's equivalent to calling `this.router.navigateByUrl()`, but it's explicit about the **direction** of the transition.
     *
     * Going **forward** means that a new page is going to be pushed to the stack of the outlet (ion-router-outlet),
     * and that it will show a "forward" animation by default.
     *
     * Navigating forward can also be triggered in a declarative manner by using the `[routerDirection]` directive:
     *
     * ```html
     * <a routerLink="/path/to/page" routerDirection="forward">Link</a>
     * ```
     */
    navigateForward(url, options = {}) {
        this.setDirection('forward', options.animated, options.animationDirection, options.animation);
        return this.navigate(url, options);
    }
    /**
     * This method uses Angular's [Router](https://angular.io/api/router/Router) under the hood,
     * it's equivalent to calling:
     *
     * ```ts
     * this.navController.setDirection('back');
     * this.router.navigateByUrl(path);
     * ```
     *
     * Going **back** means that all the pages in the stack until the navigated page is found will be popped,
     * and that it will show a "back" animation by default.
     *
     * Navigating back can also be triggered in a declarative manner by using the `[routerDirection]` directive:
     *
     * ```html
     * <a routerLink="/path/to/page" routerDirection="back">Link</a>
     * ```
     */
    navigateBack(url, options = {}) {
        this.setDirection('back', options.animated, options.animationDirection, options.animation);
        return this.navigate(url, options);
    }
    /**
     * This method uses Angular's [Router](https://angular.io/api/router/Router) under the hood,
     * it's equivalent to calling:
     *
     * ```ts
     * this.navController.setDirection('root');
     * this.router.navigateByUrl(path);
     * ```
     *
     * Going **root** means that all existing pages in the stack will be removed,
     * and the navigated page will become the single page in the stack.
     *
     * Navigating root can also be triggered in a declarative manner by using the `[routerDirection]` directive:
     *
     * ```html
     * <a routerLink="/path/to/page" routerDirection="root">Link</a>
     * ```
     */
    navigateRoot(url, options = {}) {
        this.setDirection('root', options.animated, options.animationDirection, options.animation);
        return this.navigate(url, options);
    }
    /**
     * Same as [Location](https://angular.io/api/common/Location)'s back() method.
     * It will use the standard `window.history.back()` under the hood, but featuring a `back` animation
     * by default.
     */
    back(options = { animated: true, animationDirection: 'back' }) {
        this.setDirection('back', options.animated, options.animationDirection, options.animation);
        return this.location.back();
    }
    /**
     * This methods goes back in the context of Ionic's stack navigation.
     *
     * It recursively finds the top active `ion-router-outlet` and calls `pop()`.
     * This is the recommended way to go back when you are using `ion-router-outlet`.
     *
     * Resolves to `true` if it was able to pop.
     */
    async pop() {
        let outlet = this.topOutlet;
        while (outlet) {
            if (await outlet.pop()) {
                return true;
            }
            else {
                outlet = outlet.parentOutlet;
            }
        }
        return false;
    }
    /**
     * This methods specifies the direction of the next navigation performed by the Angular router.
     *
     * `setDirection()` does not trigger any transition, it just sets some flags to be consumed by `ion-router-outlet`.
     *
     * It's recommended to use `navigateForward()`, `navigateBack()` and `navigateRoot()` instead of `setDirection()`.
     */
    setDirection(direction, animated, animationDirection, animationBuilder) {
        this.direction = direction;
        this.animated = getAnimation(direction, animated, animationDirection);
        this.animationBuilder = animationBuilder;
    }
    /**
     * @internal
     */
    setTopOutlet(outlet) {
        this.topOutlet = outlet;
    }
    /**
     * @internal
     */
    consumeTransition() {
        let direction = 'root';
        let animation;
        const animationBuilder = this.animationBuilder;
        if (this.direction === 'auto') {
            direction = this.guessDirection;
            animation = this.guessAnimation;
        }
        else {
            animation = this.animated;
            direction = this.direction;
        }
        this.direction = DEFAULT_DIRECTION;
        this.animated = DEFAULT_ANIMATED;
        this.animationBuilder = undefined;
        return {
            direction,
            animation,
            animationBuilder,
        };
    }
    navigate(url, options) {
        if (Array.isArray(url)) {
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            return this.router.navigate(url, options);
        }
        else {
            /**
             * navigateByUrl ignores any properties that
             * would change the url, so things like queryParams
             * would be ignored unless we create a url tree
             * More Info: https://github.com/angular/angular/issues/18798
             */
            const urlTree = this.serializer.parse(url.toString());
            if (options.queryParams !== undefined) {
                urlTree.queryParams = { ...options.queryParams };
            }
            if (options.fragment !== undefined) {
                urlTree.fragment = options.fragment;
            }
            /**
             * `navigateByUrl` will still apply `NavigationExtras` properties
             * that do not modify the url, such as `replaceUrl` which is why
             * `options` is passed in here.
             */
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            return this.router.navigateByUrl(urlTree, options);
        }
    }
}
/** @nocollapse */ NavController.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.12", ngImport: i0, type: NavController, deps: [{ token: i1.Platform }, { token: i2.Location }, { token: i3.UrlSerializer }, { token: i3.Router, optional: true }], target: i0.ɵɵFactoryTarget.Injectable });
/** @nocollapse */ NavController.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "14.2.12", ngImport: i0, type: NavController, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.12", ngImport: i0, type: NavController, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root',
                }]
        }], ctorParameters: function () { return [{ type: i1.Platform }, { type: i2.Location }, { type: i3.UrlSerializer }, { type: i3.Router, decorators: [{
                    type: Optional
                }] }]; } });
const getAnimation = (direction, animated, animationDirection) => {
    if (animated === false) {
        return undefined;
    }
    if (animationDirection !== undefined) {
        return animationDirection;
    }
    if (direction === 'forward' || direction === 'back') {
        return direction;
    }
    else if (direction === 'root' && animated === true) {
        return 'forward';
    }
    return undefined;
};
const DEFAULT_DIRECTION = 'auto';
const DEFAULT_ANIMATED = undefined;
//# sourceMappingURL=data:application/json;base64,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