import { DOCUMENT } from '@angular/common';
import { Inject, Injectable } from '@angular/core';
import { getPlatforms, isPlatform } from '@ionic/core/components';
import { Subject } from 'rxjs';
import * as i0 from "@angular/core";
export class Platform {
    constructor(doc, zone) {
        this.doc = doc;
        /**
         * @hidden
         */
        this.backButton = new Subject();
        /**
         * The keyboardDidShow event emits when the
         * on-screen keyboard is presented.
         */
        this.keyboardDidShow = new Subject();
        /**
         * The keyboardDidHide event emits when the
         * on-screen keyboard is hidden.
         */
        this.keyboardDidHide = new Subject();
        /**
         * The pause event emits when the native platform puts the application
         * into the background, typically when the user switches to a different
         * application. This event would emit when a Cordova app is put into
         * the background, however, it would not fire on a standard web browser.
         */
        this.pause = new Subject();
        /**
         * The resume event emits when the native platform pulls the application
         * out from the background. This event would emit when a Cordova app comes
         * out from the background, however, it would not fire on a standard web browser.
         */
        this.resume = new Subject();
        /**
         * The resize event emits when the browser window has changed dimensions. This
         * could be from a browser window being physically resized, or from a device
         * changing orientation.
         */
        this.resize = new Subject();
        zone.run(() => {
            this.win = doc.defaultView;
            this.backButton.subscribeWithPriority = function (priority, callback) {
                return this.subscribe((ev) => {
                    return ev.register(priority, (processNextHandler) => zone.run(() => callback(processNextHandler)));
                });
            };
            proxyEvent(this.pause, doc, 'pause', zone);
            proxyEvent(this.resume, doc, 'resume', zone);
            proxyEvent(this.backButton, doc, 'ionBackButton', zone);
            proxyEvent(this.resize, this.win, 'resize', zone);
            proxyEvent(this.keyboardDidShow, this.win, 'ionKeyboardDidShow', zone);
            proxyEvent(this.keyboardDidHide, this.win, 'ionKeyboardDidHide', zone);
            let readyResolve;
            this._readyPromise = new Promise((res) => {
                readyResolve = res;
            });
            if (this.win?.['cordova']) {
                doc.addEventListener('deviceready', () => {
                    readyResolve('cordova');
                }, { once: true });
            }
            else {
                // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                readyResolve('dom');
            }
        });
    }
    /**
     * @returns returns true/false based on platform.
     * @description
     * Depending on the platform the user is on, `is(platformName)` will
     * return `true` or `false`. Note that the same app can return `true`
     * for more than one platform name. For example, an app running from
     * an iPad would return `true` for the platform names: `mobile`,
     * `ios`, `ipad`, and `tablet`. Additionally, if the app was running
     * from Cordova then `cordova` would be true, and if it was running
     * from a web browser on the iPad then `mobileweb` would be `true`.
     *
     * ```
     * import { Platform } from 'ionic-angular';
     *
     * @Component({...})
     * export MyPage {
     *   constructor(public platform: Platform) {
     *     if (this.platform.is('ios')) {
     *       // This will only print when on iOS
     *       console.log('I am an iOS device!');
     *     }
     *   }
     * }
     * ```
     *
     * | Platform Name   | Description                        |
     * |-----------------|------------------------------------|
     * | android         | on a device running Android.       |
     * | capacitor       | on a device running Capacitor.     |
     * | cordova         | on a device running Cordova.       |
     * | ios             | on a device running iOS.           |
     * | ipad            | on an iPad device.                 |
     * | iphone          | on an iPhone device.               |
     * | phablet         | on a phablet device.               |
     * | tablet          | on a tablet device.                |
     * | electron        | in Electron on a desktop device.   |
     * | pwa             | as a PWA app.                      |
     * | mobile          | on a mobile device.                |
     * | mobileweb       | on a mobile device in a browser.   |
     * | desktop         | on a desktop device.               |
     * | hybrid          | is a cordova or capacitor app.     |
     *
     */
    is(platformName) {
        return isPlatform(this.win, platformName);
    }
    /**
     * @returns the array of platforms
     * @description
     * Depending on what device you are on, `platforms` can return multiple values.
     * Each possible value is a hierarchy of platforms. For example, on an iPhone,
     * it would return `mobile`, `ios`, and `iphone`.
     *
     * ```
     * import { Platform } from 'ionic-angular';
     *
     * @Component({...})
     * export MyPage {
     *   constructor(public platform: Platform) {
     *     // This will print an array of the current platforms
     *     console.log(this.platform.platforms());
     *   }
     * }
     * ```
     */
    platforms() {
        return getPlatforms(this.win);
    }
    /**
     * Returns a promise when the platform is ready and native functionality
     * can be called. If the app is running from within a web browser, then
     * the promise will resolve when the DOM is ready. When the app is running
     * from an application engine such as Cordova, then the promise will
     * resolve when Cordova triggers the `deviceready` event.
     *
     * The resolved value is the `readySource`, which states which platform
     * ready was used. For example, when Cordova is ready, the resolved ready
     * source is `cordova`. The default ready source value will be `dom`. The
     * `readySource` is useful if different logic should run depending on the
     * platform the app is running from. For example, only Cordova can execute
     * the status bar plugin, so the web should not run status bar plugin logic.
     *
     * ```
     * import { Component } from '@angular/core';
     * import { Platform } from 'ionic-angular';
     *
     * @Component({...})
     * export MyApp {
     *   constructor(public platform: Platform) {
     *     this.platform.ready().then((readySource) => {
     *       console.log('Platform ready from', readySource);
     *       // Platform now ready, execute any required native code
     *     });
     *   }
     * }
     * ```
     */
    ready() {
        return this._readyPromise;
    }
    /**
     * Returns if this app is using right-to-left language direction or not.
     * We recommend the app's `index.html` file already has the correct `dir`
     * attribute value set, such as `<html dir="ltr">` or `<html dir="rtl">`.
     * [W3C: Structural markup and right-to-left text in HTML](http://www.w3.org/International/questions/qa-html-dir)
     */
    get isRTL() {
        return this.doc.dir === 'rtl';
    }
    /**
     * Get the query string parameter
     */
    getQueryParam(key) {
        return readQueryParam(this.win.location.href, key);
    }
    /**
     * Returns `true` if the app is in landscape mode.
     */
    isLandscape() {
        return !this.isPortrait();
    }
    /**
     * Returns `true` if the app is in portrait mode.
     */
    isPortrait() {
        return this.win.matchMedia?.('(orientation: portrait)').matches;
    }
    testUserAgent(expression) {
        const nav = this.win.navigator;
        return !!(nav?.userAgent && nav.userAgent.indexOf(expression) >= 0);
    }
    /**
     * Get the current url.
     */
    url() {
        return this.win.location.href;
    }
    /**
     * Gets the width of the platform's viewport using `window.innerWidth`.
     */
    width() {
        return this.win.innerWidth;
    }
    /**
     * Gets the height of the platform's viewport using `window.innerHeight`.
     */
    height() {
        return this.win.innerHeight;
    }
}
/** @nocollapse */ Platform.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.2.12", ngImport: i0, type: Platform, deps: [{ token: DOCUMENT }, { token: i0.NgZone }], target: i0.ɵɵFactoryTarget.Injectable });
/** @nocollapse */ Platform.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "14.2.12", ngImport: i0, type: Platform, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.2.12", ngImport: i0, type: Platform, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root',
                }]
        }], ctorParameters: function () { return [{ type: undefined, decorators: [{
                    type: Inject,
                    args: [DOCUMENT]
                }] }, { type: i0.NgZone }]; } });
const readQueryParam = (url, key) => {
    key = key.replace(/[[\]\\]/g, '\\$&');
    const regex = new RegExp('[\\?&]' + key + '=([^&#]*)');
    const results = regex.exec(url);
    return results ? decodeURIComponent(results[1].replace(/\+/g, ' ')) : null;
};
const proxyEvent = (emitter, el, eventName, zone) => {
    if (el) {
        el.addEventListener(eventName, (ev) => {
            /**
             * `zone.run` is required to make sure that we are running inside the Angular zone
             * at all times. This is necessary since an app that has Capacitor will
             * override the `document.addEventListener` with its own implementation.
             * The override causes the event to no longer be in the Angular zone.
             */
            zone.run(() => {
                // ?? cordova might emit "null" events
                const value = ev != null ? ev.detail : undefined;
                emitter.next(value);
            });
        });
    }
};
//# sourceMappingURL=data:application/json;base64,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