"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.format = void 0;
const tslib_1 = require("tslib");
const child_process_1 = require("child_process");
const path = require("path");
const command_line_utils_1 = require("../../utils/command-line-utils");
const fileutils_1 = require("../../utils/fileutils");
const file_utils_1 = require("../../project-graph/file-utils");
const prettier = require("prettier");
const object_sort_1 = require("../../utils/object-sort");
const typescript_1 = require("../../plugins/js/utils/typescript");
const project_graph_1 = require("../../project-graph/project-graph");
const affected_project_graph_1 = require("../../project-graph/affected/affected-project-graph");
const configuration_1 = require("../../config/configuration");
const chunkify_1 = require("../../utils/chunkify");
const all_file_data_1 = require("../../utils/all-file-data");
const PRETTIER_PATH = require.resolve('prettier/bin-prettier');
function format(command, args) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const { nxArgs } = (0, command_line_utils_1.splitArgsIntoNxArgsAndOverrides)(args, 'affected', { printWarnings: false }, (0, configuration_1.readNxJson)());
        const patterns = (yield getPatterns(Object.assign(Object.assign({}, args), nxArgs))).map(
        // prettier removes one of the \
        // prettier-ignore
        (p) => `"${p.replace(/\$/g, "\\\$")}"`);
        // Chunkify the patterns array to prevent crashing the windows terminal
        const chunkList = (0, chunkify_1.chunkify)(patterns);
        switch (command) {
            case 'write':
                sortTsConfig();
                addRootConfigFiles(chunkList, nxArgs);
                chunkList.forEach((chunk) => write(chunk));
                break;
            case 'check':
                const pass = chunkList.reduce((pass, chunk) => check(chunk) && pass, true);
                if (!pass) {
                    process.exit(1);
                }
                break;
        }
    });
}
exports.format = format;
function getPatterns(args) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const graph = yield (0, project_graph_1.createProjectGraphAsync)({ exitOnError: true });
        const allFilesPattern = ['.'];
        if (args.all) {
            return allFilesPattern;
        }
        try {
            if (args.projects && args.projects.length > 0) {
                return getPatternsFromProjects(args.projects, graph);
            }
            const p = (0, command_line_utils_1.parseFiles)(args);
            const supportedExtensions = prettier
                .getSupportInfo()
                .languages.flatMap((language) => language.extensions)
                .filter((extension) => !!extension)
                // Prettier supports ".swcrc" as a file instead of an extension
                // So we add ".swcrc" as a supported extension manually
                // which allows it to be considered for calculating "patterns"
                .concat('.swcrc');
            const patterns = p.files.filter((f) => (0, fileutils_1.fileExists)(f) && supportedExtensions.includes(path.extname(f)));
            return args.libsAndApps
                ? yield getPatternsFromApps(patterns, yield (0, all_file_data_1.allFileData)(), graph)
                : patterns;
        }
        catch (_a) {
            return allFilesPattern;
        }
    });
}
function getPatternsFromApps(affectedFiles, allWorkspaceFiles, projectGraph) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const graph = yield (0, project_graph_1.createProjectGraphAsync)({ exitOnError: true });
        const affectedGraph = yield (0, affected_project_graph_1.filterAffected)(graph, (0, file_utils_1.calculateFileChanges)(affectedFiles, allWorkspaceFiles));
        return getPatternsFromProjects(Object.keys(affectedGraph.nodes), projectGraph);
    });
}
function addRootConfigFiles(chunkList, nxArgs) {
    if (nxArgs.all) {
        return;
    }
    const chunk = [];
    const addToChunkIfNeeded = (file) => {
        if (chunkList.every((c) => !c.includes(`"${file}"`))) {
            chunk.push(file);
        }
    };
    // if (workspaceJsonPath) {
    //   addToChunkIfNeeded(workspaceJsonPath);
    // }
    ['nx.json', (0, typescript_1.getRootTsConfigFileName)()]
        .filter(Boolean)
        .forEach(addToChunkIfNeeded);
    if (chunk.length > 0) {
        chunkList.push(chunk);
    }
}
function getPatternsFromProjects(projects, projectGraph) {
    return (0, command_line_utils_1.getProjectRoots)(projects, projectGraph);
}
function write(patterns) {
    if (patterns.length > 0) {
        const [swcrcPatterns, regularPatterns] = patterns.reduce((result, pattern) => {
            result[pattern.includes('.swcrc') ? 0 : 1].push(pattern);
            return result;
        }, [[], []]);
        (0, child_process_1.execSync)(`node "${PRETTIER_PATH}" --write --list-different ${regularPatterns.join(' ')}`, {
            stdio: [0, 1, 2],
        });
        if (swcrcPatterns.length > 0) {
            (0, child_process_1.execSync)(`node "${PRETTIER_PATH}" --write --list-different ${swcrcPatterns.join(' ')} --parser json`, {
                stdio: [0, 1, 2],
            });
        }
    }
}
function check(patterns) {
    if (patterns.length === 0) {
        return true;
    }
    try {
        (0, child_process_1.execSync)(`node "${PRETTIER_PATH}" --list-different ${patterns.join(' ')}`, {
            stdio: [0, 1, 2],
        });
        return true;
    }
    catch (_a) {
        return false;
    }
}
function sortTsConfig() {
    try {
        const tsconfigPath = (0, typescript_1.getRootTsConfigPath)();
        const tsconfig = (0, fileutils_1.readJsonFile)(tsconfigPath);
        const sortedPaths = (0, object_sort_1.sortObjectByKeys)(tsconfig.compilerOptions.paths);
        tsconfig.compilerOptions.paths = sortedPaths;
        (0, fileutils_1.writeJsonFile)(tsconfigPath, tsconfig);
    }
    catch (e) {
        // catch noop
    }
}
