"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.initHandler = void 0;
const tslib_1 = require("tslib");
const child_process_1 = require("child_process");
const enquirer_1 = require("enquirer");
const fs_1 = require("fs");
const semver_1 = require("semver");
const parser = require("yargs-parser");
const add_nx_to_monorepo_1 = require("./implementation/add-nx-to-monorepo");
const add_nx_to_nest_1 = require("./implementation/add-nx-to-nest");
const add_nx_to_npm_repo_1 = require("./implementation/add-nx-to-npm-repo");
const angular_1 = require("./implementation/angular");
const add_nx_scripts_1 = require("./implementation/dot-nx/add-nx-scripts");
const react_1 = require("./implementation/react");
const child_process_2 = require("../../utils/child-process");
const fileutils_1 = require("../../utils/fileutils");
const versions_1 = require("../../utils/versions");
function initHandler(options) {
    var _a;
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const args = process.argv.slice(2).join(' ');
        const flags = parser(args, {
            boolean: ['useDotNxInstallation'],
            alias: {
                useDotNxInstallation: ['encapsulated'],
            },
            default: {
                useDotNxInstallation: false,
            },
        });
        const version = (_a = process.env.NX_VERSION) !== null && _a !== void 0 ? _a : ((0, semver_1.prerelease)(versions_1.nxVersion) ? 'next' : 'latest');
        if (process.env.NX_VERSION) {
            console.log(`Using version ${process.env.NX_VERSION}`);
        }
        if (flags.useDotNxInstallation === true) {
            setupDotNxInstallation(version);
        }
        else if ((0, fs_1.existsSync)('package.json')) {
            const packageJson = (0, fileutils_1.readJsonFile)('package.json');
            if ((0, fs_1.existsSync)('angular.json')) {
                yield (0, angular_1.addNxToAngularCliRepo)(options);
            }
            else if (isCRA(packageJson)) {
                yield (0, react_1.addNxToCraRepo)(options);
            }
            else if (isNestCLI(packageJson)) {
                yield (0, add_nx_to_nest_1.addNxToNest)(options, packageJson);
            }
            else if (isMonorepo(packageJson)) {
                yield (0, add_nx_to_monorepo_1.addNxToMonorepo)(options);
            }
            else {
                yield (0, add_nx_to_npm_repo_1.addNxToNpmRepo)(options);
            }
        }
        else {
            const useDotNxFolder = yield (0, enquirer_1.prompt)([
                {
                    name: 'useDotNxFolder',
                    type: 'autocomplete',
                    message: 'Where should your workspace be created?',
                    choices: [
                        {
                            name: 'In a new folder under this directory',
                            value: 'false',
                        },
                        {
                            name: 'In this directory',
                            value: 'true',
                        },
                    ],
                },
            ]).then((r) => r.useDotNxFolder === 'true');
            if (useDotNxFolder) {
                setupDotNxInstallation(version);
            }
            else {
                (0, child_process_1.execSync)(`npx --yes create-nx-workspace@${version} ${args}`, {
                    stdio: [0, 1, 2],
                });
            }
        }
    });
}
exports.initHandler = initHandler;
function isCRA(packageJson) {
    const combinedDependencies = Object.assign(Object.assign({}, packageJson.dependencies), packageJson.devDependencies);
    return (
    // Required dependencies for CRA projects
    combinedDependencies['react'] &&
        combinedDependencies['react-dom'] &&
        combinedDependencies['react-scripts'] &&
        // // Don't convert customized CRA projects
        !combinedDependencies['react-app-rewired'] &&
        !combinedDependencies['@craco/craco'] &&
        (0, fileutils_1.directoryExists)('src') &&
        (0, fileutils_1.directoryExists)('public'));
}
function isNestCLI(packageJson) {
    const combinedDependencies = Object.assign(Object.assign({}, packageJson.dependencies), packageJson.devDependencies);
    return ((0, fs_1.existsSync)('nest-cli.json') &&
        combinedDependencies['@nestjs/core'] &&
        combinedDependencies['@nestjs/cli']);
}
function isMonorepo(packageJson) {
    if (!!packageJson.workspaces)
        return true;
    if ((0, fs_1.existsSync)('pnpm-workspace.yaml') || (0, fs_1.existsSync)('pnpm-workspace.yml'))
        return true;
    if ((0, fs_1.existsSync)('lerna.json'))
        return true;
    return false;
}
function setupDotNxInstallation(version) {
    if (process.platform !== 'win32') {
        console.log('Setting Nx up installation in `.nx`. You can run nx commands like: `./nx --help`');
    }
    else {
        console.log('Setting Nx up installation in `.nx`. You can run nx commands like: `./nx.bat --help`');
    }
    (0, add_nx_scripts_1.generateDotNxSetup)(version);
    // invokes the wrapper, thus invoking the initial installation process
    (0, child_process_2.runNxSync)('');
}
