"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findInstalledPackagesWeCareAbout = exports.findInstalledCommunityPlugins = exports.findMisalignedPackagesForPackage = exports.getReportData = exports.reportHandler = exports.patternsWeIgnoreInCommunityReport = exports.packagesWeCareAbout = void 0;
const tslib_1 = require("tslib");
const chalk = require("chalk");
const output_1 = require("../../utils/output");
const path_1 = require("path");
const package_manager_1 = require("../../utils/package-manager");
const fileutils_1 = require("../../utils/fileutils");
const package_json_1 = require("../../utils/package-json");
const local_plugins_1 = require("../../utils/plugins/local-plugins");
const project_graph_1 = require("../../project-graph/project-graph");
const semver_1 = require("semver");
const installed_plugins_1 = require("../../utils/plugins/installed-plugins");
const installation_directory_1 = require("../../utils/installation-directory");
const nxPackageJson = (0, fileutils_1.readJsonFile)((0, path_1.join)(__dirname, '../../../package.json'));
exports.packagesWeCareAbout = [
    'lerna',
    ...nxPackageJson['nx-migrations'].packageGroup.map((x) => typeof x === 'string' ? x : x.package),
    '@nrwl/schematics',
    'typescript',
];
exports.patternsWeIgnoreInCommunityReport = [
    ...exports.packagesWeCareAbout,
    '@schematics/angular',
    new RegExp('@angular/*'),
    '@nestjs/schematics',
];
const LINE_SEPARATOR = '---------------------------------------';
/**
 * Reports relevant version numbers for adding to an Nx issue report
 *
 * @remarks
 *
 * Must be run within an Nx workspace
 *
 */
function reportHandler() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const { pm, pmVersion, localPlugins, communityPlugins, packageVersionsWeCareAbout, outOfSyncPackageGroup, projectGraphError, } = yield getReportData();
        const bodyLines = [
            `Node   : ${process.versions.node}`,
            `OS     : ${process.platform}-${process.arch}`,
            `${pm.padEnd(7)}: ${pmVersion}`,
            ``,
        ];
        let padding = Math.max(...packageVersionsWeCareAbout.map((x) => x.package.length)) + 1;
        packageVersionsWeCareAbout.forEach((p) => {
            bodyLines.push(`${chalk.green(p.package.padEnd(padding))} : ${chalk.bold(p.version)}`);
        });
        if (communityPlugins.length) {
            bodyLines.push(LINE_SEPARATOR);
            padding = Math.max(...communityPlugins.map((x) => x.name.length)) + 1;
            bodyLines.push('Community plugins:');
            communityPlugins.forEach((p) => {
                bodyLines.push(`${chalk.green(p.name.padEnd(padding))}: ${chalk.bold(p.version)}`);
            });
        }
        if (localPlugins.length) {
            bodyLines.push(LINE_SEPARATOR);
            bodyLines.push('Local workspace plugins:');
            for (const plugin of localPlugins) {
                bodyLines.push(`\t ${chalk.green(plugin)}`);
            }
        }
        if (outOfSyncPackageGroup) {
            bodyLines.push(LINE_SEPARATOR);
            bodyLines.push(`The following packages should match the installed version of ${outOfSyncPackageGroup.basePackage}`);
            for (const pkg of outOfSyncPackageGroup.misalignedPackages) {
                bodyLines.push(`  - ${pkg.name}@${pkg.version}`);
            }
            bodyLines.push('');
            bodyLines.push(`To fix this, run \`nx migrate ${outOfSyncPackageGroup.migrateTarget}\``);
        }
        if (projectGraphError) {
            bodyLines.push(LINE_SEPARATOR);
            bodyLines.push('⚠️ Unable to construct project graph.');
            bodyLines.push(projectGraphError.message);
            bodyLines.push(projectGraphError.stack);
        }
        output_1.output.log({
            title: 'Report complete - copy this into the issue template',
            bodyLines,
        });
    });
}
exports.reportHandler = reportHandler;
function getReportData() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const pm = (0, package_manager_1.detectPackageManager)();
        const pmVersion = (0, package_manager_1.getPackageManagerVersion)(pm);
        const localPlugins = yield findLocalPlugins();
        const communityPlugins = findInstalledCommunityPlugins();
        let projectGraphError = null;
        if (isNativeAvailable()) {
            try {
                yield (0, project_graph_1.createProjectGraphAsync)();
            }
            catch (e) {
                projectGraphError = e;
            }
        }
        const packageVersionsWeCareAbout = findInstalledPackagesWeCareAbout();
        packageVersionsWeCareAbout.unshift({
            package: 'nx',
            version: nxPackageJson.version,
        });
        if (globalThis.GLOBAL_NX_VERSION) {
            packageVersionsWeCareAbout.unshift({
                package: 'nx (global)',
                version: globalThis.GLOBAL_NX_VERSION,
            });
        }
        const outOfSyncPackageGroup = findMisalignedPackagesForPackage(nxPackageJson);
        return {
            pm,
            pmVersion,
            localPlugins,
            communityPlugins,
            packageVersionsWeCareAbout,
            outOfSyncPackageGroup,
            projectGraphError,
        };
    });
}
exports.getReportData = getReportData;
function findLocalPlugins() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            const projectGraph = yield (0, project_graph_1.createProjectGraphAsync)({ exitOnError: true });
            const localPlugins = yield (0, local_plugins_1.getLocalWorkspacePlugins)((0, project_graph_1.readProjectsConfigurationFromProjectGraph)(projectGraph));
            return Array.from(localPlugins.keys());
        }
        catch (_a) {
            return [];
        }
    });
}
function readPackageJson(p) {
    try {
        return (0, package_json_1.readModulePackageJson)(p, (0, installation_directory_1.getNxRequirePaths)()).packageJson;
    }
    catch (_a) {
        return null;
    }
}
function readPackageVersion(p) {
    var _a;
    return (_a = readPackageJson(p)) === null || _a === void 0 ? void 0 : _a.version;
}
function findMisalignedPackagesForPackage(base) {
    const misalignedPackages = [];
    let migrateTarget = base.version;
    const { packageGroup } = (0, package_json_1.readNxMigrateConfig)(base);
    for (const entry of packageGroup !== null && packageGroup !== void 0 ? packageGroup : []) {
        const { package: packageName, version } = entry;
        // should be aligned
        if (version === '*') {
            const installedVersion = readPackageVersion(packageName);
            if (installedVersion && installedVersion !== base.version) {
                if ((0, semver_1.valid)(installedVersion) && (0, semver_1.gt)(installedVersion, migrateTarget)) {
                    migrateTarget = installedVersion;
                }
                misalignedPackages.push({
                    name: packageName,
                    version: installedVersion,
                });
            }
        }
    }
    return misalignedPackages.length
        ? {
            basePackage: base.name,
            misalignedPackages,
            migrateTarget: `${base.name}@${migrateTarget}`,
        }
        : undefined;
}
exports.findMisalignedPackagesForPackage = findMisalignedPackagesForPackage;
function findInstalledCommunityPlugins() {
    const installedPlugins = (0, installed_plugins_1.findInstalledPlugins)();
    return installedPlugins.filter((dep) => dep.name !== 'nx' &&
        !exports.patternsWeIgnoreInCommunityReport.some((pattern) => typeof pattern === 'string'
            ? pattern === dep.name
            : pattern.test(dep.name)));
}
exports.findInstalledCommunityPlugins = findInstalledCommunityPlugins;
function findInstalledPackagesWeCareAbout() {
    var _a, _b;
    const packagesWeMayCareAbout = {};
    // TODO (v17): Remove workaround for hiding @nrwl packages when matching @nx package is found.
    const packageChangeMap = {
        '@nrwl/nx-plugin': '@nx/plugin',
        '@nx/plugin': '@nrwl/nx-plugin',
        '@nrwl/eslint-plugin-nx': '@nx/eslint-plugin',
        '@nx/eslint-plugin': '@nrwl/eslint-plugin-nx',
        '@nrwl/nx-cloud': 'nx-cloud',
    };
    for (const pkg of exports.packagesWeCareAbout) {
        const v = readPackageVersion(pkg);
        if (v) {
            // If its a @nrwl scoped package, keep the version if there is no
            // corresponding @nx scoped package, or it has a different version.
            if (pkg.startsWith('@nrwl/')) {
                const otherPackage = (_a = packageChangeMap[pkg]) !== null && _a !== void 0 ? _a : pkg.replace('@nrwl/', '@nx/');
                const otherVersion = packagesWeMayCareAbout[otherPackage];
                if (!otherVersion || v !== otherVersion) {
                    packagesWeMayCareAbout[pkg] = v;
                }
                // If its a @nx scoped package, always keep the version, and
                // remove the corresponding @nrwl scoped package if it exists.
            }
            else if (pkg.startsWith('@nx/')) {
                const otherPackage = (_b = packageChangeMap[pkg]) !== null && _b !== void 0 ? _b : pkg.replace('@nx/', '@nrwl/');
                const otherVersion = packagesWeMayCareAbout[otherPackage];
                if (otherVersion && v === otherVersion) {
                    delete packagesWeMayCareAbout[otherPackage];
                }
                packagesWeMayCareAbout[pkg] = v;
            }
            else {
                packagesWeMayCareAbout[pkg] = v;
            }
        }
    }
    return Object.entries(packagesWeMayCareAbout).map(([pkg, version]) => ({
        package: pkg,
        version,
    }));
}
exports.findInstalledPackagesWeCareAbout = findInstalledPackagesWeCareAbout;
function isNativeAvailable() {
    try {
        require('../../native');
        return true;
    }
    catch (_a) {
        return false;
    }
}
