"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.calculateDefaultProjectName = exports.runOne = void 0;
const tslib_1 = require("tslib");
const run_command_1 = require("../../tasks-runner/run-command");
const command_line_utils_1 = require("../../utils/command-line-utils");
const connect_to_nx_cloud_1 = require("../connect/connect-to-nx-cloud");
const perf_hooks_1 = require("perf_hooks");
const project_graph_1 = require("../../project-graph/project-graph");
const workspace_root_1 = require("../../utils/workspace-root");
const split_target_1 = require("../../utils/split-target");
const output_1 = require("../../utils/output");
const configuration_1 = require("../../config/configuration");
const workspace_configuration_check_1 = require("../../utils/workspace-configuration-check");
const graph_1 = require("../graph/graph");
function runOne(cwd, args, extraTargetDependencies = {}, extraOptions = { excludeTaskDependencies: false, loadDotEnvFiles: true }) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        perf_hooks_1.performance.mark('code-loading:end');
        perf_hooks_1.performance.measure('code-loading', 'init-local', 'code-loading:end');
        (0, workspace_configuration_check_1.workspaceConfigurationCheck)();
        const nxJson = (0, configuration_1.readNxJson)();
        const projectGraph = yield (0, project_graph_1.createProjectGraphAsync)({ exitOnError: true });
        const opts = parseRunOneOptions(cwd, args, projectGraph, nxJson);
        const { nxArgs, overrides } = (0, command_line_utils_1.splitArgsIntoNxArgsAndOverrides)(Object.assign(Object.assign({}, opts.parsedArgs), { configuration: opts.configuration, targets: [opts.target] }), 'run-one', { printWarnings: args.graph !== 'stdout' }, nxJson);
        if (nxArgs.verbose) {
            process.env.NX_VERBOSE_LOGGING = 'true';
        }
        if (nxArgs.help) {
            yield (yield Promise.resolve().then(() => require('./run'))).printTargetRunHelp(opts, workspace_root_1.workspaceRoot);
            process.exit(0);
        }
        yield (0, connect_to_nx_cloud_1.connectToNxCloudIfExplicitlyAsked)(nxArgs);
        const { projects } = getProjects(projectGraph, opts.project);
        if (nxArgs.graph) {
            const projectNames = projects.map((t) => t.name);
            const file = (0, command_line_utils_1.readGraphFileFromGraphArg)(nxArgs);
            return yield (0, graph_1.generateGraph)({
                watch: false,
                open: true,
                view: 'tasks',
                targets: nxArgs.targets,
                projects: projectNames,
                file,
            }, projectNames);
        }
        else {
            yield (0, run_command_1.runCommand)(projects, projectGraph, { nxJson }, nxArgs, overrides, opts.project, extraTargetDependencies, extraOptions);
        }
    });
}
exports.runOne = runOne;
function getProjects(projectGraph, project) {
    if (!projectGraph.nodes[project]) {
        output_1.output.error({
            title: `Cannot find project '${project}'`,
        });
        process.exit(1);
    }
    let projects = [projectGraph.nodes[project]];
    let projectsMap = {
        [project]: projectGraph.nodes[project],
    };
    return { projects, projectsMap };
}
const targetAliases = {
    b: 'build',
    e: 'e2e',
    l: 'lint',
    s: 'serve',
    t: 'test',
};
function parseRunOneOptions(cwd, parsedArgs, projectGraph, nxJson) {
    const defaultProjectName = calculateDefaultProjectName(cwd, workspace_root_1.workspaceRoot, (0, project_graph_1.readProjectsConfigurationFromProjectGraph)(projectGraph), nxJson);
    let project;
    let target;
    let configuration;
    if (parsedArgs['project:target:configuration'].indexOf(':') > -1) {
        // run case
        [project, target, configuration] = (0, split_target_1.splitTarget)(parsedArgs['project:target:configuration'], projectGraph);
        // this is to account for "nx npmsript:dev"
        if (project && !target && defaultProjectName) {
            target = project;
            project = defaultProjectName;
        }
    }
    else {
        target = parsedArgs['project:target:configuration'];
    }
    if (parsedArgs.project) {
        project = parsedArgs.project;
    }
    if (!project && defaultProjectName) {
        project = defaultProjectName;
    }
    if (!project || !target) {
        throw new Error(`Both project and target have to be specified`);
    }
    if (targetAliases[target]) {
        target = targetAliases[target];
    }
    if (parsedArgs.configuration) {
        configuration = parsedArgs.configuration;
    }
    else if (parsedArgs.prod) {
        configuration = 'production';
    }
    const res = { project, target, configuration, parsedArgs };
    delete parsedArgs['c'];
    delete parsedArgs['project:target:configuration'];
    delete parsedArgs['configuration'];
    delete parsedArgs['prod'];
    delete parsedArgs['project'];
    return res;
}
function calculateDefaultProjectName(cwd, root, projectsConfigurations, nxJsonConfiguration) {
    var _a;
    if (cwd && /^[A-Z]:/.test(cwd)) {
        cwd = cwd.charAt(0).toLowerCase() + cwd.slice(1);
    }
    if (root && /^[A-Z]:/.test(root)) {
        root = root.charAt(0).toLowerCase() + root.slice(1);
    }
    let relativeCwd = cwd.replace(/\\/g, '/').split(root.replace(/\\/g, '/'))[1];
    relativeCwd = relativeCwd.startsWith('/')
        ? relativeCwd.substring(1)
        : relativeCwd;
    const matchingProject = Object.keys(projectsConfigurations.projects).find((p) => {
        const projectRoot = projectsConfigurations.projects[p].root;
        return (relativeCwd == projectRoot ||
            (relativeCwd == '' && projectRoot == '.') ||
            relativeCwd.startsWith(`${projectRoot}/`));
    });
    if (matchingProject)
        return matchingProject;
    return (((_a = nxJsonConfiguration.cli) === null || _a === void 0 ? void 0 : _a.defaultProjectName) || nxJsonConfiguration.defaultProject);
}
exports.calculateDefaultProjectName = calculateDefaultProjectName;
