"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addUpdatedAndDeletedFiles = exports.getCachedSerializedProjectGraphPromise = exports.currentProjectGraph = exports.currentProjectFileMapCache = exports.projectFileMapWithFiles = void 0;
const tslib_1 = require("tslib");
const perf_hooks_1 = require("perf_hooks");
const build_project_graph_1 = require("../../project-graph/build-project-graph");
const file_map_utils_1 = require("../../project-graph/file-map-utils");
const nx_deps_cache_1 = require("../../project-graph/nx-deps-cache");
const fileutils_1 = require("../../utils/fileutils");
const file_watcher_sockets_1 = require("./file-watching/file-watcher-sockets");
const logger_1 = require("./logger");
const workspaces_1 = require("../../config/workspaces");
const workspace_root_1 = require("../../utils/workspace-root");
const child_process_1 = require("child_process");
const file_hasher_1 = require("../../hasher/file-hasher");
const retrieve_workspace_files_1 = require("../../project-graph/utils/retrieve-workspace-files");
let cachedSerializedProjectGraphPromise;
const collectedUpdatedFiles = new Set();
const collectedDeletedFiles = new Set();
let storedWorkspaceConfigHash;
let waitPeriod = 100;
let scheduledTimeoutId;
function getCachedSerializedProjectGraphPromise() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            // recomputing it now on demand. we can ignore the scheduled timeout
            if (scheduledTimeoutId) {
                clearTimeout(scheduledTimeoutId);
                scheduledTimeoutId = undefined;
            }
            // reset the wait time
            waitPeriod = 100;
            yield resetInternalStateIfNxDepsMissing();
            if (collectedUpdatedFiles.size == 0 && collectedDeletedFiles.size == 0) {
                if (!cachedSerializedProjectGraphPromise) {
                    cachedSerializedProjectGraphPromise =
                        processFilesAndCreateAndSerializeProjectGraph();
                }
            }
            else {
                cachedSerializedProjectGraphPromise =
                    processFilesAndCreateAndSerializeProjectGraph();
            }
            return yield cachedSerializedProjectGraphPromise;
        }
        catch (e) {
            return {
                error: e,
                serializedProjectGraph: null,
                projectGraph: null,
                projectFileMap: null,
                allWorkspaceFiles: null,
            };
        }
    });
}
exports.getCachedSerializedProjectGraphPromise = getCachedSerializedProjectGraphPromise;
function addUpdatedAndDeletedFiles(createdFiles, updatedFiles, deletedFiles) {
    for (let f of [...createdFiles, ...updatedFiles]) {
        collectedDeletedFiles.delete(f);
        collectedUpdatedFiles.add(f);
    }
    for (let f of deletedFiles) {
        collectedUpdatedFiles.delete(f);
        collectedDeletedFiles.add(f);
    }
    if (updatedFiles.length > 0 || deletedFiles.length > 0) {
        (0, file_watcher_sockets_1.notifyFileWatcherSockets)(null, updatedFiles, deletedFiles);
    }
    if (createdFiles.length > 0) {
        waitPeriod = 100; // reset it to process the graph faster
    }
    if (!scheduledTimeoutId) {
        scheduledTimeoutId = setTimeout(() => tslib_1.__awaiter(this, void 0, void 0, function* () {
            scheduledTimeoutId = undefined;
            if (waitPeriod < 4000) {
                waitPeriod = waitPeriod * 2;
            }
            cachedSerializedProjectGraphPromise =
                processFilesAndCreateAndSerializeProjectGraph();
            yield cachedSerializedProjectGraphPromise;
            if (createdFiles.length > 0) {
                (0, file_watcher_sockets_1.notifyFileWatcherSockets)(createdFiles, null, null);
            }
        }), waitPeriod);
    }
}
exports.addUpdatedAndDeletedFiles = addUpdatedAndDeletedFiles;
function computeWorkspaceConfigHash(projectsConfigurations) {
    return (0, file_hasher_1.hashArray)([JSON.stringify(projectsConfigurations)]);
}
/**
 * Temporary work around to handle nested gitignores. The parcel file watcher doesn't handle them well,
 * so we need to filter them out here.
 *
 * TODO(Cammisuli): remove after 16.4 - Rust watcher handles nested gitignores
 */
function filterUpdatedFiles(files) {
    try {
        const quoted = files.map((f) => '"' + f + '"');
        const ignored = (0, child_process_1.execSync)(`git check-ignore ${quoted.join(' ')}`, {
            windowsHide: true,
        })
            .toString()
            .split('\n');
        return files.filter((f) => ignored.indexOf(f) === -1);
    }
    catch (e) {
        // none of the files were ignored
        return files;
    }
}
function processCollectedUpdatedAndDeletedFiles() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            perf_hooks_1.performance.mark('hash-watched-changes-start');
            const updatedFiles = yield file_hasher_1.fileHasher.hashFiles(filterUpdatedFiles([...collectedUpdatedFiles.values()]));
            const deletedFiles = [...collectedDeletedFiles.values()];
            perf_hooks_1.performance.mark('hash-watched-changes-end');
            perf_hooks_1.performance.measure('hash changed files from watcher', 'hash-watched-changes-start', 'hash-watched-changes-end');
            file_hasher_1.fileHasher.incrementalUpdate(updatedFiles, deletedFiles);
            let nxJson = new workspaces_1.Workspaces(workspace_root_1.workspaceRoot).readNxJson();
            const projectConfigurations = yield (0, retrieve_workspace_files_1.retrieveProjectConfigurations)(workspace_root_1.workspaceRoot, nxJson);
            const workspaceConfigHash = computeWorkspaceConfigHash(projectConfigurations);
            logger_1.serverLogger.requestLog(`Updated file-hasher based on watched changes, recomputing project graph...`);
            logger_1.serverLogger.requestLog([...updatedFiles.values()]);
            logger_1.serverLogger.requestLog([...deletedFiles]);
            // when workspace config changes we cannot incrementally update project file map
            if (workspaceConfigHash !== storedWorkspaceConfigHash) {
                storedWorkspaceConfigHash = workspaceConfigHash;
                exports.projectFileMapWithFiles = yield (0, retrieve_workspace_files_1.retrieveWorkspaceFiles)(workspace_root_1.workspaceRoot, nxJson);
            }
            else {
                if (exports.projectFileMapWithFiles) {
                    exports.projectFileMapWithFiles = (0, file_map_utils_1.updateProjectFileMap)(projectConfigurations, exports.projectFileMapWithFiles.projectFileMap, exports.projectFileMapWithFiles.allWorkspaceFiles, updatedFiles, deletedFiles);
                }
                else {
                    exports.projectFileMapWithFiles = yield (0, retrieve_workspace_files_1.retrieveWorkspaceFiles)(workspace_root_1.workspaceRoot, nxJson);
                }
            }
            collectedUpdatedFiles.clear();
            collectedDeletedFiles.clear();
        }
        catch (e) {
            // this is expected
            // for instance, project.json can be incorrect or a file we are trying to has
            // has been deleted
            // we are resetting internal state to start from scratch next time a file changes
            // given the user the opportunity to fix the error
            // if Nx requests the project graph prior to the error being fixed,
            // the error will be propagated
            logger_1.serverLogger.log(`Error detected when recomputing project file map: ${e.message}`);
            resetInternalState();
            return e;
        }
    });
}
function processFilesAndCreateAndSerializeProjectGraph() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const err = yield processCollectedUpdatedAndDeletedFiles();
        if (err) {
            return Promise.resolve({
                error: err,
                projectGraph: null,
                projectFileMap: null,
                allWorkspaceFiles: null,
                serializedProjectGraph: null,
            });
        }
        else {
            return createAndSerializeProjectGraph();
        }
    });
}
function copyFileData(d) {
    return d.map((t) => (Object.assign({}, t)));
}
function copyFileMap(m) {
    const c = {};
    for (let p of Object.keys(m)) {
        c[p] = copyFileData(m[p]);
    }
    return c;
}
function createAndSerializeProjectGraph() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            perf_hooks_1.performance.mark('create-project-graph-start');
            const projectsConfigurations = new workspaces_1.Workspaces(workspace_root_1.workspaceRoot).readProjectsConfigurations();
            const projectFileMap = copyFileMap(exports.projectFileMapWithFiles.projectFileMap);
            const allWorkspaceFiles = copyFileData(exports.projectFileMapWithFiles.allWorkspaceFiles);
            const { projectGraph, projectFileMapCache } = yield (0, build_project_graph_1.buildProjectGraphUsingProjectFileMap)(projectsConfigurations, projectFileMap, allWorkspaceFiles, exports.currentProjectFileMapCache || (0, nx_deps_cache_1.readProjectFileMapCache)(), true);
            exports.currentProjectFileMapCache = projectFileMapCache;
            exports.currentProjectGraph = projectGraph;
            perf_hooks_1.performance.mark('create-project-graph-end');
            perf_hooks_1.performance.measure('total execution time for createProjectGraph()', 'create-project-graph-start', 'create-project-graph-end');
            perf_hooks_1.performance.mark('json-stringify-start');
            const serializedProjectGraph = JSON.stringify(projectGraph);
            perf_hooks_1.performance.mark('json-stringify-end');
            perf_hooks_1.performance.measure('serialize graph', 'json-stringify-start', 'json-stringify-end');
            return {
                error: null,
                projectGraph,
                projectFileMap,
                allWorkspaceFiles,
                serializedProjectGraph,
            };
        }
        catch (e) {
            logger_1.serverLogger.log(`Error detected when creating a project graph: ${e.message}`);
            return {
                error: e,
                projectGraph: null,
                projectFileMap: null,
                allWorkspaceFiles: null,
                serializedProjectGraph: null,
            };
        }
    });
}
function resetInternalState() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        cachedSerializedProjectGraphPromise = undefined;
        exports.projectFileMapWithFiles = undefined;
        exports.currentProjectFileMapCache = undefined;
        exports.currentProjectGraph = undefined;
        collectedUpdatedFiles.clear();
        collectedDeletedFiles.clear();
        file_hasher_1.fileHasher.clear();
        yield file_hasher_1.fileHasher.ensureInitialized();
        waitPeriod = 100;
    });
}
function resetInternalStateIfNxDepsMissing() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            if (!(0, fileutils_1.fileExists)(nx_deps_cache_1.nxProjectGraph) && cachedSerializedProjectGraphPromise) {
                yield resetInternalState();
            }
        }
        catch (e) {
            yield resetInternalState();
        }
    });
}
