"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.convertChangeEventsToLogMessage = exports.subscribeToServerProcessJsonChanges = exports.subscribeToWorkspaceChanges = exports.watchOutputFiles = exports.watchWorkspace = exports.subscribeToOutputsChanges = void 0;
const tslib_1 = require("tslib");
/**
 * In addition to its native performance, another great advantage of `@parcel/watcher` is that it will
 * automatically take advantage of Facebook's watchman tool (https://facebook.github.io/watchman/) if
 * the user has it installed (but not require it if they don't).
 *
 * See https://github.com/parcel-bundler/watcher for more details.
 */
const workspace_root_1 = require("../../utils/workspace-root");
const path_1 = require("path");
const socket_utils_1 = require("../socket-utils");
const shutdown_utils_1 = require("./shutdown-utils");
const path_2 = require("../../utils/path");
const ignore_1 = require("../../utils/ignore");
const os_1 = require("os");
const cache_1 = require("../cache");
const ALWAYS_IGNORE = [...(0, ignore_1.getAlwaysIgnore)(workspace_root_1.workspaceRoot), socket_utils_1.FULL_OS_SOCKET_PATH];
function subscribeToOutputsChanges(cb) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const watcher = yield Promise.resolve().then(() => require('@parcel/watcher'));
        return yield watcher.subscribe(workspace_root_1.workspaceRoot, (err, events) => {
            if (err) {
                return cb(err, null);
            }
            else {
                const workspaceRelativeEvents = [];
                for (const event of events) {
                    const workspaceRelativeEvent = {
                        type: event.type,
                        path: (0, path_2.normalizePath)((0, path_1.relative)(workspace_root_1.workspaceRoot, event.path)),
                    };
                    workspaceRelativeEvents.push(workspaceRelativeEvent);
                }
                cb(null, workspaceRelativeEvents);
            }
        }, watcherOptions([...ALWAYS_IGNORE]));
    });
}
exports.subscribeToOutputsChanges = subscribeToOutputsChanges;
function watchWorkspace(server, cb) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const { Watcher } = yield Promise.resolve().then(() => require('../../native'));
        let relativeServerProcess = (0, path_2.normalizePath)((0, path_1.relative)(workspace_root_1.workspaceRoot, cache_1.serverProcessJsonPath));
        let watcher = new Watcher(workspace_root_1.workspaceRoot, [`!${relativeServerProcess}`]);
        watcher.watch((err, events) => {
            if (err) {
                return cb(err, null);
            }
            for (const event of events) {
                if (event.path == relativeServerProcess &&
                    (0, cache_1.getDaemonProcessIdSync)() !== process.pid) {
                    (0, shutdown_utils_1.handleServerProcessTermination)({
                        server,
                        reason: 'this process is no longer the current daemon (native)',
                    });
                }
                if (event.path.endsWith('.gitignore') || event.path === '.nxignore') {
                    // If the ignore files themselves have changed we need to dynamically update our cached ignoreGlobs
                    (0, shutdown_utils_1.handleServerProcessTermination)({
                        server,
                        reason: 'Stopping the daemon the set of ignored files changed (native)',
                    });
                }
            }
            cb(null, events);
        });
        return watcher;
    });
}
exports.watchWorkspace = watchWorkspace;
function watchOutputFiles(cb) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const { Watcher } = yield Promise.resolve().then(() => require('../../native'));
        let watcher = new Watcher(workspace_root_1.workspaceRoot, null, false);
        watcher.watch((err, events) => {
            if (err) {
                return cb(err, null);
            }
            events = events.filter((event) => {
                return (!event.path.startsWith('.git') && !event.path.includes('node_modules'));
            });
            if (events.length !== 0) {
                cb(null, events);
            }
        });
        return watcher;
    });
}
exports.watchOutputFiles = watchOutputFiles;
function subscribeToWorkspaceChanges(server, cb) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        /**
         * The imports and exports of @nx/workspace are somewhat messy and far reaching across the repo (and beyond),
         * and so it is much safer for us to lazily load here `@parcel/watcher` so that its inclusion is not inadvertently
         * executed by packages which do not have its necessary native binaries available.
         */
        const watcher = yield Promise.resolve().then(() => require('@parcel/watcher'));
        const ignoreObj = (0, ignore_1.getIgnoreObject)();
        return yield watcher.subscribe(workspace_root_1.workspaceRoot, (err, events) => {
            if (err) {
                return cb(err, null);
            }
            let hasIgnoreFileUpdate = false;
            // Most of our utilities (ignore, hashing etc) require unix-style workspace relative paths
            const workspaceRelativeEvents = [];
            for (const event of events) {
                const workspaceRelativeEvent = {
                    type: event.type,
                    path: (0, path_2.normalizePath)((0, path_1.relative)(workspace_root_1.workspaceRoot, event.path)),
                };
                if (workspaceRelativeEvent.path.endsWith('.gitignore') ||
                    workspaceRelativeEvent.path === '.nxignore') {
                    hasIgnoreFileUpdate = true;
                }
                workspaceRelativeEvents.push(workspaceRelativeEvent);
            }
            // If the ignore files themselves have changed we need to dynamically update our cached ignoreGlobs
            if (hasIgnoreFileUpdate) {
                (0, shutdown_utils_1.handleServerProcessTermination)({
                    server,
                    reason: 'Stopping the daemon the set of ignored files changed.',
                });
            }
            const nonIgnoredEvents = workspaceRelativeEvents
                .filter(({ path }) => !!path)
                .filter(({ path }) => !ignoreObj.ignores(path));
            if (nonIgnoredEvents && nonIgnoredEvents.length > 0) {
                cb(null, nonIgnoredEvents);
            }
        }, watcherOptions((0, ignore_1.getIgnoredGlobs)(workspace_root_1.workspaceRoot)));
    });
}
exports.subscribeToWorkspaceChanges = subscribeToWorkspaceChanges;
// TODO: When we update @parcel/watcher to a version that handles negation globs, then this can be folded into the workspace watcher
function subscribeToServerProcessJsonChanges(cb) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const watcher = yield Promise.resolve().then(() => require('@parcel/watcher'));
        return yield watcher.subscribe((0, path_1.dirname)(cache_1.serverProcessJsonPath), (err, events) => {
            for (const event of events) {
                if (event.path === cache_1.serverProcessJsonPath) {
                    cb();
                }
            }
        }, watcherOptions([]));
    });
}
exports.subscribeToServerProcessJsonChanges = subscribeToServerProcessJsonChanges;
/**
 * NOTE: An event type of "create" will also apply to the case where the user has restored
 * an original version of a file after modifying/deleting it by using git, so we adjust
 * our log language accordingly.
 */
function convertChangeEventsToLogMessage(changeEvents) {
    // If only a single file was changed, show the information inline
    if (changeEvents.length === 1) {
        const { path, type } = changeEvents[0];
        let typeLog = 'updated';
        switch (type) {
            case 'create':
                typeLog = 'created or restored';
                break;
            case 'update':
                typeLog = 'modified';
                break;
            case 'delete':
                typeLog = 'deleted';
                break;
        }
        return `${path} was ${typeLog}`;
    }
    let numCreatedOrRestoredFiles = 0;
    let numModifiedFiles = 0;
    let numDeletedFiles = 0;
    for (const event of changeEvents) {
        switch (event.type) {
            case 'create':
                numCreatedOrRestoredFiles++;
                break;
            case 'update':
                numModifiedFiles++;
                break;
            case 'delete':
                numDeletedFiles++;
                break;
        }
    }
    return `${numCreatedOrRestoredFiles} file(s) created or restored, ${numModifiedFiles} file(s) modified, ${numDeletedFiles} file(s) deleted`;
}
exports.convertChangeEventsToLogMessage = convertChangeEventsToLogMessage;
function watcherOptions(ignore) {
    const options = {
        ignore,
    };
    if ((0, os_1.platform)() === 'win32') {
        options.backend = 'windows';
    }
    return options;
}
