"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getRelativeProjectJsonSchemaPath = exports.getProjects = exports.readProjectConfiguration = exports.removeProjectConfiguration = exports.updateProjectConfiguration = exports.addProjectConfiguration = exports.getWorkspacePath = exports.isStandaloneProject = exports.updateWorkspaceConfiguration = exports.readWorkspaceConfiguration = exports.updateNxJson = exports.readNxJson = void 0;
const path_1 = require("path");
const workspaces_1 = require("../../config/workspaces");
const path_2 = require("../../utils/path");
const json_1 = require("./json");
const nx_json_1 = require("./nx-json");
const output_1 = require("../../utils/output");
const installation_directory_1 = require("../../utils/installation-directory");
var nx_json_2 = require("./nx-json");
Object.defineProperty(exports, "readNxJson", { enumerable: true, get: function () { return nx_json_2.readNxJson; } });
Object.defineProperty(exports, "updateNxJson", { enumerable: true, get: function () { return nx_json_2.updateNxJson; } });
var deprecated_1 = require("./deprecated");
Object.defineProperty(exports, "readWorkspaceConfiguration", { enumerable: true, get: function () { return deprecated_1.readWorkspaceConfiguration; } });
Object.defineProperty(exports, "updateWorkspaceConfiguration", { enumerable: true, get: function () { return deprecated_1.updateWorkspaceConfiguration; } });
Object.defineProperty(exports, "isStandaloneProject", { enumerable: true, get: function () { return deprecated_1.isStandaloneProject; } });
Object.defineProperty(exports, "getWorkspacePath", { enumerable: true, get: function () { return deprecated_1.getWorkspacePath; } });
/**
 * Adds project configuration to the Nx workspace.
 *
 * @param tree - the file system tree
 * @param projectName - unique name. Often directories are part of the name (e.g., mydir-mylib)
 * @param projectConfiguration - project configuration
 * @param standalone - whether the project is configured in workspace.json or not
 */
function addProjectConfiguration(tree, projectName, projectConfiguration, standalone = true) {
    const projectConfigFile = (0, path_2.joinPathFragments)(projectConfiguration.root, 'project.json');
    if (!standalone) {
        output_1.output.warn({
            title: 'Nx only supports standalone projects. Setting standalone to false is ignored.',
        });
    }
    if (tree.exists(projectConfigFile)) {
        throw new Error(`Cannot create a new project ${projectName} at ${projectConfiguration.root}. It already exists.`);
    }
    delete projectConfiguration.$schema;
    (0, json_1.writeJson)(tree, projectConfigFile, Object.assign(Object.assign({ name: projectName, $schema: getRelativeProjectJsonSchemaPath(tree, projectConfiguration) }, projectConfiguration), { root: undefined }));
}
exports.addProjectConfiguration = addProjectConfiguration;
/**
 * Updates the configuration of an existing project.
 *
 * @param tree - the file system tree
 * @param projectName - unique name. Often directories are part of the name (e.g., mydir-mylib)
 * @param projectConfiguration - project configuration
 */
function updateProjectConfiguration(tree, projectName, projectConfiguration) {
    var _a;
    const projectConfigFile = (0, path_2.joinPathFragments)(projectConfiguration.root, 'project.json');
    if (!tree.exists(projectConfigFile)) {
        throw new Error(`Cannot update Project ${projectName} at ${projectConfiguration.root}. It doesn't exist or uses package.json configuration.`);
    }
    (0, json_1.writeJson)(tree, projectConfigFile, Object.assign(Object.assign({ name: (_a = projectConfiguration.name) !== null && _a !== void 0 ? _a : projectName, $schema: getRelativeProjectJsonSchemaPath(tree, projectConfiguration) }, projectConfiguration), { root: undefined }));
}
exports.updateProjectConfiguration = updateProjectConfiguration;
/**
 * Removes the configuration of an existing project.
 *
 * @param tree - the file system tree
 * @param projectName - unique name. Often directories are part of the name (e.g., mydir-mylib)
 */
function removeProjectConfiguration(tree, projectName) {
    const projectConfiguration = readProjectConfiguration(tree, projectName);
    if (!projectConfiguration) {
        throw new Error(`Cannot delete Project ${projectName}`);
    }
    const projectConfigFile = (0, path_2.joinPathFragments)(projectConfiguration.root, 'project.json');
    if (tree.exists(projectConfigFile)) {
        tree.delete(projectConfigFile);
    }
}
exports.removeProjectConfiguration = removeProjectConfiguration;
/**
 * Reads a project configuration.
 *
 * @param tree - the file system tree
 * @param projectName - unique name. Often directories are part of the name (e.g., mydir-mylib)
 * @throws If supplied projectName cannot be found
 */
function readProjectConfiguration(tree, projectName) {
    const allProjects = readAndCombineAllProjectConfigurations(tree);
    if (!allProjects[projectName]) {
        // temporary polyfill to make sure our generators work for existing angularcli workspaces
        if (tree.exists('angular.json')) {
            const angularJson = toNewFormat((0, json_1.readJson)(tree, 'angular.json'));
            if (angularJson.projects[projectName])
                return angularJson.projects[projectName];
        }
        throw new Error(`Cannot find configuration for '${projectName}'`);
    }
    return allProjects[projectName];
}
exports.readProjectConfiguration = readProjectConfiguration;
/**
 * Get a map of all projects in a workspace.
 *
 * Use {@link readProjectConfiguration} if only one project is needed.
 */
function getProjects(tree) {
    let allProjects = readAndCombineAllProjectConfigurations(tree);
    // temporary polyfill to make sure our generators work for existing angularcli workspaces
    if (tree.exists('angular.json')) {
        const angularJson = toNewFormat((0, json_1.readJson)(tree, 'angular.json'));
        allProjects = Object.assign(Object.assign({}, allProjects), angularJson.projects);
    }
    return new Map(Object.keys(allProjects || {}).map((projectName) => {
        return [projectName, allProjects[projectName]];
    }));
}
exports.getProjects = getProjects;
function getRelativeProjectJsonSchemaPath(tree, project) {
    return (0, path_2.normalizePath)((0, path_1.relative)((0, path_1.join)(tree.root, project.root), (0, path_1.join)(tree.root, 'node_modules/nx/schemas/project-schema.json')));
}
exports.getRelativeProjectJsonSchemaPath = getRelativeProjectJsonSchemaPath;
function readAndCombineAllProjectConfigurations(tree) {
    const nxJson = (0, nx_json_1.readNxJson)(tree);
    const globbedFiles = (0, workspaces_1.globForProjectFiles)(tree.root, (0, workspaces_1.getGlobPatternsFromPlugins)(nxJson, (0, installation_directory_1.getNxRequirePaths)(tree.root), tree.root), nxJson).map(path_2.normalizePath);
    const createdFiles = findCreatedProjectFiles(tree);
    const deletedFiles = findDeletedProjectFiles(tree);
    const projectFiles = [...globbedFiles, ...createdFiles].filter((r) => deletedFiles.indexOf(r) === -1);
    return (0, workspaces_1.buildProjectsConfigurationsFromProjectPaths)(nxJson, projectFiles, (file) => (0, json_1.readJson)(tree, file)).projects;
}
/**
 * Used to ensure that projects created during
 * the same devkit generator run show up when
 * there is no project.json file, as `glob`
 * cannot find them.
 *
 * We exclude the root `package.json` from this list unless
 * considered a project during workspace generation
 */
function findCreatedProjectFiles(tree) {
    const createdProjectFiles = [];
    for (const change of tree.listChanges()) {
        if (change.type === 'CREATE') {
            const fileName = (0, path_1.basename)(change.path);
            // all created project json files are created projects
            if (fileName === 'project.json') {
                createdProjectFiles.push(change.path);
            }
            else if (fileName === 'package.json') {
                try {
                    const contents = JSON.parse(change.content.toString());
                    if (contents.nx) {
                        createdProjectFiles.push(change.path);
                    }
                }
                catch (_a) { }
            }
        }
    }
    return (0, workspaces_1.deduplicateProjectFiles)(createdProjectFiles).map(path_2.normalizePath);
}
/**
 * Used to ensure that projects created during
 * the same devkit generator run show up when
 * there is no project.json file, as `glob`
 * cannot find them.
 */
function findDeletedProjectFiles(tree) {
    return tree
        .listChanges()
        .filter((f) => {
        const fileName = (0, path_1.basename)(f.path);
        return (f.type === 'DELETE' &&
            (fileName === 'project.json' || fileName === 'package.json'));
    })
        .map((r) => r.path);
}
function toNewFormat(w) {
    const projects = {};
    Object.keys(w.projects || {}).forEach((name) => {
        if (typeof w.projects[name] === 'string')
            return;
        const projectConfig = w.projects[name];
        if (projectConfig.architect) {
            (0, workspaces_1.renamePropertyWithStableKeys)(projectConfig, 'architect', 'targets');
        }
        if (projectConfig.schematics) {
            (0, workspaces_1.renamePropertyWithStableKeys)(projectConfig, 'schematics', 'generators');
        }
        Object.values(projectConfig.targets || {}).forEach((target) => {
            if (target.builder !== undefined) {
                (0, workspaces_1.renamePropertyWithStableKeys)(target, 'builder', 'executor');
            }
        });
        projects[name] = projectConfig;
    });
    w.projects = projects;
    if (w.schematics) {
        (0, workspaces_1.renamePropertyWithStableKeys)(w, 'schematics', 'generators');
    }
    if (w.version !== 2) {
        w.version = 2;
    }
    return w;
}
