"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.filterUsingGlobPatterns = exports.expandNamedInput = exports.getInputs = exports.extractPatternsFromFileSets = exports.getTargetInputs = exports.getNamedInputs = exports.InProcessTaskHasher = exports.DaemonBasedTaskHasher = void 0;
const tslib_1 = require("tslib");
const child_process_1 = require("child_process");
const minimatch = require("minimatch");
const hasher_1 = require("../plugins/js/hasher/hasher");
const find_project_for_path_1 = require("../project-graph/utils/find-project-for-path");
const find_matching_projects_1 = require("../utils/find-matching-projects");
const file_hasher_1 = require("./file-hasher");
const utils_1 = require("../tasks-runner/utils");
const path_1 = require("path");
const set_hash_env_1 = require("./set-hash-env");
class DaemonBasedTaskHasher {
    constructor(daemonClient, taskGraph, runnerOptions) {
        this.daemonClient = daemonClient;
        this.taskGraph = taskGraph;
        this.runnerOptions = runnerOptions;
    }
    hashTasks(tasks) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return this.daemonClient.hashTasks(this.runnerOptions, tasks, this.taskGraph);
        });
    }
    hashTask(task) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return (yield this.daemonClient.hashTasks(this.runnerOptions, [task], this.taskGraph))[0];
        });
    }
}
exports.DaemonBasedTaskHasher = DaemonBasedTaskHasher;
class InProcessTaskHasher {
    constructor(projectFileMap, allWorkspaceFiles, projectGraph, taskGraph, nxJson, options, fileHasher) {
        var _a;
        this.projectFileMap = projectFileMap;
        this.allWorkspaceFiles = allWorkspaceFiles;
        this.projectGraph = projectGraph;
        this.taskGraph = taskGraph;
        this.nxJson = nxJson;
        this.options = options;
        this.fileHasher = fileHasher;
        const legacyRuntimeInputs = (this.options && this.options.runtimeCacheInputs
            ? this.options.runtimeCacheInputs
            : []).map((r) => ({ runtime: r }));
        if (process.env.NX_CLOUD_ENCRYPTION_KEY) {
            legacyRuntimeInputs.push({ env: 'NX_CLOUD_ENCRYPTION_KEY' });
        }
        const legacyFilesetInputs = [
            'nx.json',
            // ignore files will change the set of inputs to the hasher
            '.gitignore',
            '.nxignore',
        ].map((d) => ({ fileset: `{workspaceRoot}/${d}` }));
        this.taskHasher = new TaskHasherImpl(nxJson, legacyRuntimeInputs, legacyFilesetInputs, this.projectFileMap, this.allWorkspaceFiles, this.projectGraph, this.taskGraph, this.fileHasher, { selectivelyHashTsConfig: (_a = this.options.selectivelyHashTsConfig) !== null && _a !== void 0 ? _a : false });
    }
    hashTasks(tasks) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return yield Promise.all(tasks.map((t) => this.hashTask(t)));
        });
    }
    hashTask(task) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const res = yield this.taskHasher.hashTask(task, [task.target.project]);
            const command = this.hashCommand(task);
            return {
                value: (0, file_hasher_1.hashArray)([res.value, command]),
                details: {
                    command,
                    nodes: res.details,
                    implicitDeps: {},
                    runtime: {},
                },
            };
        });
    }
    hashCommand(task) {
        var _a, _b, _c;
        const overrides = Object.assign({}, task.overrides);
        delete overrides['__overrides_unparsed__'];
        const sortedOverrides = {};
        for (let k of Object.keys(overrides).sort()) {
            sortedOverrides[k] = overrides[k];
        }
        return (0, file_hasher_1.hashArray)([
            (_a = task.target.project) !== null && _a !== void 0 ? _a : '',
            (_b = task.target.target) !== null && _b !== void 0 ? _b : '',
            (_c = task.target.configuration) !== null && _c !== void 0 ? _c : '',
            JSON.stringify(sortedOverrides),
        ]);
    }
}
exports.InProcessTaskHasher = InProcessTaskHasher;
InProcessTaskHasher.version = '3.0';
const DEFAULT_INPUTS = [
    {
        fileset: '{projectRoot}/**/*',
    },
    {
        dependencies: true,
        input: 'default',
    },
];
class TaskHasherImpl {
    constructor(nxJson, legacyRuntimeInputs, legacyFilesetInputs, projectFileMap, allWorkspaceFiles, projectGraph, taskGraph, fileHasher, options) {
        this.nxJson = nxJson;
        this.legacyRuntimeInputs = legacyRuntimeInputs;
        this.legacyFilesetInputs = legacyFilesetInputs;
        this.projectFileMap = projectFileMap;
        this.allWorkspaceFiles = allWorkspaceFiles;
        this.projectGraph = projectGraph;
        this.taskGraph = taskGraph;
        this.fileHasher = fileHasher;
        this.options = options;
        this.filesetHashes = {};
        this.runtimeHashes = {};
        this.externalDepsHashCache = {};
        this.projectRootMappings = (0, find_project_for_path_1.createProjectRootMappings)(this.projectGraph.nodes);
    }
    hashTask(task, visited) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return Promise.resolve().then(() => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const { selfInputs, depsInputs, depsOutputs, projectInputs } = getInputs(task, this.projectGraph, this.nxJson);
                const selfAndInputs = yield this.hashSelfAndDepsInputs(task.target.project, task, selfInputs, depsInputs, depsOutputs, projectInputs, visited);
                const target = this.hashTarget(task.target.project, task.target.target, selfInputs);
                if (target) {
                    return {
                        value: (0, file_hasher_1.hashArray)([selfAndInputs.value, target.value]),
                        details: Object.assign(Object.assign({}, selfAndInputs.details), target.details),
                    };
                }
                return selfAndInputs;
            }));
        });
    }
    hashNamedInputForDependencies(projectName, task, namedInput, visited) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const projectNode = this.projectGraph.nodes[projectName];
            const namedInputs = Object.assign(Object.assign({ default: [{ fileset: '{projectRoot}/**/*' }] }, this.nxJson.namedInputs), projectNode.data.namedInputs);
            const expandedInputs = expandNamedInput(namedInput, namedInputs);
            const selfInputs = expandedInputs.filter(isSelfInput);
            const depsOutputs = expandedInputs.filter(isDepsOutput);
            const depsInputs = [{ input: namedInput, dependencies: true }]; // true is boolean by default
            return this.hashSelfAndDepsInputs(projectName, task, selfInputs, depsInputs, depsOutputs, [], visited);
        });
    }
    hashSelfAndDepsInputs(projectName, task, selfInputs, depsInputs, depsOutputs, projectInputs, visited) {
        var _a;
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const projectGraphDeps = (_a = this.projectGraph.dependencies[projectName]) !== null && _a !== void 0 ? _a : [];
            // we don't want random order of dependencies to change the hash
            projectGraphDeps.sort((a, b) => a.target.localeCompare(b.target));
            const self = yield this.hashSingleProjectInputs(projectName, selfInputs);
            const deps = yield this.hashDepsInputs(task, depsInputs, projectGraphDeps, visited);
            const depsOut = this.hashDepsOutputs(task, depsOutputs);
            const projects = yield this.hashProjectInputs(projectInputs, visited);
            let details = {};
            for (const s of self) {
                details = Object.assign(Object.assign({}, details), s.details);
            }
            for (const s of deps) {
                details = Object.assign(Object.assign({}, details), s.details);
            }
            for (const s of projects) {
                details = Object.assign(Object.assign({}, details), s.details);
            }
            for (const s of depsOut) {
                details = Object.assign(Object.assign({}, details), s.details);
            }
            const value = (0, file_hasher_1.hashArray)([
                ...self.map((d) => d.value),
                ...deps.map((d) => d.value),
                ...depsOut.map((d) => d.value),
                ...projects.map((d) => d.value),
            ]);
            return { value, details };
        });
    }
    hashDepsInputs(task, inputs, projectGraphDeps, visited) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return (yield Promise.all(inputs.map((input) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                return yield Promise.all(projectGraphDeps.map((d) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (visited.indexOf(d.target) > -1) {
                        return null;
                    }
                    else {
                        visited.push(d.target);
                        if (this.projectGraph.nodes[d.target]) {
                            return yield this.hashNamedInputForDependencies(d.target, task, input.input || 'default', visited);
                        }
                        else {
                            const hash = this.hashExternalDependency(d.source, d.target);
                            return {
                                value: hash,
                                details: {
                                    [d.target]: hash,
                                },
                            };
                        }
                    }
                })));
            }))))
                .flat()
                .filter((r) => !!r);
        });
    }
    hashDepsOutputs(task, depsOutputs) {
        if (depsOutputs.length === 0) {
            return [];
        }
        const result = [];
        for (const { dependentTasksOutputFiles, transitive } of depsOutputs) {
            result.push(...this.hashDepOuputs(task, dependentTasksOutputFiles, transitive));
        }
        return result;
    }
    hashDepOuputs(task, dependentTasksOutputFiles, transitive) {
        // task has no dependencies
        if (!this.taskGraph.dependencies[task.id]) {
            return [];
        }
        const partialHashes = [];
        for (const d of this.taskGraph.dependencies[task.id]) {
            const childTask = this.taskGraph.tasks[d];
            const outputDirs = (0, utils_1.getOutputsForTargetAndConfiguration)(childTask, this.projectGraph.nodes[childTask.target.project]);
            const hashes = {};
            for (const outputDir of outputDirs) {
                hashes[(0, path_1.join)(outputDir, dependentTasksOutputFiles)] =
                    this.fileHasher.hashFilesMatchingGlobs(outputDir, [
                        dependentTasksOutputFiles,
                    ]);
            }
            partialHashes.push({
                value: (0, file_hasher_1.hashArray)(Object.values(hashes)),
                details: hashes,
            });
            if (transitive) {
                partialHashes.push(...this.hashDepOuputs(childTask, dependentTasksOutputFiles, transitive));
            }
        }
        return partialHashes;
    }
    computeExternalDependencyIdentifier(sourceProjectName, targetProjectName) {
        return `${sourceProjectName}->${targetProjectName}`;
    }
    hashExternalDependency(sourceProjectName, targetProjectName, visited = new Set()) {
        // try to retrieve the hash from cache
        if (this.externalDepsHashCache[targetProjectName]) {
            return this.externalDepsHashCache[targetProjectName];
        }
        visited.add(this.computeExternalDependencyIdentifier(sourceProjectName, targetProjectName));
        const node = this.projectGraph.externalNodes[targetProjectName];
        let partialHash;
        if (node) {
            const partialHashes = [];
            if (node.data.hash) {
                // we already know the hash of this dependency
                partialHashes.push(node.data.hash);
            }
            else {
                // we take version as a hash
                partialHashes.push(node.data.version);
            }
            // we want to calculate the hash of the entire dependency tree
            if (this.projectGraph.dependencies[targetProjectName]) {
                this.projectGraph.dependencies[targetProjectName].forEach((d) => {
                    if (!visited.has(this.computeExternalDependencyIdentifier(targetProjectName, d.target))) {
                        partialHashes.push(this.hashExternalDependency(targetProjectName, d.target, visited));
                    }
                });
            }
            partialHash = (0, file_hasher_1.hashArray)(partialHashes);
        }
        else {
            // unknown dependency
            // this may occur if dependency is not an npm package
            // but rather symlinked in node_modules or it's pointing to a remote git repo
            // in this case we have no information about the versioning of the given package
            partialHash = `__${targetProjectName}__`;
        }
        this.externalDepsHashCache[targetProjectName] = partialHash;
        return partialHash;
    }
    hashTarget(projectName, targetName, selfInputs) {
        const projectNode = this.projectGraph.nodes[projectName];
        const target = projectNode.data.targets[targetName];
        if (!target) {
            return;
        }
        let hash;
        // we can only vouch for @nx packages's executor dependencies
        // if it's "run commands" or third-party we skip traversing since we have no info what this command depends on
        if (target.executor.startsWith(`@nrwl/`) ||
            target.executor.startsWith(`@nx/`)) {
            const executorPackage = target.executor.split(':')[0];
            const executorNodeName = this.findExternalDependencyNodeName(executorPackage);
            hash = this.hashExternalDependency(projectName, executorNodeName);
        }
        else {
            // use command external dependencies if available to construct the hash
            const partialHashes = [];
            let hasCommandExternalDependencies = false;
            for (const input of selfInputs) {
                if (input['externalDependencies']) {
                    // if we have externalDependencies with empty array we still want to override the default hash
                    hasCommandExternalDependencies = true;
                    const externalDependencies = input['externalDependencies'];
                    for (let dep of externalDependencies) {
                        dep = this.findExternalDependencyNodeName(dep);
                        partialHashes.push(this.hashExternalDependency(projectName, dep));
                    }
                }
            }
            if (hasCommandExternalDependencies) {
                hash = (0, file_hasher_1.hashArray)(partialHashes);
            }
            else {
                // cache the hash of the entire external dependencies tree
                if (this.externalDepsHashCache['']) {
                    hash = this.externalDepsHashCache[''];
                }
                else {
                    hash = (0, file_hasher_1.hashArray)([JSON.stringify(this.projectGraph.externalNodes)]);
                    this.externalDepsHashCache[''] = hash;
                }
            }
        }
        return {
            value: hash,
            details: {
                target: target.executor,
            },
        };
    }
    findExternalDependencyNodeName(packageName) {
        if (this.projectGraph.externalNodes[packageName]) {
            return packageName;
        }
        if (this.projectGraph.externalNodes[`npm:${packageName}`]) {
            return `npm:${packageName}`;
        }
        for (const node of Object.values(this.projectGraph.externalNodes)) {
            if (node.data.packageName === packageName) {
                return node.name;
            }
        }
        // not found, just return the package name
        return packageName;
    }
    hashSingleProjectInputs(projectName, inputs) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const filesets = extractPatternsFromFileSets(inputs);
            const projectFilesets = [];
            const workspaceFilesets = [];
            let invalidFilesetNoPrefix = null;
            let invalidFilesetWorkspaceRootNegative = null;
            for (let f of filesets) {
                if (f.startsWith('{projectRoot}/') || f.startsWith('!{projectRoot}/')) {
                    projectFilesets.push(f);
                }
                else if (f.startsWith('{workspaceRoot}/') ||
                    f.startsWith('!{workspaceRoot}/')) {
                    workspaceFilesets.push(f);
                }
                else {
                    invalidFilesetNoPrefix = f;
                }
            }
            if (invalidFilesetNoPrefix) {
                throw new Error([
                    `"${invalidFilesetNoPrefix}" is an invalid fileset.`,
                    'All filesets have to start with either {workspaceRoot} or {projectRoot}.',
                    'For instance: "!{projectRoot}/**/*.spec.ts" or "{workspaceRoot}/package.json".',
                    `If "${invalidFilesetNoPrefix}" is a named input, make sure it is defined in, for instance, nx.json.`,
                ].join('\n'));
            }
            if (invalidFilesetWorkspaceRootNegative) {
                throw new Error([
                    `"${invalidFilesetWorkspaceRootNegative}" is an invalid fileset.`,
                    'It is not possible to negative filesets starting with {workspaceRoot}.',
                ].join('\n'));
            }
            const notFilesets = inputs.filter((r) => !r['fileset']);
            return Promise.all([
                this.hashProjectFileset(projectName, projectFilesets),
                ...[
                    ...workspaceFilesets,
                    ...this.legacyFilesetInputs.map((r) => r.fileset),
                ].map((fileset) => this.hashRootFileset(fileset)),
                ...[...notFilesets, ...this.legacyRuntimeInputs].map((r) => r['runtime'] ? this.hashRuntime(r['runtime']) : this.hashEnv(r['env'])),
            ]);
        });
    }
    hashProjectInputs(projectInputs, visited) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const partialHashes = [];
            for (const input of projectInputs) {
                const projects = (0, find_matching_projects_1.findMatchingProjects)(input.projects, this.projectGraph.nodes);
                for (const project of projects) {
                    const namedInputs = getNamedInputs(this.nxJson, this.projectGraph.nodes[project]);
                    const expandedInput = expandSingleProjectInputs([{ input: input.input }], namedInputs);
                    partialHashes.push(this.hashSingleProjectInputs(project, expandedInput));
                }
            }
            return Promise.all(partialHashes).then((hashes) => hashes.flat());
        });
    }
    hashRootFileset(fileset) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const mapKey = fileset;
            const withoutWorkspaceRoot = fileset.substring(16);
            if (!this.filesetHashes[mapKey]) {
                this.filesetHashes[mapKey] = new Promise((res) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const parts = [];
                    const matchingFile = this.allWorkspaceFiles.find((t) => t.file === withoutWorkspaceRoot);
                    if (matchingFile) {
                        parts.push(matchingFile.hash);
                    }
                    else {
                        this.allWorkspaceFiles
                            .filter((f) => minimatch(f.file, withoutWorkspaceRoot))
                            .forEach((f) => {
                            parts.push(f.hash);
                        });
                    }
                    const value = (0, file_hasher_1.hashArray)(parts);
                    res({
                        value,
                        details: { [mapKey]: value },
                    });
                }));
            }
            return this.filesetHashes[mapKey];
        });
    }
    hashProjectFileset(projectName, filesetPatterns) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const mapKey = `${projectName}:${filesetPatterns.join(',')}`;
            if (!this.filesetHashes[mapKey]) {
                this.filesetHashes[mapKey] = new Promise((res) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const p = this.projectGraph.nodes[projectName];
                    const filteredFiles = filterUsingGlobPatterns(p.data.root, this.projectFileMap[projectName] || [], filesetPatterns);
                    const fileNames = filteredFiles.map((f) => f.file);
                    const values = filteredFiles.map((f) => f.hash);
                    const value = (0, file_hasher_1.hashArray)([
                        ...fileNames,
                        ...values,
                        JSON.stringify(Object.assign(Object.assign({}, p.data), { files: undefined })),
                        (0, hasher_1.hashTsConfig)(p, this.projectRootMappings, this.options),
                    ]);
                    res({
                        value,
                        details: { [mapKey]: value },
                    });
                }));
            }
            return this.filesetHashes[mapKey];
        });
    }
    hashRuntime(runtime) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const mapKey = `runtime:${runtime}`;
            if (!this.runtimeHashes[mapKey]) {
                this.runtimeHashes[mapKey] = new Promise((res, rej) => {
                    (0, child_process_1.exec)(runtime, {
                        windowsHide: true,
                    }, (err, stdout, stderr) => {
                        if (err) {
                            rej(new Error(`Nx failed to execute {runtime: '${runtime}'}. ${err}.`));
                        }
                        else {
                            const value = `${stdout}${stderr}`.trim();
                            res({
                                details: { [`runtime:${runtime}`]: value },
                                value,
                            });
                        }
                    });
                });
            }
            return this.runtimeHashes[mapKey];
        });
    }
    hashEnv(envVarName) {
        var _a;
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let env = (0, set_hash_env_1.getHashEnv)();
            const value = (0, file_hasher_1.hashArray)([(_a = env[envVarName]) !== null && _a !== void 0 ? _a : '']);
            return {
                details: { [`env:${envVarName}`]: value },
                value,
            };
        });
    }
}
function getNamedInputs(nxJson, project) {
    return Object.assign(Object.assign({ default: [{ fileset: '{projectRoot}/**/*' }] }, nxJson.namedInputs), project.data.namedInputs);
}
exports.getNamedInputs = getNamedInputs;
function getTargetInputs(nxJson, projectNode, target) {
    const namedInputs = getNamedInputs(nxJson, projectNode);
    const targetData = projectNode.data.targets[target];
    const targetDefaults = (nxJson.targetDefaults || {})[target];
    const inputs = splitInputsIntoSelfAndDependencies(targetData.inputs || (targetDefaults === null || targetDefaults === void 0 ? void 0 : targetDefaults.inputs) || DEFAULT_INPUTS, namedInputs);
    const selfInputs = extractPatternsFromFileSets(inputs.selfInputs);
    const dependencyInputs = extractPatternsFromFileSets(inputs.depsInputs.map((s) => expandNamedInput(s.input, namedInputs)).flat());
    return { selfInputs, dependencyInputs };
}
exports.getTargetInputs = getTargetInputs;
function extractPatternsFromFileSets(inputs) {
    return inputs
        .filter((c) => !!c['fileset'])
        .map((c) => c['fileset']);
}
exports.extractPatternsFromFileSets = extractPatternsFromFileSets;
function getInputs(task, projectGraph, nxJson) {
    const projectNode = projectGraph.nodes[task.target.project];
    const namedInputs = getNamedInputs(nxJson, projectNode);
    const targetData = projectNode.data.targets[task.target.target];
    const targetDefaults = (nxJson.targetDefaults || {})[task.target.target];
    const { selfInputs, depsInputs, depsOutputs, projectInputs } = splitInputsIntoSelfAndDependencies(targetData.inputs || (targetDefaults === null || targetDefaults === void 0 ? void 0 : targetDefaults.inputs) || DEFAULT_INPUTS, namedInputs);
    return { selfInputs, depsInputs, depsOutputs, projectInputs };
}
exports.getInputs = getInputs;
function splitInputsIntoSelfAndDependencies(inputs, namedInputs) {
    const depsInputs = [];
    const projectInputs = [];
    const selfInputs = [];
    for (const d of inputs) {
        if (typeof d === 'string') {
            if (d.startsWith('^')) {
                depsInputs.push({ input: d.substring(1), dependencies: true });
            }
            else {
                selfInputs.push(d);
            }
        }
        else {
            if (('dependencies' in d && d.dependencies) ||
                // Todo(@AgentEnder): Remove check in v17
                ('projects' in d &&
                    typeof d.projects === 'string' &&
                    d.projects === 'dependencies')) {
                depsInputs.push({
                    input: d.input,
                    dependencies: true,
                });
            }
            else if ('projects' in d &&
                d.projects &&
                // Todo(@AgentEnder): Remove check in v17
                !(d.projects === 'self')) {
                projectInputs.push({
                    input: d.input,
                    projects: Array.isArray(d.projects) ? d.projects : [d.projects],
                });
            }
            else {
                selfInputs.push(d);
            }
        }
    }
    const expandedInputs = expandSingleProjectInputs(selfInputs, namedInputs);
    return {
        depsInputs,
        projectInputs,
        selfInputs: expandedInputs.filter(isSelfInput),
        depsOutputs: expandedInputs.filter(isDepsOutput),
    };
}
function isSelfInput(input) {
    return !('dependentTasksOutputFiles' in input);
}
function isDepsOutput(input) {
    return 'dependentTasksOutputFiles' in input;
}
function expandSingleProjectInputs(inputs, namedInputs) {
    const expanded = [];
    for (const d of inputs) {
        if (typeof d === 'string') {
            if (d.startsWith('^'))
                throw new Error(`namedInputs definitions cannot start with ^`);
            if (namedInputs[d]) {
                expanded.push(...expandNamedInput(d, namedInputs));
            }
            else {
                expanded.push({ fileset: d });
            }
        }
        else {
            if (d.projects || d.dependencies) {
                throw new Error(`namedInputs definitions can only refer to other namedInputs definitions within the same project.`);
            }
            if (d.fileset ||
                d.env ||
                d.runtime ||
                d.externalDependencies ||
                d.dependentTasksOutputFiles) {
                expanded.push(d);
            }
            else {
                expanded.push(...expandNamedInput(d.input, namedInputs));
            }
        }
    }
    return expanded;
}
function expandNamedInput(input, namedInputs) {
    namedInputs || (namedInputs = {});
    if (!namedInputs[input])
        throw new Error(`Input '${input}' is not defined`);
    return expandSingleProjectInputs(namedInputs[input], namedInputs);
}
exports.expandNamedInput = expandNamedInput;
function filterUsingGlobPatterns(root, files, patterns) {
    const filesetWithExpandedProjectRoot = patterns
        .map((f) => f.replace('{projectRoot}', root))
        .map((r) => {
        // handling root level projects that create './' pattern that doesn't work with minimatch
        if (r.startsWith('./'))
            return r.substring(2);
        if (r.startsWith('!./'))
            return '!' + r.substring(3);
        return r;
    });
    const positive = [];
    const negative = [];
    for (const p of filesetWithExpandedProjectRoot) {
        if (p.startsWith('!')) {
            negative.push(p);
        }
        else {
            positive.push(p);
        }
    }
    if (positive.length === 0 && negative.length === 0) {
        return files;
    }
    return files.filter((f) => {
        let matchedPositive = false;
        if (positive.length === 0 ||
            (positive.length === 1 && positive[0] === `${root}/**/*`)) {
            matchedPositive = true;
        }
        else {
            matchedPositive = positive.some((pattern) => minimatch(f.file, pattern));
        }
        if (!matchedPositive)
            return false;
        return negative.every((pattern) => minimatch(f.file, pattern));
    });
}
exports.filterUsingGlobPatterns = filterUsingGlobPatterns;
