"use strict";
/**
 * This is the main API for accessing the lock file functionality.
 * It encapsulates the package manager specific logic and implementation details.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createLockFile = exports.getLockFileName = exports.parseLockFile = exports.lockFileHash = exports.lockFileExists = void 0;
const fs_1 = require("fs");
const path_1 = require("path");
const package_manager_1 = require("../../../utils/package-manager");
const workspace_root_1 = require("../../../utils/workspace-root");
const project_graph_builder_1 = require("../../../project-graph/project-graph-builder");
const file_hasher_1 = require("../../../hasher/file-hasher");
const output_1 = require("../../../utils/output");
const npm_parser_1 = require("./npm-parser");
const pnpm_parser_1 = require("./pnpm-parser");
const yarn_parser_1 = require("./yarn-parser");
const project_graph_pruning_1 = require("./project-graph-pruning");
const package_json_1 = require("./utils/package-json");
const YARN_LOCK_FILE = 'yarn.lock';
const NPM_LOCK_FILE = 'package-lock.json';
const PNPM_LOCK_FILE = 'pnpm-lock.yaml';
const YARN_LOCK_PATH = (0, path_1.join)(workspace_root_1.workspaceRoot, YARN_LOCK_FILE);
const NPM_LOCK_PATH = (0, path_1.join)(workspace_root_1.workspaceRoot, NPM_LOCK_FILE);
const PNPM_LOCK_PATH = (0, path_1.join)(workspace_root_1.workspaceRoot, PNPM_LOCK_FILE);
/**
 * Check if lock file exists
 */
function lockFileExists(packageManager = (0, package_manager_1.detectPackageManager)(workspace_root_1.workspaceRoot)) {
    if (packageManager === 'yarn') {
        return (0, fs_1.existsSync)(YARN_LOCK_PATH);
    }
    if (packageManager === 'pnpm') {
        return (0, fs_1.existsSync)(PNPM_LOCK_PATH);
    }
    if (packageManager === 'npm') {
        return (0, fs_1.existsSync)(NPM_LOCK_PATH);
    }
    throw new Error(`Unknown package manager ${packageManager} or lock file missing`);
}
exports.lockFileExists = lockFileExists;
/**
 * Hashes lock file content
 */
function lockFileHash(packageManager = (0, package_manager_1.detectPackageManager)(workspace_root_1.workspaceRoot)) {
    let content;
    if (packageManager === 'yarn') {
        content = (0, fs_1.readFileSync)(YARN_LOCK_PATH, 'utf8');
    }
    if (packageManager === 'pnpm') {
        content = (0, fs_1.readFileSync)(PNPM_LOCK_PATH, 'utf8');
    }
    if (packageManager === 'npm') {
        content = (0, fs_1.readFileSync)(NPM_LOCK_PATH, 'utf8');
    }
    if (content) {
        return (0, file_hasher_1.hashArray)([content]);
    }
    else {
        throw new Error(`Unknown package manager ${packageManager} or lock file missing`);
    }
}
exports.lockFileHash = lockFileHash;
/**
 * Parses lock file and maps dependencies and metadata to {@link LockFileGraph}
 */
function parseLockFile(packageManager = (0, package_manager_1.detectPackageManager)(workspace_root_1.workspaceRoot)) {
    const builder = new project_graph_builder_1.ProjectGraphBuilder(null, null);
    try {
        if (packageManager === 'yarn') {
            const content = (0, fs_1.readFileSync)(YARN_LOCK_PATH, 'utf8');
            (0, yarn_parser_1.parseYarnLockfile)(content, builder);
            return builder.getUpdatedProjectGraph();
        }
        if (packageManager === 'pnpm') {
            const content = (0, fs_1.readFileSync)(PNPM_LOCK_PATH, 'utf8');
            (0, pnpm_parser_1.parsePnpmLockfile)(content, builder);
            return builder.getUpdatedProjectGraph();
        }
        if (packageManager === 'npm') {
            const content = (0, fs_1.readFileSync)(NPM_LOCK_PATH, 'utf8');
            (0, npm_parser_1.parseNpmLockfile)(content, builder);
            return builder.getUpdatedProjectGraph();
        }
    }
    catch (e) {
        if (!isPostInstallProcess()) {
            output_1.output.error({
                title: `Failed to parse ${packageManager} lockfile`,
                bodyLines: errorBodyLines(e),
            });
        }
        return;
    }
    throw new Error(`Unknown package manager: ${packageManager}`);
}
exports.parseLockFile = parseLockFile;
/**
 * Returns lock file name based on the detected package manager in the root
 * @param packageManager
 * @returns
 */
function getLockFileName(packageManager = (0, package_manager_1.detectPackageManager)(workspace_root_1.workspaceRoot)) {
    if (packageManager === 'yarn') {
        return YARN_LOCK_FILE;
    }
    if (packageManager === 'pnpm') {
        return PNPM_LOCK_FILE;
    }
    if (packageManager === 'npm') {
        return NPM_LOCK_FILE;
    }
    throw new Error(`Unknown package manager: ${packageManager}`);
}
exports.getLockFileName = getLockFileName;
/**
 * Create lock file based on the root level lock file and (pruned) package.json
 *
 * @param packageJson
 * @param isProduction
 * @param packageManager
 * @returns
 */
function createLockFile(packageJson, packageManager = (0, package_manager_1.detectPackageManager)(workspace_root_1.workspaceRoot)) {
    const normalizedPackageJson = (0, package_json_1.normalizePackageJson)(packageJson);
    const content = (0, fs_1.readFileSync)(getLockFileName(packageManager), 'utf8');
    const builder = new project_graph_builder_1.ProjectGraphBuilder();
    try {
        if (packageManager === 'yarn') {
            (0, yarn_parser_1.parseYarnLockfile)(content, builder);
            const graph = builder.getUpdatedProjectGraph();
            const prunedGraph = (0, project_graph_pruning_1.pruneProjectGraph)(graph, packageJson);
            return (0, yarn_parser_1.stringifyYarnLockfile)(prunedGraph, content, normalizedPackageJson);
        }
        if (packageManager === 'pnpm') {
            (0, pnpm_parser_1.parsePnpmLockfile)(content, builder);
            const graph = builder.getUpdatedProjectGraph();
            const prunedGraph = (0, project_graph_pruning_1.pruneProjectGraph)(graph, packageJson);
            return (0, pnpm_parser_1.stringifyPnpmLockfile)(prunedGraph, content, normalizedPackageJson);
        }
        if (packageManager === 'npm') {
            (0, npm_parser_1.parseNpmLockfile)(content, builder);
            const graph = builder.getUpdatedProjectGraph();
            const prunedGraph = (0, project_graph_pruning_1.pruneProjectGraph)(graph, packageJson);
            return (0, npm_parser_1.stringifyNpmLockfile)(prunedGraph, content, normalizedPackageJson);
        }
    }
    catch (e) {
        if (!isPostInstallProcess()) {
            const additionalInfo = [
                'To prevent the build from breaking we are returning the root lock file.',
            ];
            if (packageManager === 'npm') {
                additionalInfo.push('If you run `npm install --package-lock-only` in your output folder it will regenerate the correct pruned lockfile.');
            }
            if (packageManager === 'pnpm') {
                additionalInfo.push('If you run `pnpm install --lockfile-only` in your output folder it will regenerate the correct pruned lockfile.');
            }
            output_1.output.error({
                title: 'An error occured while creating pruned lockfile',
                bodyLines: errorBodyLines(e, additionalInfo),
            });
        }
        return content;
    }
}
exports.createLockFile = createLockFile;
// generate body lines for error message
function errorBodyLines(originalError, additionalInfo = []) {
    return [
        'Please open an issue at `https://github.com/nrwl/nx/issues/new?template=1-bug.yml` and provide a reproduction.',
        ...additionalInfo,
        `\nOriginal error: ${originalError.message}\n\n`,
        originalError.stack,
    ];
}
function isPostInstallProcess() {
    return (process.env.npm_command === 'install' &&
        process.env.npm_lifecycle_event === 'postinstall');
}
