"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.stringifyNpmLockfile = exports.parseNpmLockfile = void 0;
const tslib_1 = require("tslib");
const fs_1 = require("fs");
const semver_1 = require("semver");
const workspace_root_1 = require("../../../utils/workspace-root");
const operators_1 = require("../../../project-graph/operators");
const file_hasher_1 = require("../../../hasher/file-hasher");
function parseNpmLockfile(lockFileContent, builder) {
    const data = JSON.parse(lockFileContent);
    // we use key => node map to avoid duplicate work when parsing keys
    const keyMap = new Map();
    addNodes(data, builder, keyMap);
    addDependencies(data, builder, keyMap);
}
exports.parseNpmLockfile = parseNpmLockfile;
function addNodes(data, builder, keyMap) {
    const nodes = new Map();
    if (data.lockfileVersion > 1) {
        Object.entries(data.packages).forEach(([path, snapshot]) => {
            // skip workspaces packages
            if (path === '' || !path.includes('node_modules') || snapshot.link) {
                return;
            }
            const packageName = path.split('node_modules/').pop();
            const version = findV3Version(snapshot, packageName);
            createNode(packageName, version, path, nodes, keyMap, snapshot);
        });
    }
    else {
        Object.entries(data.dependencies).forEach(([packageName, snapshot]) => {
            var _a;
            // we only care about dependencies of workspace packages
            if ((_a = snapshot.version) === null || _a === void 0 ? void 0 : _a.startsWith('file:')) {
                if (snapshot.dependencies) {
                    Object.entries(snapshot.dependencies).forEach(([depName, depSnapshot]) => {
                        addV1Node(depName, depSnapshot, `${snapshot.version.slice(5)}/node_modules/${depName}`, nodes, keyMap, builder);
                    });
                }
            }
            else {
                addV1Node(packageName, snapshot, `node_modules/${packageName}`, nodes, keyMap, builder);
            }
        });
    }
    // some packages can be both hoisted and nested
    // so we need to run this check once we have all the nodes and paths
    for (const [packageName, versionMap] of nodes.entries()) {
        const hoistedNode = keyMap.get(`node_modules/${packageName}`);
        if (hoistedNode) {
            hoistedNode.name = `npm:${packageName}`;
        }
        versionMap.forEach((node) => {
            builder.addExternalNode(node);
        });
    }
}
function addV1Node(packageName, snapshot, path, nodes, keyMap, builder) {
    createNode(packageName, snapshot.version, path, nodes, keyMap, snapshot);
    // traverse nested dependencies
    if (snapshot.dependencies) {
        Object.entries(snapshot.dependencies).forEach(([depName, depSnapshot]) => {
            addV1Node(depName, depSnapshot, `${path}/node_modules/${depName}`, nodes, keyMap, builder);
        });
    }
}
function createNode(packageName, version, key, nodes, keyMap, snapshot) {
    var _a;
    const existingNode = (_a = nodes.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version);
    if (existingNode) {
        keyMap.set(key, existingNode);
        return;
    }
    const node = {
        type: 'npm',
        name: version ? `npm:${packageName}@${version}` : `npm:${packageName}`,
        data: {
            version,
            packageName,
            hash: snapshot.integrity ||
                (0, file_hasher_1.hashArray)(snapshot.resolved
                    ? [snapshot.resolved]
                    : version
                        ? [packageName, version]
                        : [packageName]),
        },
    };
    keyMap.set(key, node);
    if (!nodes.has(packageName)) {
        nodes.set(packageName, new Map([[version, node]]));
    }
    else {
        nodes.get(packageName).set(version, node);
    }
}
function findV3Version(snapshot, packageName) {
    let version = snapshot.version;
    const resolved = snapshot.resolved;
    // for tarball packages version might not exist or be useless
    if (!version || (resolved && !resolved.includes(version))) {
        version = resolved;
    }
    // for alias packages name is set
    if (snapshot.name && snapshot.name !== packageName) {
        if (version) {
            version = `npm:${snapshot.name}@${version}`;
        }
        else {
            version = `npm:${snapshot.name}`;
        }
    }
    return version;
}
function addDependencies(data, builder, keyMap) {
    if (data.lockfileVersion > 1) {
        Object.entries(data.packages).forEach(([path, snapshot]) => {
            // we are skipping workspaces packages
            if (!keyMap.has(path)) {
                return;
            }
            const sourceName = keyMap.get(path).name;
            [
                snapshot.peerDependencies,
                snapshot.dependencies,
                snapshot.optionalDependencies,
            ].forEach((section) => {
                if (section) {
                    Object.entries(section).forEach(([name, versionRange]) => {
                        const target = findTarget(path, keyMap, name, versionRange);
                        if (target) {
                            builder.addStaticDependency(sourceName, target.name);
                        }
                    });
                }
            });
        });
    }
    else {
        Object.entries(data.dependencies).forEach(([packageName, snapshot]) => {
            addV1NodeDependencies(`node_modules/${packageName}`, snapshot, builder, keyMap);
        });
    }
}
function findTarget(sourcePath, keyMap, targetName, versionRange) {
    if (sourcePath && !sourcePath.endsWith('/')) {
        sourcePath = `${sourcePath}/`;
    }
    const searchPath = `${sourcePath}node_modules/${targetName}`;
    if (keyMap.has(searchPath)) {
        const child = keyMap.get(searchPath);
        if (child.data.version === versionRange ||
            (0, semver_1.satisfies)(child.data.version, versionRange)) {
            return child;
        }
    }
    // the hoisted package did not match, this dependency is missing
    if (!sourcePath) {
        return;
    }
    return findTarget(sourcePath.split('node_modules/').slice(0, -1).join('node_modules/'), keyMap, targetName, versionRange);
}
function addV1NodeDependencies(path, snapshot, builder, keyMap) {
    if (keyMap.has(path) && snapshot.requires) {
        const source = keyMap.get(path).name;
        Object.entries(snapshot.requires).forEach(([name, versionRange]) => {
            const target = findTarget(path, keyMap, name, versionRange);
            if (target) {
                builder.addStaticDependency(source, target.name);
            }
        });
    }
    if (snapshot.dependencies) {
        Object.entries(snapshot.dependencies).forEach(([depName, depSnapshot]) => {
            addV1NodeDependencies(`${path}/node_modules/${depName}`, depSnapshot, builder, keyMap);
        });
    }
    const { peerDependencies } = getPeerDependencies(path);
    if (peerDependencies) {
        const node = keyMap.get(path);
        Object.entries(peerDependencies).forEach(([depName, depSpec]) => {
            var _a;
            if (!((_a = builder.graph.dependencies[node.name]) === null || _a === void 0 ? void 0 : _a.find((d) => d.target === depName))) {
                const target = findTarget(path, keyMap, depName, depSpec);
                if (target) {
                    builder.addStaticDependency(node.name, target.name);
                }
            }
        });
    }
}
function stringifyNpmLockfile(graph, rootLockFileContent, packageJson) {
    const rootLockFile = JSON.parse(rootLockFileContent);
    const { lockfileVersion } = JSON.parse(rootLockFileContent);
    const mappedPackages = mapSnapshots(rootLockFile, graph);
    const output = {
        name: packageJson.name || rootLockFile.name,
        version: packageJson.version || '0.0.1',
        lockfileVersion: rootLockFile.lockfileVersion,
    };
    if (rootLockFile.requires) {
        output.requires = rootLockFile.requires;
    }
    if (lockfileVersion > 1) {
        output.packages = mapV3Snapshots(mappedPackages, packageJson);
    }
    if (lockfileVersion < 3) {
        output.dependencies = mapV1Snapshots(mappedPackages);
    }
    return JSON.stringify(output, null, 2);
}
exports.stringifyNpmLockfile = stringifyNpmLockfile;
function mapV3Snapshots(mappedPackages, packageJson) {
    const output = {};
    output[''] = packageJson;
    mappedPackages.forEach((p) => {
        output[p.path] = p.valueV3;
    });
    return output;
}
function mapV1Snapshots(mappedPackages) {
    const output = {};
    mappedPackages.forEach((p) => {
        getPackageParent(p.path, output)[p.name] = p.valueV1;
    });
    return output;
}
function getPackageParent(path, packages) {
    const segments = path.split(/\/?node_modules\//).slice(1, -1);
    if (!segments.length) {
        return packages;
    }
    let parent = packages[segments.shift()];
    if (!parent.dependencies) {
        parent.dependencies = {};
    }
    while (segments.length) {
        parent = parent.dependencies[segments.shift()];
        if (!parent.dependencies) {
            parent.dependencies = {};
        }
    }
    return parent.dependencies;
}
function mapSnapshots(rootLockFile, graph) {
    const nestedNodes = new Set();
    const visitedNodes = new Map();
    const visitedPaths = new Set();
    const remappedPackages = new Map();
    // add first level children
    Object.values(graph.externalNodes).forEach((node) => {
        if (node.name === `npm:${node.data.packageName}`) {
            const mappedPackage = mapPackage(rootLockFile, node.data.packageName, node.data.version);
            remappedPackages.set(mappedPackage.path, mappedPackage);
            visitedNodes.set(node, new Set([mappedPackage.path]));
            visitedPaths.add(mappedPackage.path);
        }
        else {
            nestedNodes.add(node);
        }
    });
    let remappedPackagesArray;
    if (nestedNodes.size) {
        const invertedGraph = (0, operators_1.reverse)(graph);
        nestMappedPackages(invertedGraph, remappedPackages, nestedNodes, visitedNodes, visitedPaths, rootLockFile);
        // initially we naively map package paths to topParent/../parent/child
        // but some of those should be nested higher up the tree
        remappedPackagesArray = elevateNestedPaths(remappedPackages);
    }
    else {
        remappedPackagesArray = Array.from(remappedPackages.values());
    }
    return remappedPackagesArray.sort((a, b) => a.path.localeCompare(b.path));
}
function mapPackage(rootLockFile, packageName, version, parentPath = '') {
    const lockfileVersion = rootLockFile.lockfileVersion;
    let valueV3, valueV1;
    if (lockfileVersion < 3) {
        valueV1 = findMatchingPackageV1(rootLockFile.dependencies, packageName, version);
    }
    if (lockfileVersion > 1) {
        valueV3 = findMatchingPackageV3(rootLockFile.packages, packageName, version);
    }
    return {
        path: parentPath + `node_modules/${packageName}`,
        name: packageName,
        valueV1,
        valueV3,
    };
}
function nestMappedPackages(invertedGraph, result, nestedNodes, visitedNodes, visitedPaths, rootLockFile) {
    const initialSize = nestedNodes.size;
    if (!initialSize) {
        return;
    }
    nestedNodes.forEach((node) => {
        let unresolvedParents = invertedGraph.dependencies[node.name].length;
        invertedGraph.dependencies[node.name].forEach(({ target }) => {
            const targetNode = invertedGraph.externalNodes[target];
            if (visitedNodes.has(targetNode)) {
                visitedNodes.get(targetNode).forEach((path) => {
                    const mappedPackage = mapPackage(rootLockFile, node.data.packageName, node.data.version, path + '/');
                    result.set(mappedPackage.path, mappedPackage);
                    if (visitedNodes.has(node)) {
                        visitedNodes.get(node).add(mappedPackage.path);
                    }
                    else {
                        visitedNodes.set(node, new Set([mappedPackage.path]));
                    }
                    visitedPaths.add(mappedPackage.path);
                });
                unresolvedParents--;
            }
        });
        if (!unresolvedParents) {
            nestedNodes.delete(node);
        }
    });
    if (initialSize === nestedNodes.size) {
        throw new Error([
            'Following packages could not be mapped to the NPM lockfile:',
            ...Array.from(nestedNodes).map((n) => `- ${n.name}`),
        ].join('\n'));
    }
    else {
        nestMappedPackages(invertedGraph, result, nestedNodes, visitedNodes, visitedPaths, rootLockFile);
    }
}
// sort paths by number of segments and then alphabetically
function sortMappedPackagesPaths(mappedPackages) {
    return Array.from(mappedPackages.keys()).sort((a, b) => {
        const aLength = a.split('/node_modules/').length;
        const bLength = b.split('/node_modules/').length;
        if (aLength > bLength) {
            return 1;
        }
        if (aLength < bLength) {
            return -1;
        }
        return a.localeCompare(b);
    });
}
function elevateNestedPaths(remappedPackages) {
    const result = new Map();
    const sortedPaths = sortMappedPackagesPaths(remappedPackages);
    sortedPaths.forEach((path) => {
        const segments = path.split('/node_modules/');
        const mappedPackage = remappedPackages.get(path);
        // we keep hoisted packages intact
        if (segments.length === 1) {
            result.set(path, mappedPackage);
            return;
        }
        const packageName = segments.pop();
        const getNewPath = (segs) => `${segs.join('/node_modules/')}/node_modules/${packageName}`;
        // check if grandparent has the same package
        const shouldElevate = (segs) => {
            var _a, _b, _c, _d;
            const elevatedPath = getNewPath(segs.slice(0, -1));
            if (result.has(elevatedPath)) {
                const match = result.get(elevatedPath);
                return (((_a = match.valueV1) === null || _a === void 0 ? void 0 : _a.version) === ((_b = mappedPackage.valueV1) === null || _b === void 0 ? void 0 : _b.version) &&
                    ((_c = match.valueV3) === null || _c === void 0 ? void 0 : _c.version) === ((_d = mappedPackage.valueV3) === null || _d === void 0 ? void 0 : _d.version));
            }
            return true;
        };
        while (segments.length > 1 && shouldElevate(segments)) {
            segments.pop();
        }
        const newPath = getNewPath(segments);
        if (path !== newPath) {
            if (!result.has(newPath)) {
                mappedPackage.path = newPath;
                result.set(newPath, mappedPackage);
            }
        }
        else {
            result.set(path, mappedPackage);
        }
    });
    return Array.from(result.values());
}
function findMatchingPackageV3(packages, name, version) {
    for (let _a of Object.entries(packages)) {
        const [key, _b] = _a, { dev, peer } = _b, snapshot = tslib_1.__rest(_b, ["dev", "peer"]);
        if (key.endsWith(`node_modules/${name}`)) {
            if ([
                snapshot.version,
                snapshot.resolved,
                `npm:${snapshot.name}@${snapshot.version}`,
            ].includes(version)) {
                return snapshot;
            }
        }
    }
}
function findMatchingPackageV1(packages, name, version) {
    for (let _a of Object.entries(packages)) {
        const [packageName, _b] = _a, { dev, peer, dependencies } = _b, snapshot = tslib_1.__rest(_b, ["dev", "peer", "dependencies"]);
        if (packageName === name) {
            if (snapshot.version === version) {
                return snapshot;
            }
        }
        if (dependencies) {
            const found = findMatchingPackageV1(dependencies, name, version);
            if (found) {
                return found;
            }
        }
    }
}
// NPM V1 does not track the peer dependencies in the lock file
// so we need to parse them directly from the package.json
function getPeerDependencies(path) {
    const fullPath = `${workspace_root_1.workspaceRoot}/${path}/package.json`;
    if ((0, fs_1.existsSync)(fullPath)) {
        const content = (0, fs_1.readFileSync)(fullPath, 'utf-8');
        const { peerDependencies, peerDependenciesMeta } = JSON.parse(content);
        return Object.assign(Object.assign({}, (peerDependencies && { peerDependencies })), (peerDependenciesMeta && { peerDependenciesMeta }));
    }
    else {
        if (process.env.NX_VERBOSE_LOGGING === 'true') {
            console.warn(`Could not find package.json at "${path}"`);
        }
        return {};
    }
}
