"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractCachedFileData = exports.shouldRecomputeWholeGraph = exports.writeCache = exports.createProjectFileMapCache = exports.readProjectGraphCache = exports.readProjectFileMapCache = exports.ensureCacheDirectory = exports.nxFileMap = exports.nxProjectGraph = void 0;
const fs_1 = require("fs");
const fs_extra_1 = require("fs-extra");
const path_1 = require("path");
const perf_hooks_1 = require("perf_hooks");
const cache_directory_1 = require("../utils/cache-directory");
const fileutils_1 = require("../utils/fileutils");
const versions_1 = require("../utils/versions");
exports.nxProjectGraph = (0, path_1.join)(cache_directory_1.projectGraphCacheDirectory, 'project-graph.json');
exports.nxFileMap = (0, path_1.join)(cache_directory_1.projectGraphCacheDirectory, 'file-map.json');
function ensureCacheDirectory() {
    try {
        if (!(0, fs_1.existsSync)(cache_directory_1.projectGraphCacheDirectory)) {
            (0, fs_extra_1.ensureDirSync)(cache_directory_1.projectGraphCacheDirectory);
        }
    }
    catch (e) {
        /*
         * @jeffbcross: Node JS docs recommend against checking for existence of directory immediately before creating it.
         * Instead, just try to create the directory and handle the error.
         *
         * We ran into race conditions when running scripts concurrently, where multiple scripts were
         * arriving here simultaneously, checking for directory existence, then trying to create the directory simultaneously.
         *
         * In this case, we're creating the directory. If the operation failed, we ensure that the directory
         * exists before continuing (or raise an exception).
         */
        if (!(0, fileutils_1.directoryExists)(cache_directory_1.projectGraphCacheDirectory)) {
            throw new Error(`Failed to create directory: ${cache_directory_1.projectGraphCacheDirectory}`);
        }
    }
}
exports.ensureCacheDirectory = ensureCacheDirectory;
function readProjectFileMapCache() {
    perf_hooks_1.performance.mark('read cache:start');
    ensureCacheDirectory();
    let data = null;
    try {
        if ((0, fileutils_1.fileExists)(exports.nxFileMap)) {
            data = (0, fileutils_1.readJsonFile)(exports.nxFileMap);
        }
    }
    catch (error) {
        console.log(`Error reading '${exports.nxFileMap}'. Continue the process without the cache.`);
        console.log(error);
    }
    perf_hooks_1.performance.mark('read cache:end');
    perf_hooks_1.performance.measure('read cache', 'read cache:start', 'read cache:end');
    return data !== null && data !== void 0 ? data : null;
}
exports.readProjectFileMapCache = readProjectFileMapCache;
function readProjectGraphCache() {
    perf_hooks_1.performance.mark('read project-graph:start');
    ensureCacheDirectory();
    let data = null;
    try {
        if ((0, fileutils_1.fileExists)(exports.nxProjectGraph)) {
            data = (0, fileutils_1.readJsonFile)(exports.nxProjectGraph);
        }
    }
    catch (error) {
        console.log(`Error reading '${exports.nxProjectGraph}'. Continue the process without the cache.`);
        console.log(error);
    }
    perf_hooks_1.performance.mark('read project-graph:end');
    perf_hooks_1.performance.measure('read cache', 'read project-graph:start', 'read project-graph:end');
    return data !== null && data !== void 0 ? data : null;
}
exports.readProjectGraphCache = readProjectGraphCache;
function createProjectFileMapCache(nxJson, packageJsonDeps, projectFileMap, tsConfig) {
    var _a;
    const nxJsonPlugins = (nxJson.plugins || []).map((p) => ({
        name: p,
        version: packageJsonDeps[p],
    }));
    const newValue = {
        version: '6.0',
        nxVersion: versions_1.nxVersion,
        deps: packageJsonDeps,
        // compilerOptions may not exist, especially for package-based repos
        pathMappings: ((_a = tsConfig === null || tsConfig === void 0 ? void 0 : tsConfig.compilerOptions) === null || _a === void 0 ? void 0 : _a.paths) || {},
        nxJsonPlugins,
        pluginsConfig: nxJson.pluginsConfig,
        projectFileMap,
    };
    return newValue;
}
exports.createProjectFileMapCache = createProjectFileMapCache;
function writeCache(cache, projectGraph) {
    perf_hooks_1.performance.mark('write cache:start');
    let retry = 1;
    let done = false;
    do {
        // write first to a unique temporary filename and then do a
        // rename of the file to the correct filename
        // this is to avoid any problems with half-written files
        // in case of crash and/or partially written files due
        // to multiple parallel processes reading and writing this file
        const unique = (Math.random().toString(16) + '0000000').slice(2, 10);
        const tmpProjectGraphPath = `${exports.nxProjectGraph}~${unique}`;
        const tmpFileMapPath = `${exports.nxFileMap}~${unique}`;
        try {
            (0, fileutils_1.writeJsonFile)(tmpProjectGraphPath, projectGraph);
            (0, fs_extra_1.renameSync)(tmpProjectGraphPath, exports.nxProjectGraph);
            (0, fileutils_1.writeJsonFile)(tmpFileMapPath, cache);
            (0, fs_extra_1.renameSync)(tmpFileMapPath, exports.nxFileMap);
            done = true;
        }
        catch (err) {
            if (err instanceof Error) {
                console.log(`ERROR (${retry}) when writing \n${err.message}\n${err.stack}`);
            }
            else {
                console.log(`ERROR  (${retry}) unknown error when writing ${exports.nxProjectGraph} and ${exports.nxFileMap}`);
            }
            ++retry;
        }
    } while (!done && retry < 5);
    perf_hooks_1.performance.mark('write cache:end');
    perf_hooks_1.performance.measure('write cache', 'write cache:start', 'write cache:end');
}
exports.writeCache = writeCache;
function shouldRecomputeWholeGraph(cache, packageJsonDeps, projects, nxJson, tsConfig) {
    if (cache.version !== '6.0') {
        return true;
    }
    if (cache.nxVersion !== versions_1.nxVersion) {
        return true;
    }
    // we have a cached project that is no longer present
    const cachedNodes = Object.keys(cache.projectFileMap);
    if (cachedNodes.some((p) => projects.projects[p] === undefined)) {
        return true;
    }
    // a path mapping for an existing project has changed
    if (Object.keys(cache.pathMappings).some((t) => {
        var _a, _b;
        const cached = cache.pathMappings && cache.pathMappings[t]
            ? JSON.stringify(cache.pathMappings[t])
            : undefined;
        const notCached = ((_a = tsConfig === null || tsConfig === void 0 ? void 0 : tsConfig.compilerOptions) === null || _a === void 0 ? void 0 : _a.paths) && ((_b = tsConfig === null || tsConfig === void 0 ? void 0 : tsConfig.compilerOptions) === null || _b === void 0 ? void 0 : _b.paths[t])
            ? JSON.stringify(tsConfig.compilerOptions.paths[t])
            : undefined;
        return cached !== notCached;
    })) {
        return true;
    }
    // a new plugin has been added
    if ((nxJson.plugins || []).length !== cache.nxJsonPlugins.length)
        return true;
    // a plugin has changed
    if ((nxJson.plugins || []).some((t) => {
        const matchingPlugin = cache.nxJsonPlugins.find((p) => p.name === t);
        if (!matchingPlugin)
            return true;
        return matchingPlugin.version !== packageJsonDeps[t];
    })) {
        return true;
    }
    if (JSON.stringify(nxJson.pluginsConfig) !== JSON.stringify(cache.pluginsConfig)) {
        return true;
    }
    return false;
}
exports.shouldRecomputeWholeGraph = shouldRecomputeWholeGraph;
/*
This can only be invoked when the list of projects is either the same
or new projects have been added, so every project in the cache has a corresponding
project in fileMap
*/
function extractCachedFileData(fileMap, c) {
    const filesToProcess = {};
    const cachedFileData = {};
    const currentProjects = Object.keys(fileMap).filter((name) => fileMap[name].length > 0);
    currentProjects.forEach((p) => {
        processProjectNode(p, c.projectFileMap, cachedFileData, filesToProcess, fileMap);
    });
    return {
        filesToProcess,
        cachedFileData,
    };
}
exports.extractCachedFileData = extractCachedFileData;
function processProjectNode(projectName, cachedFileMap, cachedFileData, filesToProcess, fileMap) {
    if (!cachedFileMap[projectName]) {
        filesToProcess[projectName] = fileMap[projectName];
        return;
    }
    const fileDataFromCache = {};
    for (let f of cachedFileMap[projectName]) {
        fileDataFromCache[f.file] = f;
    }
    if (!cachedFileData[projectName]) {
        cachedFileData[projectName] = {};
    }
    for (let f of fileMap[projectName]) {
        const fromCache = fileDataFromCache[f.file];
        if (fromCache && fromCache.hash == f.hash) {
            cachedFileData[projectName][f.file] = fromCache;
        }
        else {
            if (!filesToProcess[projectName]) {
                filesToProcess[projectName] = [];
            }
            filesToProcess[projectName].push(f);
        }
    }
}
