"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mapTargetDefaultsToDependencies = exports.createTaskGraph = exports.ProcessTasks = void 0;
const utils_1 = require("./utils");
const project_graph_utils_1 = require("../utils/project-graph-utils");
const find_matching_projects_1 = require("../utils/find-matching-projects");
const output_1 = require("../utils/output");
class ProcessTasks {
    constructor(defaultDependencyConfigs, projectGraph) {
        this.defaultDependencyConfigs = defaultDependencyConfigs;
        this.projectGraph = projectGraph;
        this.seen = new Set();
        this.tasks = {};
        this.dependencies = {};
    }
    processTasks(projectNames, targets, configuration, overrides, excludeTaskDependencies) {
        for (const projectName of projectNames) {
            for (const target of targets) {
                const project = this.projectGraph.nodes[projectName];
                if (targets.length === 1 || project.data.targets[target]) {
                    const resolvedConfiguration = this.resolveConfiguration(project, target, configuration);
                    const id = this.getId(projectName, target, resolvedConfiguration);
                    const task = this.createTask(id, project, target, resolvedConfiguration, overrides);
                    this.tasks[task.id] = task;
                    this.dependencies[task.id] = [];
                }
            }
        }
        // used when excluding tasks
        const initialTasks = Object.assign({}, this.tasks);
        for (const taskId of Object.keys(this.tasks)) {
            const task = this.tasks[taskId];
            this.processTask(task, task.target.project, configuration, overrides);
        }
        if (excludeTaskDependencies) {
            for (let t of Object.keys(this.tasks)) {
                if (!initialTasks[t]) {
                    delete this.tasks[t];
                    delete this.dependencies[t];
                }
            }
            for (let d of Object.keys(this.dependencies)) {
                this.dependencies[d] = this.dependencies[d].filter((dd) => !!initialTasks[dd]);
            }
        }
        for (const projectName of Object.keys(this.dependencies)) {
            if (this.dependencies[projectName].length > 1) {
                this.dependencies[projectName] = [
                    ...new Set(this.dependencies[projectName]).values(),
                ];
            }
        }
        return Object.keys(this.dependencies).filter((d) => this.dependencies[d].length === 0);
    }
    processTask(task, projectUsedToDeriveDependencies, configuration, overrides) {
        const seenKey = `${task.id}-${projectUsedToDeriveDependencies}`;
        if (this.seen.has(seenKey)) {
            return;
        }
        this.seen.add(seenKey);
        const dependencyConfigs = (0, utils_1.getDependencyConfigs)({ project: task.target.project, target: task.target.target }, this.defaultDependencyConfigs, this.projectGraph);
        for (const dependencyConfig of dependencyConfigs) {
            const taskOverrides = dependencyConfig.params === 'forward'
                ? overrides
                : { __overrides_unparsed__: [] };
            if (dependencyConfig.projects) {
                this.processTasksForMatchingProjects(dependencyConfig, projectUsedToDeriveDependencies, configuration, task, taskOverrides, overrides);
            }
            else if (dependencyConfig.dependencies) {
                this.processTasksForDependencies(projectUsedToDeriveDependencies, dependencyConfig, configuration, task, taskOverrides, overrides);
            }
            else {
                this.processTasksForSingleProject(task, task.target.project, dependencyConfig, configuration, taskOverrides, overrides);
            }
        }
    }
    processTasksForMatchingProjects(dependencyConfig, projectUsedToDeriveDependencies, configuration, task, taskOverrides, overrides) {
        const targetProjectSpecifiers = typeof dependencyConfig.projects === 'string'
            ? [dependencyConfig.projects]
            : dependencyConfig.projects;
        for (const projectSpecifier of targetProjectSpecifiers) {
            // Lerna uses `dependencies` in `prepNxOptions`, so we need to maintain
            // support for it until lerna can be updated to use the syntax.
            // TODO(@agentender): Remove this part in v17
            if (projectSpecifier === 'dependencies' &&
                !this.projectGraph.nodes[projectSpecifier]) {
                this.processTasksForDependencies(projectUsedToDeriveDependencies, dependencyConfig, configuration, task, taskOverrides, overrides);
            }
            else {
                // Since we need to maintain support for dependencies, it is more coherent
                // that we also support self.
                // TODO(@agentender): Remove this part in v17
                const matchingProjects = 
                /** LERNA SUPPORT START - Remove in v17 */
                projectSpecifier === 'self' &&
                    !this.projectGraph.nodes[projectSpecifier]
                    ? [task.target.project]
                    : /** LERNA SUPPORT END */
                        (0, find_matching_projects_1.findMatchingProjects)([projectSpecifier], this.projectGraph.nodes);
                if (matchingProjects.length === 0) {
                    output_1.output.warn({
                        title: `\`dependsOn\` is misconfigured for ${task.target.project}:${task.target.target}`,
                        bodyLines: [
                            `Project pattern "${projectSpecifier}" does not match any projects.`,
                        ],
                    });
                }
                for (const projectName of matchingProjects) {
                    this.processTasksForSingleProject(task, projectName, dependencyConfig, configuration, taskOverrides, overrides);
                }
            }
        }
    }
    processTasksForSingleProject(task, projectName, dependencyConfig, configuration, taskOverrides, overrides) {
        const selfProject = this.projectGraph.nodes[projectName];
        if ((0, project_graph_utils_1.projectHasTarget)(selfProject, dependencyConfig.target)) {
            const resolvedConfiguration = this.resolveConfiguration(selfProject, dependencyConfig.target, configuration);
            const selfTaskId = this.getId(selfProject.name, dependencyConfig.target, resolvedConfiguration);
            if (task.id !== selfTaskId) {
                this.dependencies[task.id].push(selfTaskId);
            }
            if (!this.tasks[selfTaskId]) {
                const newTask = this.createTask(selfTaskId, selfProject, dependencyConfig.target, resolvedConfiguration, taskOverrides);
                this.tasks[selfTaskId] = newTask;
                this.dependencies[selfTaskId] = [];
                this.processTask(newTask, newTask.target.project, configuration, overrides);
            }
        }
    }
    processTasksForDependencies(projectUsedToDeriveDependencies, dependencyConfig, configuration, task, taskOverrides, overrides) {
        for (const dep of this.projectGraph.dependencies[projectUsedToDeriveDependencies]) {
            const depProject = this.projectGraph.nodes[dep.target];
            // this is to handle external dependencies
            if (!depProject)
                continue;
            if ((0, project_graph_utils_1.projectHasTarget)(depProject, dependencyConfig.target)) {
                const resolvedConfiguration = this.resolveConfiguration(depProject, dependencyConfig.target, configuration);
                const depTargetId = this.getId(depProject.name, dependencyConfig.target, resolvedConfiguration);
                if (task.id !== depTargetId) {
                    this.dependencies[task.id].push(depTargetId);
                }
                if (!this.tasks[depTargetId]) {
                    const newTask = this.createTask(depTargetId, depProject, dependencyConfig.target, resolvedConfiguration, taskOverrides);
                    this.tasks[depTargetId] = newTask;
                    this.dependencies[depTargetId] = [];
                    this.processTask(newTask, newTask.target.project, configuration, overrides);
                }
            }
            else {
                this.processTask(task, depProject.name, configuration, overrides);
            }
        }
    }
    createTask(id, project, target, resolvedConfiguration, overrides) {
        if (!project.data.targets[target]) {
            throw new Error(`Cannot find configuration for task ${project.name}:${target}`);
        }
        if (!project.data.targets[target].executor) {
            throw new Error(`Target "${project.name}:${target}" does not have an executor configured`);
        }
        const qualifiedTarget = {
            project: project.name,
            target,
            configuration: resolvedConfiguration,
        };
        return {
            id,
            target: qualifiedTarget,
            projectRoot: project.data.root,
            overrides: interpolateOverrides(overrides, project.name, project.data),
        };
    }
    resolveConfiguration(project, target, configuration) {
        var _a, _b;
        const defaultConfiguration = (_b = (_a = project.data.targets) === null || _a === void 0 ? void 0 : _a[target]) === null || _b === void 0 ? void 0 : _b.defaultConfiguration;
        configuration !== null && configuration !== void 0 ? configuration : (configuration = defaultConfiguration);
        return (0, project_graph_utils_1.projectHasTargetAndConfiguration)(project, target, configuration)
            ? configuration
            : defaultConfiguration;
    }
    getId(project, target, configuration) {
        let id = `${project}:${target}`;
        if (configuration) {
            id += `:${configuration}`;
        }
        return id;
    }
}
exports.ProcessTasks = ProcessTasks;
function createTaskGraph(projectGraph, defaultDependencyConfigs, projectNames, targets, configuration, overrides, excludeTaskDependencies = false) {
    const p = new ProcessTasks(defaultDependencyConfigs, projectGraph);
    const roots = p.processTasks(projectNames, targets, configuration, overrides, excludeTaskDependencies);
    return {
        roots,
        tasks: p.tasks,
        dependencies: p.dependencies,
    };
}
exports.createTaskGraph = createTaskGraph;
function mapTargetDefaultsToDependencies(defaults) {
    const res = {};
    Object.keys(defaults !== null && defaults !== void 0 ? defaults : {}).forEach((k) => {
        res[k] = defaults[k].dependsOn;
    });
    return res;
}
exports.mapTargetDefaultsToDependencies = mapTargetDefaultsToDependencies;
function interpolateOverrides(args, projectName, project) {
    const interpolatedArgs = Object.assign({}, args);
    Object.entries(interpolatedArgs).forEach(([name, value]) => {
        interpolatedArgs[name] =
            typeof value === 'string'
                ? (0, utils_1.interpolate)(value, {
                    workspaceRoot: '',
                    projectRoot: project.root,
                    projectName: project.name,
                    project: Object.assign(Object.assign({}, project), { name: projectName }), // this is legacy
                })
                : value;
    });
    return interpolatedArgs;
}
