"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.assertWorkspaceValidity = void 0;
const find_matching_projects_1 = require("./find-matching-projects");
const output_1 = require("./output");
function assertWorkspaceValidity(projectsConfigurations, nxJson) {
    const projectNames = Object.keys(projectsConfigurations.projects);
    const projectGraphNodes = projectNames.reduce((graph, project) => {
        const projectConfiguration = projectsConfigurations.projects[project];
        graph[project] = {
            name: project,
            type: projectConfiguration.projectType === 'library' ? 'lib' : 'app',
            data: Object.assign({}, projectConfiguration),
        };
        return graph;
    }, {});
    const projects = Object.assign({}, projectsConfigurations.projects);
    const invalidImplicitDependencies = new Map();
    if (nxJson.implicitDependencies) {
        output_1.output.warn({
            title: 'Using `implicitDependencies` for global implicit dependencies configuration is no longer supported.',
            bodyLines: [
                'Use "namedInputs" instead. You can run "nx repair" to automatically migrate your configuration.',
                'For more information about the usage of "namedInputs" see https://nx.dev/deprecated/global-implicit-dependencies#global-implicit-dependencies',
            ],
        });
    }
    const projectsWithNonArrayImplicitDependencies = new Map();
    projectNames
        .filter((projectName) => {
        const project = projects[projectName];
        // Report if for whatever reason, a project is configured to use implicitDependencies but it is not an array
        if (!!project.implicitDependencies &&
            !Array.isArray(project.implicitDependencies)) {
            projectsWithNonArrayImplicitDependencies.set(projectName, project.implicitDependencies);
        }
        return (!!project.implicitDependencies &&
            Array.isArray(project.implicitDependencies));
    })
        .reduce((map, projectName) => {
        const project = projects[projectName];
        detectAndSetInvalidProjectGlobValues(map, projectName, project.implicitDependencies, projects, projectGraphNodes);
        return map;
    }, invalidImplicitDependencies);
    if (projectsWithNonArrayImplicitDependencies.size === 0 &&
        invalidImplicitDependencies.size === 0) {
        // No issues
        return;
    }
    let message = '';
    if (projectsWithNonArrayImplicitDependencies.size > 0) {
        message += `The following implicitDependencies should be an array of strings:\n`;
        projectsWithNonArrayImplicitDependencies.forEach((implicitDependencies, projectName) => {
            message += `  ${projectName}.implicitDependencies: "${implicitDependencies}"\n`;
        });
        message += '\n';
    }
    if (invalidImplicitDependencies.size > 0) {
        message += `The following implicitDependencies point to non-existent project(s):\n`;
        message += [...invalidImplicitDependencies.keys()]
            .map((key) => {
            const projectNames = invalidImplicitDependencies.get(key);
            return `  ${key}\n${projectNames
                .map((projectName) => `    ${projectName}`)
                .join('\n')}`;
        })
            .join('\n\n');
    }
    throw new Error(`Configuration Error\n${message}`);
}
exports.assertWorkspaceValidity = assertWorkspaceValidity;
function detectAndSetInvalidProjectGlobValues(map, sourceName, desiredImplicitDeps, projectConfigurations, projects) {
    const invalidProjectsOrGlobs = desiredImplicitDeps.filter((implicit) => {
        const projectName = implicit.startsWith('!')
            ? implicit.substring(1)
            : implicit;
        return !(projectConfigurations[projectName] ||
            (0, find_matching_projects_1.findMatchingProjects)([implicit], projects).length);
    });
    if (invalidProjectsOrGlobs.length > 0) {
        map.set(sourceName, invalidProjectsOrGlobs);
    }
}
