"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.packageRegistryPack = exports.packageRegistryView = exports.resolvePackageVersionUsingInstallation = exports.resolvePackageVersionUsingRegistry = exports.createTempNpmDirectory = exports.copyPackageManagerConfigurationFiles = exports.modifyYarnRcToFitNewDirectory = exports.modifyYarnRcYmlToFitNewDirectory = exports.findFileInPackageJsonDirectory = exports.getPackageManagerVersion = exports.getPackageManagerCommand = exports.detectPackageManager = void 0;
const tslib_1 = require("tslib");
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const fs_extra_1 = require("fs-extra");
const path_1 = require("path");
const tmp_1 = require("tmp");
const util_1 = require("util");
const fileutils_1 = require("./fileutils");
const package_json_1 = require("./package-json");
const semver_1 = require("semver");
const workspace_root_1 = require("./workspace-root");
const configuration_1 = require("../config/configuration");
const execAsync = (0, util_1.promisify)(child_process_1.exec);
/**
 * Detects which package manager is used in the workspace based on the lock file.
 */
function detectPackageManager(dir = '') {
    var _a, _b;
    const nxJson = (0, configuration_1.readNxJson)();
    return ((_b = (_a = nxJson.cli) === null || _a === void 0 ? void 0 : _a.packageManager) !== null && _b !== void 0 ? _b : ((0, fs_1.existsSync)((0, path_1.join)(dir, 'yarn.lock'))
        ? 'yarn'
        : (0, fs_1.existsSync)((0, path_1.join)(dir, 'pnpm-lock.yaml'))
            ? 'pnpm'
            : 'npm'));
}
exports.detectPackageManager = detectPackageManager;
/**
 * Returns commands for the package manager used in the workspace.
 * By default, the package manager is derived based on the lock file,
 * but it can also be passed in explicitly.
 *
 * Example:
 *
 * ```javascript
 * execSync(`${getPackageManagerCommand().addDev} my-dev-package`);
 * ```
 *
 * @param packageManager The package manager to use. If not provided, it will be detected based on the lock file.
 * @param root The directory the commands will be ran inside of. Defaults to the current workspace's root.
 */
function getPackageManagerCommand(packageManager = detectPackageManager(), root = workspace_root_1.workspaceRoot) {
    const commands = {
        yarn: () => {
            const yarnVersion = getPackageManagerVersion('yarn');
            const useBerry = (0, semver_1.gte)(yarnVersion, '2.0.0');
            return {
                preInstall: useBerry
                    ? 'yarn set version stable'
                    : 'yarn set version classic',
                install: 'yarn',
                ciInstall: useBerry
                    ? 'yarn install --immutable'
                    : 'yarn install --frozen-lockfile',
                add: useBerry ? 'yarn add' : 'yarn add -W',
                addDev: useBerry ? 'yarn add -D' : 'yarn add -D -W',
                rm: 'yarn remove',
                exec: useBerry ? 'yarn exec' : 'yarn',
                run: (script, args) => `yarn ${script} ${args}`,
                list: useBerry ? 'yarn info --name-only' : 'yarn list',
            };
        },
        pnpm: () => {
            const pnpmVersion = getPackageManagerVersion('pnpm');
            const useExec = (0, semver_1.gte)(pnpmVersion, '6.13.0');
            const includeDoubleDashBeforeArgs = (0, semver_1.lt)(pnpmVersion, '7.0.0');
            const isPnpmWorkspace = (0, fs_1.existsSync)((0, path_1.join)(root, 'pnpm-workspace.yaml'));
            return {
                install: 'pnpm install --no-frozen-lockfile',
                ciInstall: 'pnpm install --frozen-lockfile',
                add: isPnpmWorkspace ? 'pnpm add -w' : 'pnpm add',
                addDev: isPnpmWorkspace ? 'pnpm add -Dw' : 'pnpm add -D',
                rm: 'pnpm rm',
                exec: useExec ? 'pnpm exec' : 'pnpx',
                run: (script, args) => includeDoubleDashBeforeArgs
                    ? `pnpm run ${script} -- ${args}`
                    : `pnpm run ${script} ${args}`,
                list: 'pnpm ls --depth 100',
            };
        },
        npm: () => {
            var _a;
            var _b;
            // TODO: Remove this
            (_a = (_b = process.env).npm_config_legacy_peer_deps) !== null && _a !== void 0 ? _a : (_b.npm_config_legacy_peer_deps = 'true');
            return {
                install: 'npm install',
                ciInstall: 'npm ci',
                add: 'npm install',
                addDev: 'npm install -D',
                rm: 'npm rm',
                exec: 'npx',
                run: (script, args) => `npm run ${script} -- ${args}`,
                list: 'npm ls',
            };
        },
    };
    return commands[packageManager]();
}
exports.getPackageManagerCommand = getPackageManagerCommand;
/**
 * Returns the version of the package manager used in the workspace.
 * By default, the package manager is derived based on the lock file,
 * but it can also be passed in explicitly.
 */
function getPackageManagerVersion(packageManager = detectPackageManager()) {
    return (0, child_process_1.execSync)(`${packageManager} --version`).toString('utf-8').trim();
}
exports.getPackageManagerVersion = getPackageManagerVersion;
/**
 * Checks for a project level npmrc file by crawling up the file tree until
 * hitting a package.json file, as this is how npm finds them as well.
 */
function findFileInPackageJsonDirectory(file, directory = process.cwd()) {
    while (!(0, fs_1.existsSync)((0, path_1.join)(directory, 'package.json'))) {
        directory = (0, path_1.dirname)(directory);
    }
    const path = (0, path_1.join)(directory, file);
    return (0, fs_1.existsSync)(path) ? path : null;
}
exports.findFileInPackageJsonDirectory = findFileInPackageJsonDirectory;
/**
 * We copy yarnrc.yml to the temporary directory to ensure things like the specified
 * package registry are still used. However, there are a few relative paths that can
 * cause issues, so we modify them to fit the new directory.
 *
 * Exported for testing - not meant to be used outside of this file.
 *
 * @param contents The string contents of the yarnrc.yml file
 * @returns Updated string contents of the yarnrc.yml file
 */
function modifyYarnRcYmlToFitNewDirectory(contents) {
    const { parseSyml, stringifySyml } = require('@yarnpkg/parsers');
    const parsed = parseSyml(contents);
    if (parsed.yarnPath) {
        // yarnPath is relative to the workspace root, so we need to make it relative
        // to the new directory s.t. it still points to the same yarn binary.
        delete parsed.yarnPath;
    }
    if (parsed.plugins) {
        // Plugins specified by a string are relative paths from workspace root.
        // ex: https://yarnpkg.com/advanced/plugin-tutorial#writing-our-first-plugin
        delete parsed.plugins;
    }
    return stringifySyml(parsed);
}
exports.modifyYarnRcYmlToFitNewDirectory = modifyYarnRcYmlToFitNewDirectory;
/**
 * We copy .yarnrc to the temporary directory to ensure things like the specified
 * package registry are still used. However, there are a few relative paths that can
 * cause issues, so we modify them to fit the new directory.
 *
 * Exported for testing - not meant to be used outside of this file.
 *
 * @param contents The string contents of the yarnrc.yml file
 * @returns Updated string contents of the yarnrc.yml file
 */
function modifyYarnRcToFitNewDirectory(contents) {
    const lines = contents.split('\n');
    const yarnPathIndex = lines.findIndex((line) => line.startsWith('yarn-path'));
    if (yarnPathIndex !== -1) {
        lines.splice(yarnPathIndex, 1);
    }
    return lines.join('\n');
}
exports.modifyYarnRcToFitNewDirectory = modifyYarnRcToFitNewDirectory;
function copyPackageManagerConfigurationFiles(root, destination) {
    for (const packageManagerConfigFile of ['.npmrc', '.yarnrc', '.yarnrc.yml']) {
        // f is an absolute path, including the {workspaceRoot}.
        const f = findFileInPackageJsonDirectory(packageManagerConfigFile, root);
        if (f) {
            // Destination should be the same relative path from the {workspaceRoot},
            // but now relative to the destination. `relative` makes `{workspaceRoot}/some/path`
            // look like `./some/path`, and joining that gets us `{destination}/some/path
            const destinationPath = (0, path_1.join)(destination, (0, path_1.relative)(root, f));
            switch (packageManagerConfigFile) {
                case '.npmrc': {
                    (0, fs_1.copyFileSync)(f, destinationPath);
                    break;
                }
                case '.yarnrc': {
                    const updated = modifyYarnRcToFitNewDirectory((0, fileutils_1.readFileIfExisting)(f));
                    (0, fs_1.writeFileSync)(destinationPath, updated);
                    break;
                }
                case '.yarnrc.yml': {
                    const updated = modifyYarnRcYmlToFitNewDirectory((0, fileutils_1.readFileIfExisting)(f));
                    (0, fs_1.writeFileSync)(destinationPath, updated);
                    break;
                }
            }
        }
    }
}
exports.copyPackageManagerConfigurationFiles = copyPackageManagerConfigurationFiles;
/**
 * Creates a temporary directory where you can run package manager commands safely.
 *
 * For cases where you'd want to install packages that require an `.npmrc` set up,
 * this function looks up for the nearest `.npmrc` (if exists) and copies it over to the
 * temp directory.
 */
function createTempNpmDirectory() {
    const dir = (0, tmp_1.dirSync)().name;
    // A package.json is needed for pnpm pack and for .npmrc to resolve
    (0, fileutils_1.writeJsonFile)(`${dir}/package.json`, {});
    copyPackageManagerConfigurationFiles(workspace_root_1.workspaceRoot, dir);
    const cleanup = () => tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            yield (0, fs_extra_1.remove)(dir);
        }
        catch (_a) {
            // It's okay if this fails, the OS will clean it up eventually
        }
    });
    return { dir, cleanup };
}
exports.createTempNpmDirectory = createTempNpmDirectory;
/**
 * Returns the resolved version for a given package and version tag using the
 * NPM registry (when using Yarn it will fall back to NPM to fetch the info).
 */
function resolvePackageVersionUsingRegistry(packageName, version) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            const result = yield packageRegistryView(packageName, version, 'version');
            if (!result) {
                throw new Error(`Unable to resolve version ${packageName}@${version}.`);
            }
            // get the last line of the output, strip the package version and quotes
            const resolvedVersion = result
                .split('\n')
                .pop()
                .split(' ')
                .pop()
                .replace(/'/g, '');
            return resolvedVersion;
        }
        catch (_a) {
            throw new Error(`Unable to resolve version ${packageName}@${version}.`);
        }
    });
}
exports.resolvePackageVersionUsingRegistry = resolvePackageVersionUsingRegistry;
/**
 * Return the resolved version for a given package and version tag using by
 * installing it in a temporary directory and fetching the version from the
 * package.json.
 */
function resolvePackageVersionUsingInstallation(packageName, version) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const { dir, cleanup } = createTempNpmDirectory();
        try {
            const pmc = getPackageManagerCommand();
            yield execAsync(`${pmc.add} ${packageName}@${version}`, { cwd: dir });
            const { packageJson } = (0, package_json_1.readModulePackageJson)(packageName, [dir]);
            return packageJson.version;
        }
        finally {
            yield cleanup();
        }
    });
}
exports.resolvePackageVersionUsingInstallation = resolvePackageVersionUsingInstallation;
function packageRegistryView(pkg, version, args) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        let pm = detectPackageManager();
        if (pm === 'yarn') {
            /**
             * yarn has `yarn info` but it behaves differently than (p)npm,
             * which makes it's usage unreliable
             *
             * @see https://github.com/nrwl/nx/pull/9667#discussion_r842553994
             */
            pm = 'npm';
        }
        const { stdout } = yield execAsync(`${pm} view ${pkg}@${version} ${args}`);
        return stdout.toString().trim();
    });
}
exports.packageRegistryView = packageRegistryView;
function packageRegistryPack(cwd, pkg, version) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        let pm = detectPackageManager();
        if (pm === 'yarn') {
            /**
             * `(p)npm pack` will download a tarball of the specified version,
             * whereas `yarn` pack creates a tarball of the active workspace, so it
             * does not work for getting the content of a library.
             *
             * @see https://github.com/nrwl/nx/pull/9667#discussion_r842553994
             */
            pm = 'npm';
        }
        const { stdout } = yield execAsync(`${pm} pack ${pkg}@${version}`, { cwd });
        const tarballPath = stdout.trim();
        return { tarballPath };
    });
}
exports.packageRegistryPack = packageRegistryPack;
