"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.listPluginCapabilities = exports.getPluginCapabilities = void 0;
const tslib_1 = require("tslib");
const workspace_root_1 = require("../workspace-root");
const chalk = require("chalk");
const path_1 = require("path");
const output_1 = require("../output");
const shared_1 = require("./shared");
const fileutils_1 = require("../fileutils");
const package_manager_1 = require("../package-manager");
const nx_plugin_1 = require("../nx-plugin");
const installation_directory_1 = require("../installation-directory");
function tryGetCollection(packageJsonPath, collectionFile, propName) {
    if (!collectionFile) {
        return null;
    }
    try {
        const collectionFilePath = (0, path_1.join)((0, path_1.dirname)(packageJsonPath), collectionFile);
        return (0, fileutils_1.readJsonFile)(collectionFilePath)[propName];
    }
    catch (_a) {
        return null;
    }
}
function getPluginCapabilities(workspaceRoot, pluginName) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            const { json: packageJson, path: packageJsonPath } = (0, nx_plugin_1.readPluginPackageJson)(pluginName, (0, installation_directory_1.getNxRequirePaths)(workspaceRoot));
            const pluginModule = yield tryGetModule(packageJson, workspaceRoot);
            return {
                name: pluginName,
                generators: Object.assign(Object.assign(Object.assign(Object.assign({}, tryGetCollection(packageJsonPath, packageJson.schematics, 'schematics')), tryGetCollection(packageJsonPath, packageJson.generators, 'schematics')), tryGetCollection(packageJsonPath, packageJson.schematics, 'generators')), tryGetCollection(packageJsonPath, packageJson.generators, 'generators')),
                executors: Object.assign(Object.assign(Object.assign(Object.assign({}, tryGetCollection(packageJsonPath, packageJson.builders, 'builders')), tryGetCollection(packageJsonPath, packageJson.executors, 'builders')), tryGetCollection(packageJsonPath, packageJson.builders, 'executors')), tryGetCollection(packageJsonPath, packageJson.executors, 'executors')),
                projectGraphExtension: !!(pluginModule === null || pluginModule === void 0 ? void 0 : pluginModule.processProjectGraph),
                projectInference: !!(pluginModule === null || pluginModule === void 0 ? void 0 : pluginModule.projectFilePatterns),
            };
        }
        catch (_a) {
            return null;
        }
    });
}
exports.getPluginCapabilities = getPluginCapabilities;
function tryGetModule(packageJson, workspaceRoot) {
    var _a, _b, _c, _d;
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            return ((_d = (_c = (_b = (_a = packageJson.generators) !== null && _a !== void 0 ? _a : packageJson.executors) !== null && _b !== void 0 ? _b : packageJson['nx-migrations']) !== null && _c !== void 0 ? _c : packageJson['schematics']) !== null && _d !== void 0 ? _d : packageJson['builders'])
                ? yield (0, nx_plugin_1.loadNxPluginAsync)(packageJson.name, (0, installation_directory_1.getNxRequirePaths)(workspaceRoot), workspaceRoot)
                : {
                    name: packageJson.name,
                };
        }
        catch (_e) {
            return null;
        }
    });
}
function listPluginCapabilities(pluginName) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const plugin = yield getPluginCapabilities(workspace_root_1.workspaceRoot, pluginName);
        if (!plugin) {
            const pmc = (0, package_manager_1.getPackageManagerCommand)();
            output_1.output.note({
                title: `${pluginName} is not currently installed`,
                bodyLines: [
                    `Use "${pmc.addDev} ${pluginName}" to install the plugin.`,
                    `After that, use "${pmc.exec} nx g ${pluginName}:init" to add the required peer deps and initialize the plugin.`,
                ],
            });
            return;
        }
        const hasBuilders = (0, shared_1.hasElements)(plugin.executors);
        const hasGenerators = (0, shared_1.hasElements)(plugin.generators);
        const hasProjectGraphExtension = !!plugin.projectGraphExtension;
        const hasProjectInference = !!plugin.projectInference;
        if (!hasBuilders &&
            !hasGenerators &&
            !hasProjectGraphExtension &&
            !hasProjectInference) {
            output_1.output.warn({ title: `No capabilities found in ${pluginName}` });
            return;
        }
        const bodyLines = [];
        if (hasGenerators) {
            bodyLines.push(chalk.bold(chalk.green('GENERATORS')));
            bodyLines.push('');
            bodyLines.push(...Object.keys(plugin.generators).map((name) => `${chalk.bold(name)} : ${plugin.generators[name].description}`));
            if (hasBuilders) {
                bodyLines.push('');
            }
        }
        if (hasBuilders) {
            bodyLines.push(chalk.bold(chalk.green('EXECUTORS/BUILDERS')));
            bodyLines.push('');
            bodyLines.push(...Object.keys(plugin.executors).map((name) => `${chalk.bold(name)} : ${plugin.executors[name].description}`));
        }
        if (hasProjectGraphExtension) {
            bodyLines.push(`✔️  Project Graph Extension`);
        }
        if (hasProjectInference) {
            bodyLines.push(`✔️  Project Inference`);
        }
        output_1.output.log({
            title: `Capabilities in ${plugin.name}:`,
            bodyLines,
        });
    });
}
exports.listPluginCapabilities = listPluginCapabilities;
