import { Component, OnInit } from '@angular/core';
import { Router, NavigationExtras, ActivatedRoute } from '@angular/router';
import { AlertController } from '@ionic/angular';

@Component({
  selector: 'app-actions-select',
  templateUrl: './actions-select.page.html',
  styleUrls: ['./actions-select.page.scss'],
})
export class ActionsSelectPage implements OnInit {
  public itemType: string = '';
  public selectedLength: number = 0;
  public filter: string = '';
  // Variable to determine if a Bull (Sex ID = 1) is present in list of selected / passed animals
  public bullPresent: boolean = false;
  private selectedAmount: string = '';
  private selected: any = [];

  constructor(
    private alertCtrl: AlertController,
    public router: Router,
    private route: ActivatedRoute
  ) {
    this.route.queryParams.subscribe(_ => {
      if (this.router.getCurrentNavigation()?.extras.state) {
        // `animals` or `groups`
        this.itemType = this.router.getCurrentNavigation()?.extras.state?.['itemType'];

        // animals or groups selected by the user
        this.selected = this.router.getCurrentNavigation()?.extras.state?.['selected'];
        console.log(this.selected);

        // current length of selected items (groups or animals)
        this.selectedLength = this.selected.length;

        // e.g., 'allAnimals'
        this.filter = this.router.getCurrentNavigation()?.extras.state?.['filter'];

        /*
        * If the user has selected items (animals or groups)
          - Get length of current selected items
        * Determine if the length is a single item or multiple items
        */
        if (this.selected) {
          /*
          * Iterate through the selected animals for Record Data
          * If a Bull is present in the animals (Sex ID = 1), then
              hide the Breeding action, as we should only use this when cow(s) are selected
          * NOTE: User will select a Bull to be applied to Cows for any of the Breeding events
          */
          for (const s of this.selected) {
            if (s.sex_id === 1) {
              // If there is a Bull, set variable to true and break from loop
              console.log('Bull present in selected animals; hiding Breeding action...')
              this.bullPresent = true;
              break;
            }
          }

          // Gather the current length of the selected animals for Record Data
          this.selectedLength = this.selected.length;
          if (this.selectedLength) {
            if (this.selectedLength === 1) {
              this.selectedAmount = 'single';
            } else if (this.selectedLength > 1) {
              this.selectedAmount = 'multiple';
            }

          } else {
            this.router.navigateByUrl('/tabs/home');
          }
        } else {
          this.router.navigateByUrl('/tabs/home');
        }
      } else {
        this.router.navigateByUrl('/tabs/home');
      }
    });
  }

  /*
  * If the user decides to cancel applying actions to their selected items
  * Navigate back to the main `Home` page
  */
  cancelActions() {
    this.router.navigateByUrl('/tabs/home');
  }

  actionInput(actionType: string, action: string) {

    if (actionType && action) {
      const navigationExtras: NavigationExtras = {
        state: {
          // e.g., string: `animals` or `groups`
          itemType: this.itemType,
          // e.g., string: `health`, `vaccination`, `custom`, `breeding`
          actionType,
          // e.g. string: `observation`, `treatment`, `vaccination`, `pregnancy`, `custom`, `breeding`
          action,
          // e.g., list: animal(s)/group(s) selected by user
          selected: this.selected,
          // e.g., string: `single` or `multiple`
          selectedAmount: this.selectedAmount
        }
      };

      // Navigate to appropriate page based on `actionType`
      if (actionType === 'health') {
        this.router.navigateByUrl('/tabs/home/actions/health', navigationExtras);
      } else if (actionType === 'vaccination') {
        this.router.navigateByUrl('/tabs/home/actions/vaccination', navigationExtras);
      } else if (actionType === 'custom') {
        this.router.navigateByUrl('/tabs/home/actions/custom-form', navigationExtras);
      } else if (actionType === 'breeding') {
        this.router.navigateByUrl('/tabs/home/actions/breeding', navigationExtras);
      }
    }
  }

  // Determine what type of health event the user is recording
  async health() {
    const alert = await this.alertCtrl.create({
      header: 'Health',
      message: 'What type of health event are you recording?',
      buttons: [
        {
          text: 'Observation (No Treatment)',
          cssClass: 'alertGreen',
          handler: () => {
            this.actionInput('health', 'observation');
          }
        },
        {
          text: 'Observation and Treatment',
          cssClass: 'alertGreen',
          handler: () => {
            this.actionInput('health', 'treatment');
          }
        },
        {
          text: 'Cancel',
          role: 'cancel',
          cssClass: 'alertRed'
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  // User is recording a breeding event; in the form they will select the type of event
  async breeding() {
    this.actionInput('breeding', 'breeding');
  }

  // Navigate the user to building a custom form related to apply data to their selected items
  customForm() {
    this.actionInput('custom', 'custom');
  }

  moveToNewGroup() {
    const navigationExtras: NavigationExtras = {
      state: {
        itemType: this.itemType, // itemType --> `animals` or `groups`
        viewType: 'actions',
        passedSelected: this.selected
      }
    };

    // navigate to the add page for multiple animals and pass the current group id
    this.router.navigateByUrl('/tabs/home/actions/manage', navigationExtras);
  }

  ngOnInit() { }

}
