import { Component, OnInit } from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { FormBuilder, FormGroup, Validators, FormControl } from '@angular/forms';
import { LoadingController, ModalController, AlertController } from '@ionic/angular';
import { SelectDamSirePage } from '../select-dam-sire/select-dam-sire.page';
import { AddService } from 'src/app/services/add.service';
import { HttpClient } from '@angular/common/http';
import { DataService } from 'src/app/services/data.service';

@Component({
  selector: 'app-add-animal',
  templateUrl: './add-animal.page.html',
  styleUrls: ['./add-animal.page.scss'],
})
export class AddAnimalPage implements OnInit {

  public reasons = [
    { id: 2, name: 'Reference' },
    { id: 3, name: 'Culled' },
    { id: 4, name: 'Sold' },
    { id: 5, name: 'Died' },
  ];

  public referenceCauses = [
    { id: 2, name: 'A.I. sire' }
  ];

  public culledCauses = [
    { id: 3, name: 'Age' },
    { id: 4, name: 'Appearance' },
    { id: 5, name: 'Body condition' },
    { id: 6, name: 'Calf died before weaning' },
    { id: 7, name: 'Calving late in season' },
    { id: 8, name: 'Conceived but aborted' },
    { id: 9, name: 'Disposition' },
    { id: 10, name: 'Exposed, failed to conceive' },
    { id: 11, name: 'Genetic defect/carrier status' },
    { id: 12, name: 'Herd reduction' },
    { id: 13, name: 'Hoof condition/lameness' },
    { id: 14, name: 'Illness/disease' },
    { id: 15, name: 'Injury or accident' },
    { id: 16, name: 'Poor performance' },
    { id: 17, name: 'Prolapse' },
    { id: 18, name: 'Structure/conformation' },
    { id: 19, name: 'Udder quality' },
    // THE FOLLOWING CAUSES FOR REASON `CULLED` WILL BE ADDED INLINE, CHECK selectChange()
    // Bulls only
    // {id: 39, name: 'Infertility', sex: '1'},
    // Bulls only
    // {id: 40, name: 'Failed BSE', sex: '1'}
  ];

  public soldCauses = [
    { id: 20, name: 'Breeding bull - virgin' },
    { id: 21, name: 'Breeding bull - non-virgin ' },
    { id: 22, name: 'Calf sold with dam' },
    { id: 23, name: 'Embryo donor' },
    { id: 24, name: 'Feeder calf' },
    { id: 25, name: 'Replacement - bred cow ' },
    { id: 26, name: 'Replacement - bred heifer' },
    { id: 27, name: 'Replacement - open heifer' },
    { id: 28, name: 'Replacement - open cow' }
  ];

  public diedCauses = [
    { id: 29, name: 'Birth defect' },
    { id: 30, name: 'Body condition' },
    { id: 31, name: 'Calving difficulty' },
    { id: 32, name: 'Illness' },
    { id: 33, name: 'Injury/accident' },
    { id: 34, name: 'Predation' },
    { id: 35, name: 'Premature calf' },
    { id: 36, name: 'Stillborn calf' },
    { id: 37, name: 'Unknown ' },
    { id: 38, name: 'Weather' }
  ];

  public statusChange = false;
  public reasonChange = false;
  public causes: any = [];
  public status: any;

  public sex: any;
  public addAnimalForm: any;
  public isBreedAvailable: boolean = false;
  public filteredBreeds: any = [];
  public breedSelected: any;
  public amount: any;
  public birthDate: any;
  public purchaseDate: any;
  // Used to trigger show/hide for date options
  public globalAdditionType: any;
  private groupID: any;
  private groupName: any;
  private type: any;
  public additionType: any;
  private prefix: any;
  private prefixNumber: number = 1;
  private formDataList: any = [];
  private breeds: any = [];

  // Initialize sex ID as an empty string; only changes if value is passed to page when adding multiple animals
  private sexID: any;

  constructor(
    private route: ActivatedRoute,
    private router: Router,
    private modalCtrl: ModalController,
    private addService: AddService,
    private alertCtrl: AlertController,
    private httpClient: HttpClient,
    private dataService: DataService,
    public formBuilder: FormBuilder
  ) {
    this.route.queryParams.subscribe(() => {
      if (this.router.getCurrentNavigation()?.extras.state) {
        /*
        NOTE:
        - Parameters: `prefix`, `date`, and `additionType` are only passed when adding multiple animals
        - Parameters: `groupID` and `groupName` are passed when adding individual or multiple animals
        */
        if (this.router.getCurrentNavigation()?.extras.state?.['prefix']) {
          this.prefix = this.router.getCurrentNavigation()?.extras.state?.['prefix'];
          console.log(`Prefix ${this.prefix}`);
        }
        if (this.router.getCurrentNavigation()?.extras.state?.['birthDate']) {
          this.birthDate = this.router.getCurrentNavigation()?.extras.state?.['birthDate'];
          console.log(`Birth Date ${this.birthDate}`);
        }
        if (this.router.getCurrentNavigation()?.extras.state?.['purchaseDate']) {
          this.purchaseDate = this.router.getCurrentNavigation()?.extras.state?.['purchaseDate'];
          console.log(`Purchase Date ${this.purchaseDate}`);
        }
        if (this.router.getCurrentNavigation()?.extras.state?.['additionType']) {
          this.additionType = this.router.getCurrentNavigation()?.extras.state?.['additionType'];
          console.log(`Addition Type ${this.additionType}`);
        }

        this.groupID = this.router.getCurrentNavigation()?.extras.state?.['groupID'];
        console.log(`Group ID ${this.groupID}`);
        this.groupName = this.router.getCurrentNavigation()?.extras.state?.['groupName'];
        console.log(`Group Name ${this.groupName}`);

        /*
        - Check to see if passed sex ID, only set if user is adding multiple animals and does not know
            number of sexes. Helps user by defaulting to Unknown sex for multiple animals
        - Otherwise default to empty string which allows them to set per each
        */
        if (this.router.getCurrentNavigation()?.extras.state?.['sexID']) {
          this.sexID = this.router.getCurrentNavigation()?.extras.state?.['sexID'];
          console.log(`sexID: ${this.sexID}`);
        }

        /*
        * NOTE: Initially set values for each form field to empty string (''),
          instead of null because passing data to server converts null to string ('null')
        */
        this.addAnimalForm = formBuilder.group({
          primaryID: new FormControl('', [Validators.required, Validators.maxLength(255)]),
          sex: new FormControl(this.sexID, Validators.required),
          additionType: new FormControl('', Validators.required),
          birthDate: new FormControl(''),
          purchaseDate: new FormControl(''),
          breed: new FormControl(''),
          breedID: new FormControl(''),
          color: new FormControl(''),
          group: new FormControl(this.groupID),
          weight: new FormControl(''),
          tagColor: new FormControl('', [Validators.maxLength(255)]),
          officialNUES: new FormControl('', [Validators.minLength(8), Validators.maxLength(9)]),
          officialAIN: new FormControl('', [Validators.minLength(15), Validators.maxLength(15)]),
          twinStatus: new FormControl(''),
          calvingVigorScore: new FormControl(''),
          dam: new FormControl(''),
          /*
          * `damID` refers to the id of the animal that the user selected as the dam
              this is not actually shown on the form as a field, acts as a hidden field
          * However, the `dam` field is shown with the animal name for the user to select and change if needed
          */
          damID: new FormControl(''),
          teatScore: new FormControl(''),
          udderScore: new FormControl(''),
          calvingEase: new FormControl(''),
          damBCS: new FormControl(''),
          sire: new FormControl(''),
          /*
          * `sireID` refers to the id of the animal that the user selected as the sire
              this is not actually shown on the form as a field, acts as a hidden field
          * However, the `sire` field is shown with the animal name for the user to select and change if needed
          */
          sireID: new FormControl(''),
          /*
          * Set the status for the current animal to be added
          * 1 = Active, 2 = Inactive
          * Active (1) is default Value
          * Inactive (2) prompts user and requires additional information to be added (reasoning)
          */
          status: new FormControl('1', [Validators.required]),
          // Set reasoning if status is "Inactive" (2)
          reason: new FormControl('1', [Validators.required]),
          // Set cause if reasoning is true due to status being "Inactive" (2)
          cause: new FormControl('1', [Validators.required]),
          notes: new FormControl('', [Validators.maxLength(255)])
        });
      }
    });
  }

  async selectChange(changeType: string, event: any) {
    if (event.detail.value) {
      if (changeType === 'status') {
        this.statusChange = true;
        this.status = +event.detail.value;

        // If the user selects status "Active"; reset values
        if (event.detail.value === '1') {
          this.addAnimalForm.patchValue({ reason: '1' });
          this.addAnimalForm.patchValue({ cause: '1' });
          this.statusChange = false;
          this.reasonChange = false;
          this.causes = null;
        }
        else if (event.detail.value === '2') {
          // Else if, the user select status "Inactive";

          // Set current reason and cause to empty string (string "1" is default for active, needed to toggle Done button)
          this.addAnimalForm.patchValue({ reason: '' });
          this.addAnimalForm.patchValue({ cause: '' });

          // Display prompt when setting animal as "Inactive"
          const alert = await this.alertCtrl.create({
            header: 'Inactive Animal',
            message: 'Animals with "Inactive" status cannot be used to record data. After adding the animal, the status can be changed later under the "Search" tab by tapping "Edit".',
            buttons: ['OK'],
            backdropDismiss: false
          });

          await alert.present();

        }
      } else if (changeType === 'reason') {
        // Else if the user has selected an status "Inactive" and a reason for being inactive
        this.reasonChange = true;
        this.causes = null;

        // Reset the cause value (in case of previously selected reason)
        this.addAnimalForm.patchValue({ cause: '' });

        // If the user selects reason 2 (Reference)
        if (event.detail.value === '2') {
          // Set cause options for 2 (Reference)
          this.causes = this.referenceCauses;
        } else if (event.detail.value === '3') {
          // If the user selects reason 3 (Culled)

          // Retrieve current value for Sex (integer value)
          const currentSex = this.addAnimalForm.get('sex').value;

          /* 
          * Set causes options for 3 (Culled)
          * Use JSON stringify and JSON parse to "deep copy" array of objects
          * Therefore later modifying `this.causes` will not change initial array `this.culledCauses`
          */
          this.causes = JSON.parse(JSON.stringify(this.culledCauses));

          // If the current sex value is 1 (Bull), apply additional Culled causes specific to Bulls only
          if (currentSex === 1) {
            this.causes.push({ id: 39, name: 'Infertility' })
            this.causes.push({ id: 40, name: 'Failed BSE' })
          }

        } else if (event.detail.value === '4') {
          // If the user selects reason 4 (Sold)

          // Set cause options for 4 (Sold)
          this.causes = this.soldCauses;
        } else if (event.detail.value === '5') {
          // If the user selects reason 5 (Died)

          // Set cause options for 5 (Died)
          this.causes = this.diedCauses;
        } else {
          // Else, reset causes to null value (hides select option)
          this.causes = null;
        }

      } else if (changeType === 'sex') {
        /*
        * If the user changes Sex value then reset Reason and Cause
        * This is due to specific causes being added to reason `Culled` for Bulls only
        * Set current reason and cause to string "1" is default for active, needed to toggle Done button
        */
        this.addAnimalForm.patchValue({ reason: '1' });
        this.addAnimalForm.patchValue({ cause: '1' });

        /*
        * Prompt user if they change sex value to "Unknown"
        * Value string "4" refers to user selection of "Unknown" sex
        */
        if (event.detail.value === '4') {
          const alert = await this.alertCtrl.create({
            header: 'Unknown Sex',
            message: 'Animals with "Unknown" sex will not appear in reports that require a sex.',
            buttons: ['OK'],
            backdropDismiss: false
          });

          await alert.present();
        }
      } else if (changeType === 'addition') {
        /*
        * Set addition type for current animal
        * Patch the form value for `additionType` based on user selected option
        */
        console.log(event.detail.value);
        console.log(typeof (event.detail.value));
        this.globalAdditionType = event.detail.value
        this.addAnimalForm.patchValue({
          additionType: event.detail.value,
        });

        // Reset `Optional Details` form field values
        this.resetOptionalDetails();
      }
    }
  }

  // Set breed for current animal
  setBreed(breed: any) {
    if (breed) {
      this.breedSelected = breed;
      this.isBreedAvailable = false;

      // Patch the form value for `breed`
      this.addAnimalForm.patchValue({
        breed: this.breedSelected.name,
        breedID: this.breedSelected.id
      });
    }
  }

  // Grab breeds data from local JSON file
  initializeItems() {
    return this.httpClient.get('/assets/data/breeds.json');
  }

  // When the user begins searching (typing) a breed, filter results from `breeds.json`
  getBreeds(event: any) {
    /*
    * If breeds data has already been called and set into variable
      then only worry about filtering results based on user input
    */
    if (this.breeds) {
      // Set `inputValue` to the value of the search bar
      const inputValue = event.target.value;

      // If `inputValue` is an empty string don't filter the items
      if (inputValue && inputValue.trim() !== '') {
        this.isBreedAvailable = true;
        this.filteredBreeds = this.breeds.filter((item: any) =>
          (item.name.toLowerCase().indexOf(inputValue.toLowerCase()) > -1));
      } else {
        /*
        * Else, the `inputValue` is empty, either there is no input or the user cleared existing input
        * Reset variables
        */
        this.isBreedAvailable = false;
        this.breedSelected = null;
      }
    } else {
      /*
      * Else, no breed data has been called
      * Get data from local `breeds.json` file
      */
      this.initializeItems().subscribe(resp => {
        this.dataService.presentLoading();
        if (resp) {
          this.breeds = resp;
          this.getBreeds(event);
        } else {
          this.isBreedAvailable = false;
        }
      });
    }
  }

  // Display alert if values for Group Name or Group ID is not present
  async missingAlert() {
    const alert = await this.alertCtrl.create({
      header: 'Missing Data',
      message: 'No data for group name or group ID.',
      buttons: ['OK'],
      backdropDismiss: false
    });

    await alert.present();
  }

  // Reset form values after each entry when adding multiple animals
  resetOptionalDetails() {
    this.addAnimalForm.patchValue({
      birthDate: '',
      purchaseDate: '',
      breed: '',
      color: '',
      group: this.groupID,
      weight: '',
      tagColor: '',
      officialNUES: '',
      officialAIN: '',
      twinStatus: '',
      calvingVigorScore: '',
      dam: '',
      damID: '',
      teatScore: '',
      udderScore: '',
      calvingEase: '',
      damBCS: '',
      sire: '',
      sireID: '',
    });
  }

  // When the user taps to select a Dam or Sire
  // Open modal that displays all animals (from all groups) associated to the user's account
  async selectDamSire(animalType: string) {

    // initialize current dam and sire ids to null
    // we do not know if the user has attempted to select a value yet
    let currentDam = null;
    let currentSire = null;

    // if the dam id has a value, then set the current dam value to that
    if (this.addAnimalForm.get('damID').value) {
      currentDam = +this.addAnimalForm.get('damID').value;
    }

    // if the sire id has a value, then set the current dam value to that
    if (this.addAnimalForm.get('sireID').value) {
      currentSire = +this.addAnimalForm.get('sireID').value;
    }

    const modal = await this.modalCtrl.create({
      component: SelectDamSirePage,
      componentProps: {
        selectType: animalType,
        /*
        * Pass the ID of the current animal being viewed so the list of animals presented when selecting
        * the Dam/Sire will not show the selected animals; therefore the user cannot make ana animal it's own Dam/Sire
        */
        currentAnimalID: null,
        currentAnimalDamID: currentDam,
        currentAnimalSireID: currentSire
      }
    });

    /*
    * When the modal is dismissed, see if we have any returned data (user has selected a dam or sire)
    * If there is data, check to see if we are setting the dam or the sire
    * NOTE: The "innerHTML" of the form cannot be set so we cannot set the value
        and the text separately for one form field thus, we have a hidden form field
        that holds the value for the dam/sire while the actual field the user sees only shows the name
    * NOTE: attempted to add this function to the addService file but could not get a return value
    */
    modal.onDidDismiss()
      .then((data: any) => {
        const animal = data.data;
        if (animalType === 'dam') {
          if (animal) {
            this.addAnimalForm.patchValue({
              dam: animal.primary_id,
              damID: +animal.id
            });
          } else {
            this.addAnimalForm.patchValue({
              dam: '',
              damID: ''
            });
          }
        } else if (animalType === 'sire') {
          if (animal) {
            this.addAnimalForm.patchValue({
              sire: animal.primary_id,
              sireID: +animal.id
            });
          } else {
            this.addAnimalForm.patchValue({
              sire: '',
              sireID: ''
            });

          }
        } else {
          this.addAnimalForm.patchValue({
            dam: '',
            damID: '',
            sire: '',
            sireID: ''
          });
        }
      })
      .catch(() => {
        if (animalType === 'dam') {
          this.addAnimalForm.patchValue({
            dam: '',
            damID: ''
          });
        } else if (animalType === 'sire') {
          this.addAnimalForm.patchValue({
            sire: '',
            sireID: ''
          });
        } else {
          this.addAnimalForm.patchValue({
            dam: '',
            damID: '',
            sire: '',
            sireID: ''
          });
        }
      });

    return await modal.present();
  }

  // Cancel adding animal, leave current page view
  cancelAdd() {
    this.addService.cancelAdd('animal', this.groupID, this.groupName);
  }

  next() {
    // Initialize new form data to hold current input data from user
    let formData: any;
    formData = new FormData();

    /*
    * Iterate through form object and append to form data object
    * Do not append `dam`, `sire`, or `breed` values to list since those are only names
    * Only care about the associated ID values (`dam_id`, `sire_id`, `breedID`)
    */
    Object.keys(this.addAnimalForm.controls).forEach(key => {
      if (key === 'dam' || key === 'sire' || key === 'breed') {
        return;
      } else {
        formData.append(key, this.addAnimalForm.get(key).value);
      }
    });

    /*
    * Call resetOptionalDetails() to set optional details form fields to empty string ('')
    * NOTE: reset() function for form does not act as needed, sets string values
        to null and number values to 0 on reset
    */
    this.resetOptionalDetails();

    /*
    * If a prefix is defined, set primary id as prefix with the current number animal
    * Increment prefix number by 1
    */
    if (this.prefix) {
      this.addAnimalForm.patchValue({
        primaryID: `${this.prefix}${this.prefixNumber}`
      });
      this.prefixNumber++;
    }

    // If the user provided a birth date when adding/creating multiple animals, set when click Next
    if (this.birthDate) {
      this.addAnimalForm.patchValue({
        birthDate: `${this.birthDate}`
      });
    }

    // If the user provided a purchase date when adding/creating multiple animals, set when click Next
    if (this.purchaseDate) {
      this.addAnimalForm.patchValue({
        purchaseDate: `${this.purchaseDate}`
      });
    }

    /*
    * Add current form data to form data list (for multiple entries)
    * "Next" button will only appear on pages when there is more than 1 entry
    */
    this.formDataList.push(formData);

    console.log(`next() formDataList: ${this.formDataList}`);

    /*
    * Reduce amount of items we are adding to determine when we are done
    * `amount` = 1 signals last animal to add (swap button "Next" to "Done")
    */
    this.amount--;

    // show loading spinner in-between forms for additional animal entry
    this.dataService.presentLoading();
  }

  /* When the user is done adding items, check the type (individual/multiple)
  * If individual, send form data and upload to server
  * Else if multiple, iterate through each form data instance and upload to
      server one at a time (find a better way to do this)
  */
  done() {
    console.log('done()');
    let formData: any;
    formData = new FormData();

    console.log(`done() groupID: ${this.groupID}`);
    console.log(`done() groupName: ${this.groupName}`);

    if (this.type === 'individual') {
      /*
      * Iterate through form object and append to form data object
      * Do not append `dam`, `sire`, or `breed` values to list since those are only names
      * Only care about the associated ID values (`dam_id`, `sire_id`, `breedID`)
      */
      Object.keys(this.addAnimalForm.controls).forEach(key => {
        console.log(`Key: ${key}, Value: ${this.addAnimalForm.get(key).value}`);
        if (key === 'dam' || key === 'sire' || key === 'breed') {
          return;
        } else {
          formData.append(key, this.addAnimalForm.get(key).value);
        }
      });

      // call our service that inserts an individual record into database
      this.addService.addIndividual('animal', this.groupID, this.groupName, formData);
    } else if (this.type === 'multiple') {
      /*
      * Iterate through form object and append to form data object
      * Do not append `dam`, `sire`, or `breed` values to list since those are only names
      * Only care about the associated ID values (`dam_id`, `sire_id`, `breedID`)
      */
      Object.keys(this.addAnimalForm.controls).forEach(key => {
        if (key === 'dam' || key === 'sire' || key === 'breed') {
          return;
        } else {
          formData.append(key, this.addAnimalForm.get(key).value);
        }
      });

      /* Push our last animal to be added into our form data list which contains
          all previous animals from multiple entry
      */
      this.formDataList.push(formData);
      console.log(this.formDataList);

      // call our service that inserts multiple records into database
      this.addService.addMultiple('animal', this.groupID, this.groupName, this.formDataList);
    }
  }

  /*
  * When the page enters current view
  * If we are adding multiple items, prompt user for prefix
  */
  async ionViewWillEnter() {
    if (this.type === 'multiple' && this.amount > 1) {
      // const result = await this.addService.askPrefix();

      // // If the prefix from the result has a value
      // if (result.prefix) {
      //   this.prefix = result.prefix;
      // } else {
      //   // Else, set default prefix string
      //   this.prefix = 'DefaultAnimalName';
      // }

      // // If the starting number from the prefix result has a value
      // if (result.startingNumber) {
      //   /*
      //   * Check to see if the prefix starting number is a number
      //   * If so, set number; otherwise use default value of 1
      //   */
      //   if (!isNaN(result.startingNumber)) {
      //     this.prefixNumber = +result.startingNumber;
      //   }
      // }

      console.log(`addAnimalPage ionViewWillEnter prefix: ${this.prefix}${this.prefixNumber}`);
      console.log(`addAnimalPage ionViewWillEnter additionType: ${this.additionType}`);
      console.log(`addAnimalPage ionViewWillEnter sex: ${this.sexID}`);
      console.log(`addAnimalPage ionViewWillEnter birthDate: ${this.birthDate}`);
      console.log(`addAnimalPage ionViewWillEnter purchaseDate: ${this.purchaseDate}`);

      this.sex = this.sexID;

      // Set value for animal primary id with defined prefix and starting prefix number
      this.addAnimalForm.patchValue({
        primaryID: `${this.prefix}${this.prefixNumber}`,
        additionType: this.additionType,
        sex: this.sexID,
        birthDate: this.birthDate,
        purchaseDate: this.purchaseDate
      });

      // Increase prefix number by 1
      this.prefixNumber++;
    }
  }

  ngOnInit() {
    /*
    * Check to see we received passed parameters
      - "type" = single or multiple
      - "amount" = integer value (1, 2, 3, etc.)
    * If so, set variables otherwise navigate back to home page
    */
    if (this.route.snapshot.paramMap.get('type')
      && this.route.snapshot.paramMap.get('amount')) {
      this.type = this.route.snapshot.paramMap.get?.('type');
      this.amount = this.route.snapshot.paramMap.get?.('amount');
      this.amount = +this.amount;
    } else {
      this.router.navigateByUrl('/tabs/home');
    }
  }
}
