import { Component, OnInit } from '@angular/core';
import { DataService } from 'src/app/services/data.service';
import { AddService } from 'src/app/services/add.service';
import { ActivatedRoute, Router } from '@angular/router';
import { FormBuilder, FormGroup, Validators, FormControl } from '@angular/forms';
import { ModalController } from '@ionic/angular';
import { CategoriesPage } from '../categories/categories.page';
import { Geolocation } from '@capacitor/geolocation';

@Component({
  selector: 'app-add-group',
  templateUrl: './add-group.page.html',
  styleUrls: ['./add-group.page.scss'],
})
export class AddGroupPage implements OnInit {
  public type: any;
  public amount: any;
  public prefix: string = '';
  public prefixNumber: number = 1;
  public addGroupForm: FormGroup;
  // List to hold multiple group FormData
  public formDataList: any = [];

  constructor(
    private dataService: DataService,
    private addService: AddService,
    private route: ActivatedRoute,
    private router: Router,
    public formBuilder: FormBuilder,
    private modalCtrl: ModalController,
  ) {
    // Initialize form group group containing fields for form on the page
    this.addGroupForm = formBuilder.group({
      groupName: new FormControl('', [Validators.required, Validators.maxLength(255)]),
      groupCategory: new FormControl('', Validators.required),
      groupCategoryID: new FormControl('', Validators.required),
      locationName: new FormControl('', [Validators.maxLength(255)]),
      latitude: new FormControl('', [Validators.maxLength(10)]),
      longitude: new FormControl('', [Validators.maxLength(10)]),
      groupDescription: new FormControl('', [Validators.maxLength(255)])
    });
  }

  // Cancel adding group, leave current page view
  cancelAdd() {
    // Parameters --> viewType: string, groupID: number, groupName: string
    this.addService.cancelAdd('group', null, null);
  }

  // Reset `addGroup` form values after each entry when adding multiple groups
  resetForm() {
    this.addGroupForm.setValue({
      groupName: '',
      groupCategory: '',
      groupCategoryID: '',
      locationName: '',
      latitude: '',
      longitude: '',
      groupDescription: ''
    });
  }

  // Get the current location of the user (for setting new group location)
  async getCurrentLocation() {
    const coordinates = await Geolocation.getCurrentPosition();
    if (coordinates) {
      this.addGroupForm.patchValue({
        latitude: +coordinates.coords.longitude,
        longitude: +coordinates.coords.longitude
      });
    }
  }

  /*
  * Check to length of the description for a group
  * TODO: Add check length/type/etc for latitude, longitude
  */
  checkLength() {
    if (this.addGroupForm.get('groupDescription')?.value.length > 255) {
      this.addService.displayAlert('Error Adding Group',
        `The maximum amount of characters for a group description is 255.`
        + ` You have ${this.addGroupForm.get('groupDescription')?.value.length}. Please try again.`);
    }
  }

  /*
  * Present modal that shows the user's group categories in a list they can select from
  * Once the user has dismissed the modal, check to see if any data is returned (category)
  * If so, set appropriate others, otherwise, set to empty string
  */
  async groupCategories(view: string) {
    const modal = await this.modalCtrl.create({
      component: CategoriesPage,
      componentProps: {
        viewType: view
      }
    });

    modal.onDidDismiss()
      .then((data: any) => {
        if (data.data) {
          const category = data.data;
          this.addGroupForm.patchValue({
            groupCategory: category.category_name,
            groupCategoryID: +category.id
          });

        } else {
          this.addGroupForm.patchValue({
            groupCategory: '',
            groupCategoryID: ''
          });

        }
      });

    return await modal.present();
  }

  next() {
    // Initialize new form data to hold current input data from user
    const formData = new FormData();

    Object.keys(this.addGroupForm.controls).forEach(key => {
      if (key !== 'groupCategory') {
        formData.append(key, this.addGroupForm.get(key)?.value);
      }
    });

    // Reset form values
    this.resetForm();

    /*
    * If a prefix is defined
    * Set group name as prefix with the current number group
    * Increase prefix number
    */
    if (this.prefix) {
      this.addGroupForm.patchValue({
        groupName: `${this.prefix}${this.prefixNumber}`
      });
      this.prefixNumber++;
    }

    /*
    * Add current form data to form data list (for multiple entries)
    * `Next` will only appear on pages when there is more than 1 (one) entry
    */
    this.formDataList.push(formData);

    /*
    * Reduce amount of items we are adding to determine when we are done
    * amount = 1 (one) signals last group to add (swap button `Next` to `Done`)
    */
    this.amount--;

    // Show loading spinner to compensate for loading next form for additional group entry
    this.dataService.presentLoading();
  }


  /*
  * When the user is done adding items, check the type (individual/multiple)
  * If individual, send form data and upload to server
  * Else if multiple, iterate through each form data instance and upload to server one at a time (find a better way to do this)
  */
  done() {
    this.checkLength();
    const formData = new FormData();

    if (this.type === 'individual') {
      /*
      * If adding an individual (single) group
      * Iterate through form object
      * Append form data, except for `groupCategory` key
      */
      Object.keys(this.addGroupForm.controls).forEach(key => {

        if (key !== 'groupCategory') {
          formData.append(key, this.addGroupForm.get(key)?.value);
        }
      });

      /*
      * Call Add service that inserts an individual record into database
      * Parameters --> category: string, groupID: number, groupName: string, formData: FormData
      */
      this.addService.addIndividual('group', null, null, formData);

    } else if (this.type === 'multiple') {
      /*
      * Else if, adding multiple groups
      * Iterate through our last form object
      * Append form data, except for `groupCategory` key
      */
      Object.keys(this.addGroupForm.controls).forEach(key => {
        if (key !== 'groupCategory') {
          formData.append(key, this.addGroupForm.get(key)?.value);
        }
      });

      // Put last adding group into form data list (list of other group form data)
      this.formDataList.push(formData);

      /*
      * Call Add service that inserts multiple records into database
      * Parameters --> category: string, groupID: number, groupName: string, formDataList: any
      */
      this.addService.addMultiple('group', null, null, this.formDataList);
    }
  }


  /*
  * When the page enters view, check passed parameters
  * If the passed parameters have value, set variables, otherwise navigate back to home page
  * If the user is adding multiple items, prompt user to set prefix (optional)
  */
  async ionViewWillEnter() {
    if (this.route.snapshot.paramMap.get('type')
      && this.route.snapshot.paramMap.get('amount')) {
      this.type = this.route.snapshot.paramMap.get('type');
      this.amount = this.route.snapshot.paramMap.get?.('amount');
      this.amount = +this.amount;

      /*
      * If the user is adding multiple items and they have entered an amount of item greater than 1 (one)
      * Ask if the user wants to define a prefix for the multiple item names
      */
      if (this.type === 'multiple' && this.amount > 1) {
        const result = await this.addService.askPrefix();

        // If there is a defined prefix, set it
        if (result.prefix) {
          this.prefix = result.prefix;

          // If starting number for prefix is defined, set value, otherwise default to 1
          if (result.startingNumber) {

            /*
            * Check to see if the prefix starting number is a number
            * If so, set number; otherwise use default value of 1 (one)
            */
            if (!isNaN(result.startingNumber)) {
              this.prefixNumber = +result.startingNumber;
            }
          }
        } else {
          /*
          * Else, the user elected to define a prefix, but submitted an empty string ''
          * Set the prefix to a default placeholder for the group name
          */
          this.prefix = 'DefaultGroupName';
        }

        // Set value for group name with defined prefix and starting prefix number
        this.addGroupForm.controls['groupName'].setValue(`${this.prefix}${this.prefixNumber}`);

        // Increase prefix number by 1 (one)
        this.prefixNumber++;
      }
    } else {
      this.router.navigateByUrl('/tabs/home');
    }
  }

  ngOnInit() {
  }

}
