import { Component, OnInit } from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { FormBuilder, FormGroup, Validators, FormControl } from '@angular/forms';
import { DataService } from 'src/app/services/data.service';
import { ActionsService } from 'src/app/services/actions.service';
import { ModalController } from '@ionic/angular';
import { SelectBullPage } from '../select-bull/select-bull.page';

@Component({
  selector: 'app-breeding',
  templateUrl: './breeding.page.html',
  styleUrls: ['./breeding.page.scss'],
})
export class BreedingPage implements OnInit {
  public actionID = 0;
  public action: string = '';
  public actionType: string = '';
  public breedingForm: FormGroup;
  public bulls: any;
  public requiredBull = false;
  private actionTypeID = 5;
  private itemType: string = '';
  private selected: any;
  private selectedAmount: string = '';
  private cows: any;
  private cowIDs: any = [];

  constructor(
    public formBuilder: FormBuilder,
    private route: ActivatedRoute,
    private router: Router,
    private dataService: DataService,
    private actionsService: ActionsService,
    private modalCtrl: ModalController
  ) {
    this.route.queryParams.subscribe(_ => {
      if (this.router.getCurrentNavigation()) {
        if (this.router.getCurrentNavigation()?.extras.state) {
          // `animals` or `groups`
          this.itemType = this.router.getCurrentNavigation()?.extras.state?.['itemType'];
          // e.g., `breeding`
          this.actionType = this.router.getCurrentNavigation()?.extras.state?.['actionType'];
          // e.g. `natural` or `ai`
          this.action = this.router.getCurrentNavigation()?.extras.state?.['action'];
          // animals or groups selected by the user
          this.selected = this.router.getCurrentNavigation()?.extras.state?.['selected'];
          // `single` or `multiple`
          this.selectedAmount = this.router.getCurrentNavigation()?.extras.state?.['selectedAmount'];

          if (this.selected) {
            if (this.itemType === 'animals') {
              // Filter all bulls from selected animals
              this.bulls = this.selected.filter((s: any) => s.sex_id === 1);

              // Filter all cows from selected animals
              this.cows = this.selected.filter((s: any) => s.sex_id === 3);

              // Iterate through filtered cows and append their ID to new list
              // IDs are used when processing data for database record insert
              for (const c of this.cows) {
                this.cowIDs.push(c.id);
              }
            }
          }
        }
      }
    });

    this.breedingForm = formBuilder.group({
      bull: new FormControl('', Validators.required),
      bullID: new FormControl(''),
      eventType: new FormControl('', Validators.required),
      date: new FormControl('', Validators.required),
      technician: new FormControl(''),
      notes: new FormControl('')
    });
  }

  /*
  * When the user taps to select a Bull
  * Open modal that displays all bulls (from all groups) associated to the user's account
  */
  async selectBull() {
    const modal = await this.modalCtrl.create({
      component: SelectBullPage,
    });

    /*
    * When the modal is dismissed, see if we have any returned data (user has selected a bull)
    * If there is data, check to see if we are setting the bull
    * NOTE: The "innerHTML" of the form cannot be set so we cannot set the value and the
        text separately for one form field thus, we have a hidden form field that holds the
        value for the bull while the actual field the user sees only shows the name
    * NOTE: attempted to add this function to the addService file but could not get a return value
    */
    modal.onDidDismiss()
      .then((data: any) => {
        const bull = data.data;
        if (bull) {
          this.requiredBull = true;

          this.breedingForm.patchValue({
            bull: bull.primary_id,
            bullID: +bull.id
          });
        } else {
          this.requiredBull = false;

          this.breedingForm.patchValue({
            bull: '',
            bullID: ''
          });
        }
      })
      .catch(() => {
        this.requiredBull = false;
        this.breedingForm.patchValue({
          bull: '',
          bullID: ''
        });
      });

    return await modal.present();
  }

  /*
  * If the user decides to cancel applying actions to their selected items
  * Navigate back to the main `Home` page
  */
  cancelActions() {
    this.router.navigateByUrl('/tabs/home');
  }

  /*
  * If the user taps "Done" (required data entered), continue with processing the form
  */
  done(action: string) {
    // Initialize variables
    let formDictionary: any = {};
    let form: any = this.breedingForm;

    // Iterate through the form data and add the key/value pairings to the dictionary
    Object.keys(form.controls).forEach(key => {
      formDictionary[key] = form.get(key).value;
    });

    // If the formDictionary has data, continue with making call and passing data for database record insertion
    if (formDictionary) {
      // EXAMPLE formDictionary: (5, 'natural', 'animals', [31], 'single', {formData})

      for (const [key, value] of Object.entries(formDictionary)) {
        console.log(key, value);
      }

      /*
      * Else, the user navigated to the Health page normally via Actions page
      * Call data service to insert Health event into database
      */
      this.dataService.addBreedingEvent(this.actionTypeID, action,
        this.itemType, this.cowIDs, this.selectedAmount, formDictionary)
        .then((response: any) => {
          /*
          * If there is a response (successful), then reset the list
          * Display message to user and navigate back to Actions page
          */
          if (response) {
            this.actionsService.actionToast('breeding', this.itemType);
            this.router.navigateByUrl('/tabs/home');
          } else {
            // Else, there is not a response, display error message to user
            this.actionsService.actionError('Actions Breeding Event Response Error');
          }
        })
        .catch((error: any) => {
          // Else, there is an unexpected error, display error message to user
          this.actionsService.actionError(error);
        });
    }
  }

  ngOnInit() {
  }

}
