import { Component, OnInit } from '@angular/core';
import { DataService } from 'src/app/services/data.service';
import { AddService } from 'src/app/services/add.service';
import { ModalController, AlertController } from '@ionic/angular';

@Component({
  selector: 'app-categories',
  templateUrl: './categories.page.html',
  styleUrls: ['./categories.page.scss'],
})
export class CategoriesPage implements OnInit {
  public categories: any;
  public selected: any;
  public filteredCategories: any;
  public viewType: any;
  private categoryToDelete: any;

  constructor(
    private dataService: DataService,
    private addService: AddService,
    private modalCtrl: ModalController,
    private alertCtrl: AlertController
  ) { }

  /*
  * Prompt the user about the action they are about to do
  * Make sure the user understands they are changing the category for affected groups
      and they are deleting the previously selected category
  */
  async confirmModify(newCategory: any) {
    const alert = await this.alertCtrl.create({
      header: 'Confirm Category Change',
      message: `Are you sure you want to update the groups with the category "${this.categoryToDelete.category_name}" to the category "${newCategory.category_name}". This will replace the current category for those groups with the new category you have selected. This will also delete the category "${this.categoryToDelete.category_name}".`,
      buttons: [
        {
          text: 'Cancel',
          role: 'cancel',
          cssClass: 'alertRed',
        },
        {
          text: 'Confirm',
          cssClass: 'alertGreen',
          handler: () => {
            // Parameters --> category(name: string, category: any, categoryRename: string, newCategoryID: number)
            this.dataService.category('updateGroupCategory', this.categoryToDelete, null, newCategory)
              .then((resp: any) => {
                if (resp) {
                  // Dismiss the current Categories view which the user used to update the category
                  this.dismiss();
                } else {
                  this.addService.displayAlert('Error Updating Category',
                    'An error occurred while trying to update group category. Check your network connection and try again.');
                }
              })
              .catch(() => {
                this.addService.displayAlert('Error Updating Category',
                  'An error occurred while trying to update group category. Check your network connection and try again.');
              });
          }
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  /*
  * Dismiss the Categories modal page
  * Return the selected category
  */
  done() {
    this.modalCtrl.dismiss(this.selected);
  }

  /*
  * Dismiss the categories modal
  * User did not select a category
  */
  dismiss() {
    this.modalCtrl.dismiss();
  }

  // Present modal that shows the user's group categories in a list they can view and edit
  async viewCategories(view: string, category: any) {
    const modal = await this.modalCtrl.create({
      component: CategoriesPage,
      componentProps: {
        viewType: view,
        categoryToDelete: category
      }
    });

    /*
    * Once the user is 'Done' (dismisses) the 'Group Categories' modal where they are selecting the new category
        to update for groups, re-enter the view from the initial modal to update the list of categories to reflect
        the deletion of the initial selected category
    */
    modal.onDidDismiss()
      .then(() => {
        this.ionViewDidEnter();
      });

    return await modal.present();
  }

  /*
  * Allow the user to automatically modify the groups with the category they are trying to delete
  * Call the server and replace (update) the category string for all groups of the category the user wants to delete
  */
  async updateGroupCategory() {
    const alert = await this.alertCtrl.create({
      header: 'Cannot Modify Category',
      message: `You have groups that are using the category "${this.selected.category_name}". Since a category is required for a group, you must update all groups that contain the category you are wanting to delete. Tap "Cancel" to do this manually for each group. Tap "Update" to add a new category or select an existing one from your list of categories to apply to those groups.`,
      buttons: [
        {
          text: 'Cancel',
          role: 'cancel',
          cssClass: 'alertRed',
          handler: () => {
            // "un-check" the selected category
            this.selected.isChecked = false;
            this.selected = null;
          }
        },
        {
          text: 'Update',
          cssClass: 'alertGreen',
          handler: () => {
            this.viewCategories('update-category', this.selected);
          }
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  modifyCategory(categoryAction: any, category: any, rename: any) {
    // Parameters --> category(name: string, category: any, categoryRename: string, newCategoryID: number)
    this.dataService.category('checkGroupCategory', category, null, null)
      .then((response: any) => {
        if (response >= 0) {
          if (categoryAction === 'delete') {
            /*
          * If there are any groups with this category
          * Inform the user that they cannot delete the group until all groups
              with the category have been modified to a new category
          * Otherwise, animals will not be associated with a category
          */
            if (response > 0) {
              this.updateGroupCategory();
            } else if (response === 0) {
              // Parameters --> category(name: string, category: any, categoryRename: string, newCategoryID: number)
              this.dataService.category('deleteGroupCategory', category, null, null)
                .then((resp: any) => {
                  // If a response from the server was received, then a group category was deleted
                  if (resp) {
                    // Refresh view to get updated list of categories
                    this.ionViewDidEnter();
                  } else {
                    this.addService.displayAlert('Error Deleting Category',
                      'An error occurred while trying to delete group category. Check your network connection and try again.');
                  }
                })
                .catch(() => {
                  this.addService.displayAlert('Error Deleting Category',
                    'An error occurred while trying to delete group category. Check your network connection and try again.');
                });
            } else {
              this.addService.displayAlert('Unknown Modify Category',
                'An error occurred while trying to modify group category. Check your network connection and try again.');
            }
          } else if (categoryAction === 'rename') {
            // Parameters --> category(name: string, category: any, categoryRename: string, newCategoryID: number)
            this.dataService.category('renameGroupCategory', category, rename, null)
              .then((resp: any) => {
                // If a response from the server was received, then a group category was renamed
                if (resp) {
                  // Refresh view to get updated list of categories
                  this.ionViewDidEnter();
                } else {
                  this.addService.displayAlert('Error Renaming Category',
                    'An error occurred while trying to delete group category. Check your network connection and try again.');
                }
              })
              .catch(() => {
                this.addService.displayAlert('Error Renaming Category',
                  'An error occurred while trying to delete group category. Check your network connection and try again.');
              });
          }
        } else {
          this.addService.displayAlert('Error Modifying Category',
            'An error occurred while trying to modify group category. Check your network connection and try again.');
        }
      })
      .catch(() => {
        this.addService.displayAlert('Error Modifying Category',
          'An error occurred while trying to modify group category. Check your network connection and try again.');
      });
  }

  /*
  * Prompt the user to delete their selected category
  * Check to see that the category they are wanting to delete does not belong to any existing groups
  */
  async promptModifyCategory(categoryAction: any, category: any, rename: any) {
    let alert: any;

    if (categoryAction === 'delete') {
      alert = await this.alertCtrl.create({
        header: 'Delete Group Category',
        message: `Are you sure you want to delete group category "<b>${category.category_name}</b>"?`,
        buttons: [
          {
            text: 'Cancel',
            role: 'cancel',
            cssClass: 'alertGreen',
          },
          {
            text: 'Confirm',
            cssClass: 'alertRed',
            handler: () => {
              this.modifyCategory(categoryAction, category, null);
            }
          }
        ],
        backdropDismiss: false
      });
    } else if (categoryAction === 'rename') {
      alert = await this.alertCtrl.create({
        header: 'Rename Group Category',
        message: `Are you sure you want to rename group category "<b>${category.category_name}</b>" to "<b>${rename}</b>"?`,
        buttons: [
          {
            text: 'Cancel',
            role: 'cancel',
            cssClass: 'alertRed'
          },
          {
            text: 'Confirm',
            cssClass: 'alertGreen',
            handler: () => {
              this.modifyCategory(categoryAction, category, rename);
            }
          }
        ],
        backdropDismiss: false
      });
    }

    await alert.present();
  }

  // Prompt the user for input in renaming a category
  async renameCategory(category: any) {
    const alert = await this.alertCtrl.create({
      header: 'Rename Category',
      message: 'Enter the new name for the category',
      inputs: [
        {
          name: 'rename',
          type: 'text',
          placeholder: 'Enter new category name',
          min: 1,
          max: 255
        }
      ],
      buttons: [
        {
          text: 'Cancel',
          role: 'cancel',
          cssClass: 'alertRed'
        },
        {
          text: 'OK',
          cssClass: 'alertGreen',
          handler: (data) => {
            // Initialize new variable equal to the input from the user, and trim any whitespace
            const renameCategory = data.rename.trim();

            if (renameCategory && renameCategory.length > 0 && renameCategory.length <= 255) {
              this.promptModifyCategory('rename', category, renameCategory);
            } else {
              // Else, `renameCategory` is not between 1 to 255 characters, display error
              this.addService.displayAlert('Error Renaming Category',
                `You must enter a category name that has 1-255 characters.`
                + ` You have ${renameCategory.length}. Please try again.`);
            }
          }
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  checkDuplicate(categoryName: string) {
    /*
    * Check to see if the category name from user input matches any current category names
    * If so, alert the user that they have a category with the desired name, input another
    */
    for (const cat of this.categories) {
      if (cat.category_name.toLowerCase() === categoryName.toLowerCase()) {
        return true
      }
    }
    return false;
  }

  // Allow the user to add a new category (inserts record into database)
  async addNewCategory() {
    const alert = await this.alertCtrl.create({
      header: 'Add New Group Category',
      message: 'Enter the name of your new group category',
      inputs: [
        {
          name: 'categoryName',
          type: 'text',
          placeholder: 'Enter category name',
          min: 1,
          max: 255
        },
      ],
      buttons: [
        {
          text: 'Cancel',
          role: 'cancel',
          cssClass: 'alertRed'
        },
        {
          text: 'Create',
          cssClass: 'alertGreen',
          handler: (data) => {
            // Initialize new variable equal to the input from the user; trim whitespace
            const inputCategoryName = data.categoryName.trim();

            /*
            * If there is an `inputCategoryName` with a length between 1 and 255
            * Prompt user to confirm they want to change the category
            */
            if (inputCategoryName && inputCategoryName.length > 0 && inputCategoryName.length <= 255) {

              // if we have a category we are deleting
              if (this.categoryToDelete) {

                /*
                * Check to make sure the user is not re-defining the same category that they are trying to delete
                * Make both names lowercase to check all possible cases
                */
                if (this.categoryToDelete.category_name.toLowerCase() === inputCategoryName.toLowerCase()) {
                  this.addService.displayAlert('Category Already Exists',
                    'You cannot create the same group category name before it is deleted!');
                  return;
                }
              }

              // Check to see if the category name from user input already exists in the list of categories for the user
              if (this.checkDuplicate(inputCategoryName)) {
                this.addService.displayAlert('Category Already Exists',
                  'The category name you tried to create already exists. Please enter a different category name.');

              } else {
                // Check to see that the length of the category name has at least 1 character and less than 255 characters
                if (inputCategoryName.length > 0 && inputCategoryName.length <= 255) {

                  // Parameters --> category(name: string, category: any, categoryRename: string, newCategoryID: number)
                  this.dataService.category('addGroupCategory', null, inputCategoryName, null)
                    .then((response: any) => {

                      if (response === 'success') {
                        /*
                        * Refresh the view to grab the new category
                        * Makes call to database to get current list of this.categories
                        */
                        this.ionViewDidEnter();

                      } else {
                        // Else, the response was not `success`
                        this.addService.displayAlert('Error Adding Category',
                          'An error occurred while trying to add group category. Check your network connection and try again.');
                      }
                    })
                    .catch(() => {
                      this.addService.displayAlert('Error Adding Category',
                        'An error occurred while trying to add group category. Check your network connection and try again.');
                    });
                } else {
                  // Else, the `inputCategoryName` has 0 characters or exceeds 255 characters
                  this.addService.displayAlert('Error Adding Category',
                    `You must enter a category name that has 1-255 characters.`
                    + ` You have ${inputCategoryName.length}. Please try again.`);
                }
              }
            } else {
              // Else, the user did not input a category name
              this.addService.displayAlert('Error Adding Category',
                `You must enter a category name that has 1-255 characters.`
                + ` You have ${inputCategoryName.length}. Please try again.`);
            }
          }
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  checkboxChanged(category: any) {
    // If the current checkbox was previously selected
    if (this.selected && category.id === this.selected.id) {
      if (category.isChecked) {
        // The checkbox was re-selected, do nothing (since the state is already updated)
      } else {
        // The checkbox was deselected
        this.selected = null;
      }
    } else {
      // A new checkbox was selected
      if (this.selected) {
        // Deselect the previous checkbox
        this.selected.isChecked = false;
      }
      this.selected = category;  // Set the new checkbox as selected
    }
    console.log(this.selected);
  }

  // Filter group categories based on group category name
  filterCategories(event: any) {
    this.filteredCategories = this.categories.filter((cat: any) =>
      cat.category_name.toLowerCase().indexOf(event.detail.value.toLowerCase()) > -1);
  }

  /*
  * When the page enters the view
  * Grab the categories from the database and display in list
  */
  ionViewDidEnter() {
    /*
    * If there was a selected item and the view was refreshed
    * Deselect that item
    */
    if (this.selected) {
      this.selected.isChecked = false;
      this.selected = null;
    }

    // Retrieve the group categories for the current user
    this.dataService.get('getGroupCategories', null)
      .then((data: any) => {
        // If there is a result and the length is greater than 0
        if (data && data.length > 0) {
          this.dataService.presentLoading();

          // Set categories from received data
          this.categories = data;

          /*
          * If the user has opened the categories (modal) page again to update, then
              remove the previous selected category that is going to be deleted after the user
          * Selects (and adds) a new category
          */
          if (this.viewType === 'update-category') {
            if (this.categoryToDelete) {
              const sireIndex = this.categories.findIndex((d: any) => d.id === this.categoryToDelete.id);
              this.categories.splice(sireIndex, 1);
            }
          }
        }

        // Set filtered categories equal to categories;
        this.filteredCategories = this.categories;
      })
      .catch((error: any) => {
        console.log(`Categories getGroupCategories Error: ${error}`);
        this.addService.displayAlert('Group Categories Error',
          'An error occurred while retrieving group categories. Check your network connection and try again.');
        this.filteredCategories = this.categories;
      });
  }

  ngOnInit() { }

}
