import { Component, OnInit } from '@angular/core';
import { DataService } from 'src/app/services/data.service';
import { ActionsService } from 'src/app/services/actions.service';
import { FormBuilder, FormGroup, FormControl } from '@angular/forms';
import { AlertController, ModalController } from '@ionic/angular';
import { ActivatedRoute, Router } from '@angular/router';
import { HealthPage } from '../health/health.page';
import { VaccinationPage } from '../vaccination/vaccination.page';
import { AnimalChangePage } from '../animal-change/animal-change.page';


@Component({
  selector: 'app-custom-form',
  templateUrl: './custom-form.page.html',
  styleUrls: ['./custom-form.page.scss'],
})
export class CustomFormPage implements OnInit {

  // Initialize to hold form builder group
  public customForm: any;

  // Initialize flag related to the form being confirmed (built)
  // The flag is default false and remains false until the user has selected initial option(s) and/or additional option(s)
  // from the initial screen. Once they have selected options, then the "Confirm" button is enabled and they can proceed with their custom form
  public confirmFlag = false;

  // Initial options for the user to input data for selected items under Custom Processing Form
  // Depending on the type of animal(s) selected, some options may or may not be shown
  public initialOptions = [
    { value: 'Weight', isChecked: false, type: 'all' },
    { value: 'Hip Height', isChecked: false, type: 'all' },
    { value: 'Body Condition Score', isChecked: false, type: 'all' },
    { value: 'Reproductive Tract Score', isChecked: false, type: 'cow' },
    { value: 'Chute Docility Score', isChecked: false, type: 'all' },
    { value: 'Foot Angle Scoring', isChecked: false, type: 'all' },
    { value: 'Claw Set Score', isChecked: false, type: 'all' },
    { value: 'Teat Score', isChecked: false, type: 'cow' },
    { value: 'Udder Score', isChecked: false, type: 'cow' },
    { value: 'Breeding Soundness Exam', isChecked: false, type: 'bull' },
    { value: 'Pregnancy Diagnosis', isChecked: false, type: 'cow' },
    { value: 'Castration', isChecked: false, type: 'bull' },
    { value: 'Dehorning', isChecked: false, type: 'all' },
    { value: 'Spaying', isChecked: false, type: 'cow' },
    { value: 'Body Temperature', isChecked: false, type: 'all' }
  ];

  // List to holder initial options which have been filtered based on animal type (cow, bull, all)
  public filteredInitialOptions: any = [];

  // List of animal objects which have been selected by the user from the Record Data list view
  public selectedAnimals: any[] = [];
  // e.g. 
  // [
  //    { id: 1, name: 'Animal1', selectedTreatments: [], selectedVaccinations: [] },
  //    { id: 2, name: 'Animal2', selectedTreatments: [], selectedVaccinations: [] },
  //    ...
  // ]

  // Hard-coded string value for custom form
  public actionType: string = "custom";

  // Corresponds to the ID of action for a custom form within the database
  public actionTypeID: number = 4;

  // Hard-coded string value for custom form
  public action: string = "custom"

  // String value representing if the user selected a single animal or multiple animals, "single" or "multiple"
  public selectedAmount: string = "";

  // Flag to determine if the Select All button for the Initial Options has been selected by the user
  // False is default value, therefore nothing has been selected. If the user taps, then value changes to true and all items
  // currently in filteredInitialOptions is selected (isChecked = true)
  public initialSelectAllFlag: boolean = false;

  // Flag to determine if the Select All button for the Additional Options has been selected by the user
  // False is default value, therefore nothing has been selected. If the user taps, then value changes to true and all items
  public additionalSelectAllFlag: boolean = false;

  // Additional options where the user can select up to four (4) treatments and four (4) vaccinations
  public additionalOptions = [
    {
      value: 'Treatment 1', initialChecked: false, isChecked: false, type: 'treatment', text: '', index: -1,
      data: {
        date: "", volume: "", units: "", product: "", productOther: "", route: "", reason: "", system: "", condition: "", weight: "", notes: ""
      }
    },
    {
      value: 'Treatment 2', initialChecked: false, isChecked: false, type: 'treatment', text: '', index: -1,
      data: {
        date: "", volume: "", units: "", product: "", productOther: "", route: "", reason: "", system: "", condition: "", weight: "", notes: ""
      }
    },
    {
      value: 'Treatment 3', initialChecked: false, isChecked: false, type: 'treatment', text: '', index: -1,
      data: {
        date: "", volume: "", units: "", product: "", productOther: "", route: "", reason: "", system: "", condition: "", weight: "", notes: ""
      }
    },
    {
      value: 'Treatment 4', initialChecked: false, isChecked: false, type: 'treatment', text: '', index: -1,
      data: {
        date: "", volume: "", units: "", product: "", productOther: "", route: "", reason: "", system: "", condition: "", weight: "", notes: ""
      }
    },
    {
      value: 'Vaccination 1', initialChecked: false, isChecked: false, type: 'vaccination', text: '', index: -1,
      data: {
        date: "", expirationDate: "", route: "", serialNumber: "", units: "", vaccine: "", vaccineOther: "", volume: "",
      }
    },
    {
      value: 'Vaccination 2', initialChecked: false, isChecked: false, type: 'vaccination', text: '', index: -1,
      data: {
        date: "", expirationDate: "", route: "", serialNumber: "", units: "", vaccine: "", vaccineOther: "", volume: "",
      }
    },
    {
      value: 'Vaccination 3', initialChecked: false, isChecked: false, type: 'vaccination', text: '', index: -1,
      data: {
        date: "", expirationDate: "", route: "", serialNumber: "", units: "", vaccine: "", vaccineOther: "", volume: "",
      }
    },
    {
      value: 'Vaccination 4', initialChecked: false, isChecked: false, type: 'vaccination', text: '', index: -1,
      data: {
        date: "", expirationDate: "", route: "", serialNumber: "", units: "", vaccine: "", vaccineOther: "", volume: "",
      }
    }
  ]

  // Array to keep track of selected initial options names/strings (i.e., Weight, Hip Height, etc.)
  public selectedInitialOptions: any[] = [];

  // Initialize an index value to keep track of the index of animal within the selectedAnimals list
  // This allows the user the option to select a new animal by using "Tap to Select Animal" portion of Custom Form
  public currentAnimalIndex = 0;


  // Hold form data for each of the selected animals (input from user)
  private formDataList: any = [];

  // Flag to see if the user is done with the form (User has reached end of selected animals array and tapped Done)
  // Note: The user does not have to enter data for all selected animals for "Done" to be enabled
  private doneFlag: boolean = false;


  constructor(
    public formBuilder: FormBuilder,
    private dataService: DataService,
    private actionsService: ActionsService,
    private route: ActivatedRoute,
    private router: Router,
    private alertCtrl: AlertController,
    private modalCtrl: ModalController
  ) {

    this.route.queryParams.subscribe(_ => {
      if (this.router.getCurrentNavigation()?.extras.state) {

        // animals selected by the user
        const animalsFromRouter = this.router.getCurrentNavigation()?.extras.state?.['selected'];
        this.selectedAnimals = animalsFromRouter.map((animal: any) => ({
          ...animal,
          selectedTreatments: [],
          selectedVaccinations: []
        }));
        console.log('selected animals:', this.selectedAnimals);

        // `single` or `multiple`
        this.selectedAmount = this.router.getCurrentNavigation()?.extras.state?.['selectedAmount'];

        /*
          - Determine if the animals are all male, all female or a mixture
          - This helps determine what options to display if the user selects a `Processing` event
          - Sex IDs: 1 = Bull (Male), 2 = Steer (Male), 3 = Cow (Female)
        * Set variables if the length of the filtered animals equals the length of the original selected animals
        * If none of the variables are set, then that means there is a mixture
        */
        // Filter out any selected animals with Sex ID of 1 (Bull)
        const bulls = this.selectedAnimals.filter((s: any) => s.sex_id === 1);
        // Filter out any selected animals with Sex ID of 3 (Cow)
        const cows = this.selectedAnimals.filter((s: any) => s.sex_id === 3);

        // If the amount of filtered animals is equal to the full amount of selected animals, then all the animals are male
        if (bulls.length === this.selectedAnimals?.length) {
          this.filteredInitialOptions = this.initialOptions.filter(o => o.type !== 'cow');
        } else if (cows.length === this.selectedAnimals?.length) {
          // If the amount of filtered animals is equal to the full amount of selected animals, then all the animals are female
          this.filteredInitialOptions = this.initialOptions.filter(o => o.type !== 'bull');
        } else {
          this.filteredInitialOptions = this.initialOptions.filter(o => o.type === 'all');
        }
      } else {
        // If there are not any query parameters, navigate back to `Home` page
        this.router.navigateByUrl('/tabs/home');
      }
    });

    this.customForm = formBuilder.group({
      weight: new FormControl(''),
      weightType: new FormControl(''),
      hipHeight: new FormControl(''),
      bodyConditionScore: new FormControl(''),
      reproductiveTractScore: new FormControl(''),
      chuteDocilityScore: new FormControl(''),
      footAngleScore: new FormControl(''),
      clawSetScore: new FormControl(''),
      teatScore: new FormControl(''),
      udderScore: new FormControl(''),
      bseMotility: new FormControl(''),
      bseMorphology: new FormControl(''),
      bseAcceptableBreeder: new FormControl(''),
      pregnancyStatus: new FormControl(''),
      gestationLength: new FormControl(''),
      gestationLengthUnits: new FormControl(''),
      castrationTechnique: new FormControl(''),
      castrationDate: new FormControl(''),
      dehorningTechnique: new FormControl(''),
      dehorningDate: new FormControl(''),
      spayingTechnique: new FormControl(''),
      spayingDate: new FormControl(''),
      bodyTemperature: new FormControl(''),
      notes: new FormControl('')
    });
  }

  submit() {
    /*
    * In case the user taps "Done" once and the forms fail to process (submit), set "done" flag to true,
        so if the user attempts to submit again (tapping "Done"),
        the last form will be not re-added to the end of the list of forms
    */
    this.doneFlag = true;

    // Call data service to insert Custom Processing Form event into database
    this.dataService.addCustomProcessingFormEvent(this.actionTypeID, this.action,
      this.selectedAmount, this.selectedAnimals, this.additionalOptions, this.formDataList)
      .then((response: any) => {
        /*
        * If there is a response (successful), then reset the list
        * Display message to user and navigate back to Actions page
        */
        if (response) {
          this.reset();
          this.selectedAnimals = [];
          this.actionsService.actionToast('Custom Processing Form', "animals");
          this.router.navigateByUrl('/tabs/home');
        } else {
          // Else, there is not a response, display error message to user
          this.actionsService.actionError('Actions Custom Processing Form Event Response Error');
        }
      })
      .catch((error: any) => {
        // Else, there is not a response, display error message to user
        this.actionsService.actionError(error);
      });
  }

  // Reset everything
  // If the user completes a (set of) custom forms, it is possible that previous values may be retained
  reset() {
    console.log('reset()');

    // Reset the value in the overall custom form
    this.resetForm();

    // Reset variables related to the custom form page
    this.doneFlag = false;
    this.confirmFlag = false;
    this.formDataList = [];
    this.selectedAnimals = [];
    this.initialOptions = [];
    this.filteredInitialOptions = [];
    this.additionalOptions = [];
  }




  /**
   * Handles the change in animal selection and updates the relevant modal accordingly.
   * 
   * When invoked:
   * - It processes any form changes and logs the currently selected animals.
   * - Presents a modal that displays animals for user selection.
   * 
   * After the modal is dismissed:
   * - For each animal returned from the modal:
   *   - If the animal is checked, it is moved to the beginning of the 'selected' list.
   *   - The currentIndex is reset to the start of the list to reflect this change.
   *   - The length of the selected items is recalculated for UI presentation.
   * - If any animal is selected, the function ensures all related UI elements and variables reflect the new state.
   * - Lastly, the function checks for any existing data associated with the current ID and presents a loading screen.
   * 
   * The function ensures that any changes in animal selection are accurately represented in the UI and that relevant data is fetched and presented.
   * 
   * @returns Promise that resolves when the modal has been presented.
   */
  async animalChange() {
    this.processForm('change');
    console.log('animalChange selected:', this.selectedAnimals);

    const modal = await this.modalCtrl.create({
      component: AnimalChangePage,
      componentProps: {
        selectedAnimals: this.selectedAnimals,
        currentAnimal: this.selectedAnimals[this.currentAnimalIndex]
      }
    });

    modal.onDidDismiss()
      .then((result: any) => {
        if (result && result.data) {
          this.handleReturnedAnimals(result.data);
          this.additionalSelectAllFlag = false;
          this.checkExisting();
          this.dataService.presentLoading();
        }
      })
      .catch(error => {
        console.log('Actions Modal Search Criteria Error', error);
      });

    return await modal.present();
  }

  /*
* Display alert related to the state from which it came
* The user will either tap the "Clear" button or the "Done" button
*/
  async formAlert(state: string) {

    let headerText = '';
    let messageText = '';

    if (state === 'done') {
      headerText = 'Submit All Forms';
      // Set message text for the prompt depending on remaining items
      if (this.selectedAnimals.length > 1) {
        messageText = 'Have you finished inputting custom form data for ALL selected items? \
          If not, tap "No". If you still need to enter data, use "Next" or "Tap to Select New Animal" to proceed';
      } else {
        messageText = 'Have you finished inputting custom form data for ALL selected items?';
      }
    } else if (state === 'clear') {
      headerText = 'Clear Current Form';
      messageText = 'Are you sure you want to clear the current form?';
    }

    if (headerText && messageText) {

      const alert = await this.alertCtrl.create({
        header: headerText,
        message: messageText,
        buttons: [
          {
            text: 'No',
            role: 'cancel',
            cssClass: 'alertRed',
            handler: () => false
          }, {
            text: 'Yes',
            cssClass: 'alertGreen',
            handler: () => {
              if (state === 'done') {
                this.submit();
              } else if (state === 'clear') {
                this.resetForm();
              }
            }
          }
        ],
        backdropDismiss: false
      });

      await alert.present();
    } else {
      return false;
    }

    return false;
  }


  /*
  * When the user taps "Next" or "Done", process the form by appending the
      key/value pairings to an object and append to a list of all forms across the selected items
  */
  processForm(state: string) {
    console.log('processForm state:', state);

    /*
    * If the user taps `Done`, but there is some error,
        check to see that they have already tapped done and added the last item to the list
    * This prevents the last item from continuously being added to the list if the user keeps tapping `Done`
    */
    if (!this.doneFlag) {
      console.log('processForm !this.doneFlag');

      // Initialize variable to hold existing data for the current item already in the list, if applicable
      let existingFormData = null;

      /*
      * If the formDataList already has data,
          check to see if the current form data being processed already exists in the list
      * In this case, overwrite the previous data with the new data
      * If there is existing form data with the same ID, overwrite that data with the new data
      */
      if (this.formDataList) {
        existingFormData = this.formDataList.find((f: any) => f.id === this.selectedAnimals[this.currentAnimalIndex].id);
        const index = this.formDataList.indexOf(existingFormData);

        if (existingFormData && index > -1) {
          this.formDataList.splice(index, 1);
        }
      }

      /*
      * Initialize new dictionary to hold current form input data from user
      * Initialize "id" key as null, to be set later using the ID of the current item
      */
      let formDictionary: any = { id: null };

      /*
      * Set the ID for the current item as an identifier for the current
          form dictionary associated with the current form data
      * Used to distinguish which data corresponds to which selected item,
          so if the user swaps between multiple items
      * Any data that has already been input can be loaded for editing or
          the user can choose to move onto the next item until they are done
      */
      formDictionary.id = this.selectedAnimals[this.currentAnimalIndex].id;

      /*
      * Iterate through form object and append to form dictionary
      */
      Object.keys(this.customForm.controls).forEach(key => {
        formDictionary[key] = this.customForm.get(key).value;
      });

      // Push current item's form dictionary added into list which contains all previous items
      this.formDataList.push(formDictionary);
    }

    if (state === 'next') {
      console.log('Next currentAnimalIndex BEFORE:', this.currentAnimalIndex);
      // Increase index used for displaying current selected item's name (primary ID if animal)
      this.currentAnimalIndex += 1;
      console.log('Next currentAnimalIndex AFTER:', this.currentAnimalIndex);

      // Reset isChecked values for all options
      this.additionalOptions.forEach((option) => {
        option.isChecked = false;
      });


      console.log('Next additionalOptions:', this.additionalOptions);

      // Reset "Select All" for additional options to unchecked
      this.additionalSelectAllFlag = false;
      console.log('Next additionalSelectAllFlag:', this.additionalSelectAllFlag);

      /*
      * Call resetForm() to set all form fields to empty string ('')
      * NOTE: `reset()` function for form [form.reset()] does not act as needed,
          sets string values to null and number values to 0 on reset
      */
      this.resetForm();

      // Check for existing data associated with the current animal
      this.checkExisting();

      // Show loading spinner in-between forms for additional entry
      this.dataService.presentLoading();
    } else if (state === 'done') {
      // Prompt the user to make sure they are finished inputting data for all forms
      this.formAlert('done');
    }
  }

  checkExisting() {

    /*
    * Check to see if the ID for the current item already has existing data saved
    * The user could input information then move on, but go back to a previous input form
    * Load the previous input if applicable
    */
    console.log('checkExisting formDataList:', this.formDataList);
    console.log('checkExisting selectedAnimals:', this.selectedAnimals);
    console.log('checkExisting currentAnimalIndex:', this.currentAnimalIndex);
    const existingData = this.formDataList.find((f: any) => f.id === this.selectedAnimals[this.currentAnimalIndex].id);

    console.log('checkExisting existingData:', existingData);

    if (existingData) {
      /*
      * Iterate through the existing data for the current item ID
      * Patch the form with the values for the appropriate keys
      */

      this.customForm.patchValue({
        weight: existingData?.['weight'],
        weightType: existingData?.['weightType'],
        hipHeight: existingData?.['hipHeight'],
        bodyConditionScore: existingData?.['bodyConditionScore'],
        reproductiveTractScore: existingData?.['reproductiveTractScore'],
        chuteDocilityScore: existingData?.['chuteDocilityScore'],
        footAngleScore: existingData?.['footAngleScore'],
        clawSetScore: existingData?.['clawSetScore'],
        teatScore: existingData?.['teatScore'],
        udderScore: existingData?.['udderScore'],
        bseMotility: existingData?.['bseMotility'],
        bseMorphology: existingData?.['bseMorphology'],
        bseAcceptableBreeder: existingData?.['bseAcceptableBreeder'],
        pregnancyStatus: existingData?.['pregnancyStatus'],
        gestationLength: existingData?.['gestationLength'],
        gestationLengthUnits: existingData?.['gestationLengthUnits'],
        castrationTechnique: existingData?.['castrationTechnique'],
        castrationDate: existingData?.['castrationDate'],
        dehorningTechnique: existingData?.['dehorningTechnique'],
        dehorningDate: existingData?.['dehorningDate'],
        spayingTechnique: existingData?.['spayingTechnique'],
        spayingDate: existingData?.['spayingDate'],
        bodyTemperature: existingData?.['bodyTemperature'],
        notes: existingData?.['notes']
      });

      // Reset isChecked values for all options
      this.additionalOptions.forEach((option) => {
        option.isChecked = false;
      });

      console.log('checkExisting currentAnimalIndex:', this.currentAnimalIndex);
      console.log('checkExisting selectedAnimals:', this.selectedAnimals);
      console.log('checkExisting this.selectedAnimals[this.currentAnimalIndex].selectedTreatments', this.selectedAnimals[this.currentAnimalIndex].selectedTreatments)

      // For the current animal, if they have associated treatments, set the isChecked value to true for that option
      this.selectedAnimals[this.currentAnimalIndex].selectedTreatments.forEach((treatmentIndex: number) => {
        console.log(treatmentIndex);
        this.additionalOptions[treatmentIndex].isChecked = true;
      });

      // For the current animal, if they have associated vaccinations, set the isChecked value to true for that option
      this.selectedAnimals[this.currentAnimalIndex].selectedVaccinations.forEach((vaccinationIndex: number) => {
        console.log(vaccinationIndex);
        this.additionalOptions[vaccinationIndex].isChecked = true;
      });
    } else {
      // If there is not any existing data, reset the form to empty fields for new input of current item
      this.resetForm();
    }
  }

  // Reset form values after each entry when adding multiple animals
  resetForm() {
    console.log('resetForm()');
    this.customForm.patchValue({
      weight: '',
      weightType: '',
      hipHeight: '',
      bodyConditionScore: '',
      reproductiveTractScore: '',
      chuteDocilityScore: '',
      footAngleScore: '',
      clawSetScore: '',
      teatScore: '',
      udderScore: '',
      bseMotility: '',
      bseMorphology: '',
      bseAcceptableBreeder: '',
      pregnancyStatus: '',
      gestationLength: '',
      gestationLengthUnits: '',
      castrationTechnique: '',
      castrationDate: '',
      dehorningTechnique: '',
      dehorningDate: '',
      spayingTechnique: '',
      spayingDate: '',
      bodyTemperature: '',
      notes: ''
    });

    /*
    * Uncheck all additional options (treatments and vaccinations)
    * isChecked = false
    */
    for (const option of this.additionalOptions) {
      option.isChecked = false;
    }
    console.log('resetForm additionalOptions:', this.additionalOptions);
  }



  /**
 * Handles the list of animals returned from a modal.
 * 
 * For each returned animal:
 * - If the animal is checked, it is moved to the beginning of the 'selected' list.
 * - The currentIndex is reset to the start of the list to reflect this change.
 * - The length of the selected items is recalculated for UI presentation.
 * 
 * This function ensures that the list of animals remains accurate in the UI after changes are made in the modal.
 * It specifically manages the order of appearance and updates indices to ensure smooth user navigation.
 * 
 * @param animals Array of animal objects returned from the modal.
 */
  handleReturnedAnimals(animals: any[]) {
    console.log('handleReturnedAnimals animals:', animals);
    console.log('handleReturnedAnimals selectedAnimals:', this.selectedAnimals);
    console.log('handleReturnedAnimals currentAnimalIndex:', this.currentAnimalIndex);
    for (let index = 0; index < animals.length; index++) {
      const animal = animals[index];
      if (animal.isChecked) {
        this.selectedAnimals.splice(index, 1);
        this.selectedAnimals.unshift(animal);
        this.currentAnimalIndex = 0;
        console.log('handleReturnedAnimals selectedAnimals:', this.selectedAnimals);
        console.log('handleReturnedAnimals currentAnimalIndex:', this.currentAnimalIndex);
        break;
      }
    }
  }


  /**
   * Displays a confirmation alert to the user when attempting to exit the custom form.
   * 
   * If the user chooses to leave:
   * - Redirects them back to the Actions (or Home) page.
   * 
   * If the user chooses to dismiss:
   * - Closes the alert and allows them to continue editing the custom form.
   * 
   * The alert is made modal (non-dismissible by backdrop click) to ensure the user makes an explicit choice.
   */
  async cancelForm() {
    const alert = await this.alertCtrl.create({
      header: 'Exit Custom Form',
      message: 'Are you sure you want to leave this page? ' +
        'All data you have entered will be lost and you will be returned to the Actions page!',
      buttons: [
        {
          text: 'Leave Page',
          cssClass: 'alertRed',
          handler: () => {
            this.router.navigateByUrl('/tabs/home');
          }
        },
        {
          text: 'Dismiss',
          role: 'cancel',
          cssClass: 'alertGreen',
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  /**
   * Sets the `confirmFlag` to true, indicating that a certain action or process has been confirmed by the user.
   * 
   * This function can be expanded or modified based on specific contexts where user confirmation is required.
   */
  async confirm() {
    // Check for options with text set but unchecked
    const uncheckedOptionsWithText = this.additionalOptions.filter(option => option.text && !option.isChecked);

    if (uncheckedOptionsWithText.length > 0) {
      // Assuming you have alertController already set up and imported
      const alert = await this.alertCtrl.create({
        header: 'Unchecked Options',
        message: 'You have set additional options but they are not checked. Please review them before confirming.',
        buttons: ['OK']
      });

      await alert.present();

      return; // Exit the function to allow the user to review their choices
    }

    this.confirmFlag = true;

    // Reset value of isChecked to false if needed
    for (const option of this.additionalOptions) {
      option.isChecked = false;
    }

    console.log(this.selectedAnimals);
    console.log(this.selectedInitialOptions);
    console.log(this.additionalOptions);
  }


  /**
   * Toggles the selection state of all filtered initial options based on the `initialSelectAllFlag`.
   *
   * When `initialSelectAllFlag` is true:
   * - Deselects all items in `filteredInitialOptions`.
   * - Clears the `selectedInitialOptions` array.
   *
   * When `initialSelectAllFlag` is false:
   * - Selects all items in `filteredInitialOptions`.
   * - Populates the `selectedInitialOptions` array with the `value` properties of the selected options.
   * 
   * Finally, the `initialSelectAllFlag` is toggled to its opposite state.
   */
  initialSelectAll() {
    if (this.initialSelectAllFlag) {
      this.filteredInitialOptions = this.filteredInitialOptions.map((option: any) => ({ ...option, isChecked: false }));
      this.selectedInitialOptions = []; // clear the selected options array
    } else {
      this.filteredInitialOptions = this.filteredInitialOptions.map((option: any) => ({ ...option, isChecked: true }));
      this.selectedInitialOptions = this.filteredInitialOptions.map((option: { value: any; }) => option.value); // populate the array with the values of the selected options
    }
    this.initialSelectAllFlag = !this.initialSelectAllFlag;
    console.log(this.selectedInitialOptions);
    this.updateInitialOptionSelectAllButton();

  }

  /**
   * Updates the `selectedInitialOptions` array based on changes to an individual checkbox's state.
   *
   * If the passed `option` is checked:
   * - The `option.value` is added to the `selectedInitialOptions` array.
   *
   * If the passed `option` is unchecked:
   * - The `option.value` is removed from the `selectedInitialOptions` array, if present.
   *
   * @param option - The option whose selection state has changed.
   */

  updateSelectedInitialOptions(option: any) {
    console.log('updateSelectedInitialOptions option:', option)
    console.log('updateSelectedInitialOptions option.value:', option.value)

    if (option.isChecked) {
      // If the option is checked, add its value to the array
      this.selectedInitialOptions.push(option.value);
    } else {
      // If the option is unchecked, remove its value from the array
      const index = this.selectedInitialOptions.indexOf(option.value);
      if (index !== -1) {
        this.selectedInitialOptions.splice(index, 1);
      }
    }

    console.log(this.selectedInitialOptions);
    this.updateInitialOptionSelectAllButton();

  }

  /**
 * Counts the number of initial options that are currently checked.
 *
 * @returns {number} The number of initial options with `isChecked` set to true.
 */
  countCheckedInitialOptions(): number {
    return this.filteredInitialOptions.filter((option: { isChecked: any; }) => option.isChecked).length;
  }

  /**
   * Updates the state of the "Select All" button for initial options based on the checked state of individual options.
   *
   * - If all options are checked, set `initialSelectAllFlag` to true.
   * - If no options are checked, set `initialSelectAllFlag` to false.
   * - If some options are checked, make no change to `initialSelectAllFlag`.
   */
  updateInitialOptionSelectAllButton() {
    const checkedCount = this.countCheckedInitialOptions();
    if (checkedCount === 0) {
      this.initialSelectAllFlag = false;
    } else if (checkedCount === this.filteredInitialOptions.length) {
      this.initialSelectAllFlag = true;
    }
  }




  additionalSelectAll() {
    console.log('Entering additionalSelectAll()');
    console.log('Flag (additionalSelectAllFlag):', this.additionalSelectAllFlag);
    console.log('Current Animal Index:', this.currentAnimalIndex);

    // Iterate over each option in the additionalOptions array with index
    this.additionalOptions.forEach((option, index) => {
      console.log('Processing Option:', option.value, '| Index:', index, '| Initial Checked:', option.initialChecked);

      // Operate only on options that were initially checked
      if (option.initialChecked) {
        // Update the isChecked property of the option based on the additionalSelectAllFlag
        option.isChecked = this.additionalSelectAllFlag;
        console.log('Updated isChecked to:', option.isChecked);

        // Directly access the selected treatments and vaccinations arrays of the current animal
        let selectedTreatments = this.selectedAnimals[this.currentAnimalIndex].selectedTreatments;
        let selectedVaccinations = this.selectedAnimals[this.currentAnimalIndex].selectedVaccinations;

        // If additionalSelectAllFlag is true, add the index to the selected arrays
        if (this.additionalSelectAllFlag) {
          if (option.type === "treatment") {
            if (!selectedTreatments.includes(index)) {
              selectedTreatments.push(index);
              console.log('Added to selectedTreatments:', index);
            } else {
              console.log('Index already exists in selectedTreatments:', index);
            }
          } else if (option.type === "vaccination") {
            if (!selectedVaccinations.includes(index)) {
              selectedVaccinations.push(index);
              console.log('Added to selectedVaccinations:', index);
            } else {
              console.log('Index already exists in selectedVaccinations:', index);
            }
          } else {
            console.log('Option type not recognized:', option.type);
          }
        }
        // If additionalSelectAllFlag is false, remove the index from the selected arrays
        else {
          if (option.type === "treatment") {
            if (selectedTreatments.includes(index)) {
              this.selectedAnimals[this.currentAnimalIndex].selectedTreatments = selectedTreatments.filter((item: number) => item !== index);
              console.log('Removed from selectedTreatments:', index);
            } else {
              console.log('Index does not exist in selectedTreatments:', index);
            }
          } else if (option.type === "vaccination") {
            if (selectedVaccinations.includes(index)) {
              this.selectedAnimals[this.currentAnimalIndex].selectedVaccinations = selectedVaccinations.filter((item: number) => item !== index);
              console.log('Removed from selectedVaccinations:', index);
            } else {
              console.log('Index does not exist in selectedVaccinations:', index);
            }
          } else {
            console.log('Option type not recognized:', option.type);
          }
        }
      } else {
        console.log('Skipping option as it is not initially checked');
      }
    });

    console.log('Final state of selectedAnimals:', this.selectedAnimals);
    this.updateAdditionalOptionSelectAllButton();
  }




  // additionalSelectAll() {
  //   console.log('here additionalSelectAll()');
  //   if (this.additionalSelectAllFlag) {
  //     this.additionalOptions.forEach(option => {
  //       option.isChecked = false; // clear the form checked flag
  //     });

  //     console.log('A:',  this.selectedAnimals);

  //     // Clear the selected treatments and vaccinations arrays for all animals
  //     this.selectedAnimals.forEach(animal => {
  //       animal.selectedTreatments = [];
  //       animal.selectedVaccinations = [];
  //     });
  //   } else {
  //     this.additionalOptions.forEach((option, index) => {
  //       if (option.initialChecked) { // only operate on options that were initially checked
  //         option.isChecked = true; // set the form checked flag

  //         if (option.type === "treatment") {
  //           console.log('treatment');
  //           this.selectedAnimals.forEach(animal => {
  //             if (!animal.selectedTreatments.includes(index)) {
  //               animal.selectedTreatments.push(index);
  //             }
  //             console.log(animal.selectedTreatments);
  //           });
  //         } else if (option.type === "vaccination") {
  //           console.log('vaccination');
  //           this.selectedAnimals.forEach(animal => {
  //             if (!animal.selectedVaccinations.includes(index)) {
  //               animal.selectedVaccinations.push(index);
  //             }
  //             console.log(animal.selectedVaccinations);
  //           });
  //         }
  //       }
  //     });
  //   }

  //   this.additionalSelectAllFlag = !this.additionalSelectAllFlag;

  //   console.log('B:',  this.selectedAnimals);
  // }

  /**
   * Updates the option's initialChecked status based on the checkbox status.
   * 
   * If the option is checked:
   * - Set its initialChecked as true.
   * 
   * If the option is unchecked:
   * - Set its initialChecked as false.
   * 
   * @param option The option for which the checkbox value changed.
   */
  updateSelectedAdditionalOptions(option: any) {
    option.initialChecked = option.isChecked;
    console.log('updateSelectedAdditionalOptions option:', option);
    console.log('updateSelectedAdditionalOptions additionalOptions:', this.additionalOptions);
  }

  /**
   * Update the additional option within the Form
   * 
   * @param event Event that contains the checkbox status.
   * @param option The option for which the checkbox value changed.
   */
  updateAdditionalOptionForm(event: any, option: any, index: number) {
    option.isChecked = event.detail.checked;
    console.log('updateAdditionalOptionForm option:', option);
    console.log('updateAdditionalOptionForm additionalOptions:', this.additionalOptions);

    const currentAnimal = this.selectedAnimals[this.currentAnimalIndex];
    console.log('current animal:', currentAnimal);

    // Define a variable to store the selected array based on option type
    let selectedArray;
    if (option.type === 'treatment') {
      selectedArray = currentAnimal.selectedTreatments;
    } else if (option.type === 'vaccination') {
      selectedArray = currentAnimal.selectedVaccinations;
    } else {
      console.error('Option type not recognized:', option.type);
      return;
    }

    // Update the selected array based on the checkbox status
    if (option.isChecked && !selectedArray.includes(index)) {
      selectedArray.push(index);
    } else if (!option.isChecked) {
      const idx = selectedArray.indexOf(index);
      if (idx > -1) {
        selectedArray.splice(idx, 1);
      }
    }

    console.log('currentAnimal.selectedTreatments:', currentAnimal.selectedTreatments);
    console.log('currentAnimal.selectedVaccinations:', currentAnimal.selectedVaccinations);

    console.log('updateAdditionalOptionForm selectedAnimals:', this.selectedAnimals);

    this.updateAdditionalOptionSelectAllButton();
  }


  /**
   * Counts the number of additional options that are currently checked.
   * It iterates through the `additionalOptions` array and filters out the options 
   * where the isChecked property is true, then returns the count of these options.
   * 
   * @returns {number} The number of additional options that are checked.
   */
  countCheckedAdditionalOptions(): number {
    return this.additionalOptions.filter(option => option.isChecked).length;
  }

  /**
   * Updates the state of the "Select All" button based on the number of additional options that are checked.
   * 
   * If no options are checked, it sets `additionalSelectAllFlag` to false, 
   * indicating that the "Select All" button should be unchecked.
   * 
   * If all options are checked, it sets `additionalSelectAllFlag` to true,
   * indicating that the "Select All" button should be checked.
   * 
   * This function should be called whenever an individual additional option is toggled
   * or after performing a "Select All" or "Deselect All" action to ensure that the 
   * "Select All" button's state accurately reflects the state of the individual options.
   */
  updateAdditionalOptionSelectAllButton() {
    const checkedCount = this.countCheckedAdditionalOptions();
    if (checkedCount === 0) {
      this.additionalSelectAllFlag = false;
    } else if (checkedCount === this.additionalOptions.length) {
      this.additionalSelectAllFlag = true;
    }
    // Note: If we want to handle the "indeterminate" state (i.e., some but not all options are selected),
    // we can add an else block here to set additionalSelectAllFlag to a third state or update another property.
  }

  /**
   * Handles the action to be taken when an additional option is triggered.
   * 
   * If the option has a text property:
   * - Presents an alert allowing the user to modify the option.
   * 
   * If the option does not have a text property:
   * - Opens a modal for adding or viewing additional option details.
   * 
   * @param event The event that triggered this function.
   * @param option The option which was clicked or triggered.
   * @param index The index of the option in the list.
   */
  async handleAdditionalOption(event: any, option: any, index: number) {
    console.log('handleAdditionalOption option:', option);
    if (option.text) {
      await this.presentModifyAlert(event, option, index);
    } else {
      this.presentAdditionalOptionModal(event, option, index);
    }
  }

  /**
   * Checks if at least one of the initial or additional options has been checked.
   *
   * @returns {boolean} True if at least one option is checked, otherwise false.
   */
  get hasCheckedOption(): boolean {
    return this.filteredInitialOptions.some((option: { isChecked: any; }) => option.isChecked) ||
      this.additionalOptions.some(option => option.isChecked);
  }

  /**
   * Determines if the additional options should be displayed to the user.
   * 
   * The additional options will be displayed if:
   * - At least one of the options has been initially checked.
   * - The confirm flag is set (indicating the user has proceeded to this stage of the process).
   * 
   * @returns {boolean} True if conditions for showing additional options are met, otherwise false.
   */
  get showAdditionalOptions(): boolean {
    return this.additionalOptions.some(opt => opt.initialChecked) && this.confirmFlag;
  }

  /**
   * Computes the number of additional options that have been initially checked by the user.
   *
   * @returns {number} The count of options that have their initialChecked property set to true.
   */
  get countOfInitialCheckedOptions(): number {
    return this.additionalOptions.filter(option => option.initialChecked).length;
  }


  /**
   * Presents an alert to the user allowing them to clear or modify a selected additional option.
   * 
   * The alert provides three options to the user:
   * 1. Clear the selected option. If cleared, it checks whether the option exists in the 
   *    `selectedAdditionalOptions` array and removes it if found.
   * 2. Modify the option. This will present the appropriate modal for modifying based on the option's value.
   * 3. Cancel the action and close the alert.
   * 
   * @param event The event that triggered this function.
   * @param option The option to be cleared or modified.
   * @param index The index of the option in the list.
   */
  async presentModifyAlert(event: any, option: any, index: number) {
    const alert = await this.alertCtrl.create({
      header: 'Update Option',
      message: 'Do you want to clear this additional option or modify it?',
      buttons: [
        {
          text: 'Clear Selected Option',
          cssClass: 'alertGreen',
          handler: () => {
            option.text = '';
            option.initialChecked = false;  // Clear the initial checked flag
          }
        },
        {
          text: 'Modify Option',
          cssClass: 'alertGreen',
          handler: () => {
            this.presentAdditionalOptionModal(event, option, index);
          }
        },
        {
          text: 'Cancel',
          role: 'cancel',
          cssClass: 'alertRed',
          handler: () => {
            // Handle the event here. Ensure event object is correctly passed and used.
            console.log('Cancel presentModifyAlert')
          }
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  /**
   * Determines and presents the appropriate modal based on the option's value.
   * 
   * - If the option's value contains 'Treatment', it presents a 'health treatment' modal.
   * - If the option's value contains 'Vaccination', it presents a 'vaccination' modal.
   * 
   * @param event The event that triggered this function.
   * @param option The option based on which the modal is determined.
   * @param index The index of the option in the list.
   */
  presentAdditionalOptionModal(event: any, option: any, index: number) {
    if (option.value.includes('Treatment')) {
      console.log('Treatment');
      this.presentModal('health', 'treatment', event, option, index);
    } else if (option.value.includes('Vaccination')) {
      this.presentModal('vaccination', '', event, option, index);
    }
  }

  /**
   * Presents the appropriate modal to the user based on the specified actionType.
   * 
   * Depending on the actionType, it initializes and presents either a Health or a Vaccination modal.
   * 
   * For the Health modal:
   * - If a Treatment is selected, the user is prompted to provide details on the treatment or just an observation.
   * - Upon dismissal of the modal, it processes the provided data and constructs a displayable text.
   * - It unchecks the associated option in the custom form upon dismissal, requiring re-selection by the user.
   * 
   * For the Vaccination modal:
   * - The user is prompted to provide vaccination details.
   * - Upon dismissal, it processes the data, checks for completeness, and constructs a displayable text.
   * - Similarly, it unchecks the associated option in the custom form upon dismissal.
   * 
   * @param actionType Specifies the type of action - either "health" or "vaccination".
   * @param action If actionType is "health", provides further classification - "treatment" indicates observation and treatment, otherwise, just observation.
   * @param event The event that triggered the modal's presentation.
   * @param option The associated option in the custom form.
   * @param index The index of the option in the list.
   */
  async presentModal(actionType: string, action: string, event: any, option: any, index: number) {
    let modal = null;

    if (actionType === 'health') {
      modal = await this.modalCtrl.create({
        component: HealthPage,
        componentProps: {
          isModal: true,
          actionType,
          action
        }
      });

      modal.onDidDismiss()
        .then((data: any) => {
          if (this.additionalOptions[index].initialChecked) {
            this.additionalOptions[index].initialChecked = false;
            console.log(this.additionalOptions[index])
          }

          if (!data.data) return;

          const [formDictionary, bodySystem, bodyCondition, selectedProduct, productOther, routeOfAdmin] = data.data;

          if (!bodySystem || !bodyCondition || !selectedProduct || !routeOfAdmin) {
            console.warn("Essential treatment data missing, not processing modal data.");
            return;
          }

          this.additionalOptions[index].data = formDictionary;
          console.log('this.additionalOptions:', this.additionalOptions);
          console.log('this.additionalOptions[index]:', this.additionalOptions[index]);
          console.log('this.additionalOptions[index].data:', this.additionalOptions[index].data);

          if (productOther) {
            option.text = `${bodySystem}; ${bodyCondition}; ${selectedProduct} (${productOther}); ${routeOfAdmin}`;
          } else {
            option.text = `${bodySystem}; ${bodyCondition}; ${selectedProduct}; ${routeOfAdmin}`;
          }
        })
        .catch((error: any) => {
          console.log('Custom Form Health Modal Error', error);
        });

      return await modal.present();

    } else if (actionType === 'vaccination') {
      modal = await this.modalCtrl.create({
        component: VaccinationPage,
        componentProps: {
          isModal: true
        }
      });

      modal.onDidDismiss()
        .then((data: any) => {
          if (this.additionalOptions[index].initialChecked) {
            this.additionalOptions[index].initialChecked = false;
            console.log(this.additionalOptions[index])
          }

          if (!data.data) return;

          const [formDictionary, selectedVaccine, vaccineOther, routeOfAdmin] = data.data;

          if (!selectedVaccine || !routeOfAdmin) {
            console.warn("Essential vaccination data missing, not processing modal data.");
            return;
          }

          this.additionalOptions[index].data = formDictionary;
          console.log('this.additionalOptions:', this.additionalOptions);
          console.log('this.additionalOptions[index]:', this.additionalOptions[index]);
          console.log('this.additionalOptions[index].data:', this.additionalOptions[index].data);

          if (vaccineOther) {
            option.text = `${selectedVaccine} (${vaccineOther}); ${routeOfAdmin}`;
          } else {
            option.text = `${selectedVaccine}; ${routeOfAdmin}`;
          }
        })
        .catch((error: any) => {
          console.log('Custom Form Vaccination Modal Error', error);
        });

      return await modal.present();
    }
  }

  ionViewDidEnter() {
    this.doneFlag = false;
    this.confirmFlag = false;
  }


  ngOnInit() {
  }

}
