

import { Component } from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { FormBuilder, FormGroup, Validators, FormControl } from '@angular/forms';
import { AlertController, ModalController } from '@ionic/angular';
import { DataService } from 'src/app/services/data.service';
import { SelectDamSirePage } from '../select-dam-sire/select-dam-sire.page';
import { HttpClient } from '@angular/common/http';

@Component({
  selector: 'app-edit-animal',
  templateUrl: './edit-animal.page.html',
  styleUrls: ['./edit-animal.page.scss'],
})
export class EditAnimalPage {

  public reasons = [
    { id: 2, name: 'Reference' },
    { id: 3, name: 'Culled' },
    { id: 4, name: 'Sold' },
    { id: 5, name: 'Died' },
  ];

  public referenceCauses = [
    { id: 2, name: 'A.I. sire' }
  ];

  public culledCauses = [
    { id: 3, name: 'Age' },
    { id: 4, name: 'Appearance' },
    { id: 5, name: 'Body condition' },
    { id: 6, name: 'Calf died before weaning' },
    { id: 7, name: 'Calving late in season' },
    { id: 8, name: 'Conceived but aborted' },
    { id: 9, name: 'Disposition' },
    { id: 10, name: 'Exposed, failed to conceive' },
    { id: 11, name: 'Genetic defect/carrier status' },
    { id: 12, name: 'Herd reduction' },
    { id: 13, name: 'Hoof condition/lameness' },
    { id: 14, name: 'Illness/disease' },
    { id: 15, name: 'Injury or accident' },
    { id: 16, name: 'Poor performance' },
    { id: 17, name: 'Prolapse' },
    { id: 18, name: 'Structure/conformation' },
    { id: 19, name: 'Udder quality' }
    // THE FOLLOWING CAUSES FOR REASON `CULLED` WILL BE ADDED INLINE, CHECK selectChange()
    // Bulls only
    // {id: 39, name: 'Infertility', sex: '1'},
    // Bulls only
    // {id: 40, name: 'Failed BSE', sex: '1'}
  ];

  public soldCauses = [
    { id: 20, name: 'Breeding bull - virgin' },
    { id: 21, name: 'Breeding bull - non-virgin ' },
    { id: 22, name: 'Calf sold with dam' },
    { id: 23, name: 'Embryo donor' },
    { id: 24, name: 'Feeder calf' },
    { id: 25, name: 'Replacement - bred cow ' },
    { id: 26, name: 'Replacement - bred heifer' },
    { id: 27, name: 'Replacement - open heifer' },
    { id: 28, name: 'Replacement - open cow' }
  ];

  public diedCauses = [
    { id: 29, name: 'Birth defect' },
    { id: 30, name: 'Body condition' },
    { id: 31, name: 'Calving difficulty' },
    { id: 32, name: 'Illness' },
    { id: 33, name: 'Injury/accident' },
    { id: 34, name: 'Predation' },
    { id: 35, name: 'Premature calf' },
    { id: 36, name: 'Stillborn calf' },
    { id: 37, name: 'Unknown ' },
    { id: 38, name: 'Weather' }
  ];

  public animalID: any;
  public breedSelected: any;
  public causes: any;
  public editAnimalForm: any;
  public editFlag: boolean = false;
  public filteredBreeds: any;
  public inputValue: any;
  public isBreedAvailable: boolean = false;
  public reasonChange: boolean = false;
  public selected: any;
  public statusChange: boolean = false;

  private breeds: any;
  private currentAnimalID: number = 0;
  private currentAnimalDamID: any;
  private currentAnimalSireID: any;
  private groupID: any;
  private groupName: any;
  private page: any;

  // Determine if the Select Dam / Select Sire modal is open (prevent duplicate/bubbling modals)
  private modalOpen: boolean = false;

  constructor(
    public formBuilder: FormBuilder,
    private dataService: DataService,
    private route: ActivatedRoute,
    private router: Router,
    private alertCtrl: AlertController,
    private modalCtrl: ModalController,
    private httpClient: HttpClient
  ) {
    this.route.queryParams.subscribe(() => {
      // If there is passed navigation data, proceed
      if (this.router.getCurrentNavigation()?.extras.state) {
        this.selected = this.router.getCurrentNavigation()?.extras.state?.['data'];
        this.page = this.router.getCurrentNavigation()?.extras.state?.['page'];

        // If there is a selected animal to edit, proceed
        if (this.selected) {

          this.currentAnimalID = this.selected.id;
          this.currentAnimalDamID = this.selected.dam;
          this.currentAnimalSireID = this.selected.sire;

          this.editAnimalForm = new FormGroup({
            /*
            * `id` refers to the database record id of the animal
            * This is not actually shown on the form as a field (hidden field)
            */
            id: new FormControl(this.selected.id, Validators.required),
            primaryID: new FormControl(this.selected.primary_id.toString(), [Validators.required, Validators.maxLength(255)]),
            sex: new FormControl((this.selected.sex_id ? this.selected.sex_id.toString() : ''), Validators.required),
            additionType: new FormControl(this.selected.addition_type.toString(), Validators.required),
            birthDate: new FormControl(this.selected.date_birth),
            purchaseDate: new FormControl(this.selected.date_purchase),
            breed: new FormControl(''),
            breedID: new FormControl(this.selected.breed_id ? this.selected.breed_id.toString() : ''),
            color: new FormControl(this.selected.color_id ? this.selected.color_id.toString() : ''),
            weight: new FormControl(this.selected.weight ? this.selected.weight.toString() : ''),
            tagColor: new FormControl(this.selected.tag_color ? this.selected.tag_color.toString() : '', [Validators.maxLength(255)]),
            officialNUES: new FormControl(this.selected.official_nues ? this.selected.official_nues.toString() : '',
              [Validators.minLength(8), Validators.maxLength(9)]),
            officialAIN: new FormControl(this.selected.official_ain ? this.selected.official_ain.toString() : '',
              [Validators.minLength(15), Validators.maxLength(15)]),
            twinStatus: new FormControl(this.selected.twin_status_id ? this.selected.twin_status_id.toString() : ''),
            calvingVigorScore: new FormControl(
              (this.selected.calving_vigor_score_id ? this.selected.calving_vigor_score_id.toString() : '')),
            dam: new FormControl(this.selected.dam_name ? this.selected.dam_name.toString() : ''),
            /*
            * `damID` refers to the ID of the animal that the user selected as the dam
            * This is not actually shown on the form as a field, acts as a hidden field
            * However, the `dam` field is shown with the animal name for the user to select and change if needed
            */
            damID: new FormControl(this.selected.dam ? this.selected.dam.toString() : ''),
            teatScore: new FormControl(this.selected.teat_score_id ? this.selected.teat_score_id.toString() : ''),
            udderScore: new FormControl(this.selected.udder_score_id ? this.selected.udder_score_id.toString() : ''),
            calvingEase: new FormControl(this.selected.calving_ease_id ? this.selected.calving_ease_id.toString() : ''),
            damBCS: new FormControl(this.selected.dam_bcs_id ? this.selected.dam_bcs_id.toString() : ''),
            sire: new FormControl(this.selected.sire_name ? this.selected.sire_name.toString() : ''),
            /*
            * `sireID` refers to the ID of the animal that the user selected as the sire
            * This is not actually shown on the form as a field, acts as a hidden field
            * However, the `sire` field is shown with the animal name for the user to select and change if needed
            */
            sireID: new FormControl(this.selected.sire ? this.selected.sire.toString() : ''),
            /*
            * Set the status for the current animal to be added
            * 1 = Active, 2 = Inactive
            * Active (1) is default Value
            * Inactive (2) prompts user and requires additional information to be added (reasoning)
            */
            status: new FormControl(this.selected.status_id ? this.selected.status_id.toString() : '1', [Validators.required]),
            // Set reasoning if status is "Inactive" (2)
            reason: new FormControl(this.selected.reason_id ? this.selected.reason_id.toString() : '1', [Validators.required]),
            // Set cause if reasoning is true due to status being "Inactive" (2)
            cause: new FormControl(this.selected.cause_id ? this.selected.cause_id.toString() : '1', [Validators.required]),
            notes: new FormControl(this.selected.notes ? this.selected.notes.toString() : '', [Validators.maxLength(255)]),
          });

          // If there is a Reason ID, set appropriate `cause` options based on the `reason`
          if (this.selected.reason_id) {
            if (this.selected.reason_id.toString() === '2') {
              // If the user selects reason "Reference"; set causes options
              this.causes = this.referenceCauses;
            } else if (this.selected.reason_id.toString() === '3') {
              // If the user selects reason "Culled"; set causes options
              this.causes = this.culledCauses;
            } else if (this.selected.reason_id.toString() === '4') {
              // If the user selects reason "Sold"; set causes options
              this.causes = this.soldCauses;
            } else if (this.selected.reason_id.toString() === '5') {
              // If the user selects reason "Died"; set causes options
              this.causes = this.diedCauses;
            }
          }

          if (this.editAnimalForm.controls['breedID'].value) {
            this.getBreeds('initial', null);
          } else {
            // Check to see if there are any changes in the form
            this.editAnimalForm.valueChanges.subscribe(() => {
              // Show edit message to the user
              this.editFlag = true;
              /*
              * Check to see if the user is changing the status of the current animal from "Active" (1) to "Inactive" (2)
              * If so, then check to see if they have set values for the "Reason" and "Cause", as these are required fields
                  when setting the animal status to "Inactive"
              * If the values are not empty, then that means the user has selected options for both fields
              * Allow the user to save the edits by enabling the "Save" button in the top right corner
              */
              if (this.editAnimalForm.controls['status'].value === '2') {
                if (this.editAnimalForm.controls['reason'].value !== '' && this.editAnimalForm.controls['cause'].value !== '') {
                  this.editFlag = true;
                } else {
                  this.editFlag = false;
                }
              } else {
                this.editFlag = true;
              }
            });
          }

        }
      }
    });
  }

  async selectChange(changeType: string, event: any) {
    if (event.detail.value) {
      if (changeType === 'status') {
        this.statusChange = true;

        // If the user selects status "Active"; reset values
        if (event.detail.value === '1') {
          this.editAnimalForm.patchValue({ reason: '1' });
          this.editAnimalForm.patchValue({ cause: '1' });
          this.statusChange = false;
          this.reasonChange = false;
          this.causes = null;
        }
        else if (event.detail.value === '2') {
          // Else if, the user select status "Inactive";

          // Set current reason and cause to empty string (string "1" is default for active, needed to toggle Done button)
          this.editAnimalForm.patchValue({ reason: '' });
          this.editAnimalForm.patchValue({ cause: '' });

          // Display prompt when setting animal as "Inactive"
          const alert = await this.alertCtrl.create({
            header: 'Inactive Animal',
            message: 'Animals with "Inactive" status cannot be used to record data.',
            buttons: ['OK'],
            backdropDismiss: false
          });
          await alert.present();

        }
      } else if (changeType === 'reason') {
        // Else if the user has selected an status "Inactive" and a reason for being inactive
        this.reasonChange = true;
        this.causes = null;

        this.editAnimalForm.patchValue({ cause: '' });

        // If the user selects reason 2 (Reference)
        if (event.detail.value === '2') {
          // Set cause options for 2 (Reference)
          this.causes = this.referenceCauses;
        } else if (event.detail.value === '3') {
          // If the user selects reason 3 (Culled)

          // Retrieve current value for Sex (integer value)
          const currentSex = this.editAnimalForm.get('sex')?.value;

          /* 
          * Set causes options for 3 (Culled)
          * Use JSON stringify and JSON parse to "deep copy" array of objects
          * Therefore later modifying `this.causes` will not change initial array `this.culledCauses`
          */
          this.causes = JSON.parse(JSON.stringify(this.culledCauses));

          // If the current sex value is 1 (Bull), apply additional Culled causes specific to Bulls only
          if (currentSex === 1) {
            this.causes.push({ id: 39, name: 'Infertility' })
            this.causes.push({ id: 40, name: 'Failed BSE' })
          }

        } else if (event.detail.value === '4') {
          // If the user selects reason 4 (Sold)

          // Set cause options for 4 (Sold)
          this.causes = this.soldCauses;
        } else if (event.detail.value === '5') {
          // If the user selects reason 5 (Died)

          // Set cause options for 5 (Died)
          this.causes = this.diedCauses;
        } else {
          // Else, reset causes to null value (hides select option)
          this.causes = null;
        }

      } else if (changeType === 'sex') {
        /*
        * If the user changes Sex value then reset Reason and Cause
        * This is due to specific causes being added to reason `Culled` for Bulls only
        * Set current reason and cause to empty string (string "1" is default for active, needed to toggle Done button)
        */
        this.editAnimalForm.patchValue({ reason: '1' });
        this.editAnimalForm.patchValue({ cause: '1' });

        /*
        * Prompt user if they change sex value to "Unknown"
        * Value string "4" refers to user selection of "Unknown" sex
        */
        if (event.detail.value === '4') {
          const alert = await this.alertCtrl.create({
            header: 'Unknown Sex',
            message: 'Animals with "Unknown" sex will not appear in reports that require a sex.',
            buttons: ['OK'],
            backdropDismiss: false
          });

          await alert.present();
        }
      } else if (changeType === 'addition') {
        /*
        * Set addition type for current animal
        * Patch the form value for `additionType` based on user selected option
        */
        this.editAnimalForm.patchValue({
          additionType: event.detail.value,
        });

        /*
        * If the user changes the addition type and the value is 1 (Birth)
            then we can assume they swapped from Purchase (2) to Birth (1)
        * Therefore, clear the purchase date, even if not set initially
        */
        if (event.detail.value == 1) {
          this.editAnimalForm.patchValue({
            purchaseDate: '',
          });
        }
      }
    }
  }

  // Set breed for current animal
  setBreed(breed: any) {
    if (breed) {
      this.breedSelected = breed;
      this.isBreedAvailable = false;

      this.editAnimalForm.patchValue({
        breed: this.breedSelected.name,
        breedID: this.breedSelected.id
      });
    }
  }

  // Grab breeds data from local JSON file
  initializeItems() {
    return this.httpClient.get('/assets/data/breeds.json');
  }

  /*
  * On the initial page call, if there is a breed that was previously set by the user
  * Use the `breedID` to find the breed name and set as `editAnimalForm` value
  */
  findBreed() {
    if (this.breeds && this.editAnimalForm.controls['breedID'].value) {
      const breedID = this.editAnimalForm.controls['breedID'].value;
      console.log(`findBreed breedID: ${breedID}`);
      this.breedSelected = this.breeds.filter((item: any) => (item.id === +breedID));
      console.log(`findBreed breed: ${this.breedSelected}`);

      /* If breedSelect has a value (object array) and there is an object at index 0 of array
      * Patch the form `breed` field with the name of the selected breed
      * NOTE: THIS WILL TRIGGER (ionChange) on the form and make another call to getBreeds()
      * In turn, this will randomly display select options; cannot resolve at this time
      */
      if (this.breedSelected && this.breedSelected[0]) {
        this.editAnimalForm.patchValue({
          breed: this.breedSelected[0].name,
        });

        this.isBreedAvailable = false;

        // If there are any value changes in the form, set `editFlag` to true; display edit message
        this.editAnimalForm.valueChanges.subscribe(() => {
          this.editFlag = true;
        });
      }
    }
  }

  getBreeds(callType: string, event: any) {
    console.log(`getBreeds() callType: ${callType}`);

    /*
    * If breeds data has already been called and set into variable
        then only worry about filtering results based on user input
    */
    if (this.breeds) {
      if (callType === 'initial') {
        this.findBreed();
      } else if (callType === 'input') {
        this.filterBreeds(event);
      }

    } else {
      /*
      * Else, no breed data has been called
      * Get data from local `breeds.json` file
      */
      this.initializeItems().subscribe(resp => {
        if (resp) {
          this.breeds = resp;
          this.getBreeds(callType, event);
        } else {
          this.isBreedAvailable = false;
        }
      });
    }
  }

  // When the user begins searching (typing) a breed, filter results from `breeds.json`
  filterBreeds(event: any) {
    // set `inputValue` to the value of the search bar
    const inputValue = event.target.value;
    console.log(`filterBreeds() inputValue: ${inputValue}`);

    this.breedSelected = null;

    // If `inputValue` is an empty string don't filter the items
    if (inputValue && inputValue.trim() !== '') {
      this.isBreedAvailable = true;

      // Filter breed names based on `inputValue` from user
      this.filteredBreeds = this.breeds.filter((item: any) => (item.name.toLowerCase().indexOf(inputValue.toLowerCase()) > -1));
    } else {
      this.isBreedAvailable = false;

      // set breeds value as empty; user either clicked X to remove input or backspaced/deleted
      if (event.detail.value == '') {
        this.editAnimalForm.patchValue({
          breed: '',
          breedID: ''
        });
      }
    }

    console.log(`filterBreeds() isBreedAvailable: ${this.isBreedAvailable}`);
    console.log(`filterBreeds() breedSelected: ${this.breedSelected}`);
    console.log(`filterBreeds() filteredBreeds: ${this.filteredBreeds}`);

  }

  /*
  * When the user taps to select a Dam or Sire
  * Open modal that displays all animals (from all groups) associated to the user's account
  */
  async selectDamSire(animalType: string) {

    // If modalOpen is already true; return
    if (this.modalOpen) return;

    // Set modalOpen to true
    this.modalOpen = true;

    const modal = await this.modalCtrl.create({
      component: SelectDamSirePage,
      componentProps: {
        selectType: animalType,
        /*
        * Pass the ID of the current animal being viewed so the list of animals presented when selecting
            dam/sire will not show the current animal, thus the user cannot make the animal it's own dam/sire
        */
        currentAnimalID: +this.currentAnimalID,
        /*
        * Pass the IDs of the current dam/sire so the list of animals presented when selecting
            dam/sire will not include any already selected animal
        * (i.e. dam cannot see sire selection, sire cannot see dam selection)
        */
        currentAnimalDamID: +this.currentAnimalDamID,
        currentAnimalSireID: +this.currentAnimalSireID
      }
    });

    /*
    * When the modal is dismissed, see if we have any returned data (user has selected a dam or sire)
    * If there is data, check to see if we are setting the dam or the sire
    * NOTE: The "innerHTML" of the form cannot be set, so we cannot set the value and
        the text separately for one form field thus, we have a hidden form field that
        holds the value for the dam/sire while the actual field the user sees only shows the name
    * NOTE: attempted to add this function to the `add.service.ts` (addService) file but could not get a return value
        have to place directly in files for usage
    */
    modal.onDidDismiss()
      .then((data: any) => {
        const animal = data.data;
        if (animalType === 'dam') {
          // if we have animal data then set values
          if (animal) {
            this.currentAnimalDamID = animal.id;
            this.editAnimalForm.patchValue({
              dam: animal.primary_id,
              damID: +animal.id
            });

          } else {
            this.currentAnimalDamID = null;
            this.editAnimalForm.patchValue({
              dam: '',
              damID: ''
            });

          }
        } else if (animalType === 'sire') {
          if (animal) {
            this.currentAnimalSireID = animal.id;
            this.editAnimalForm.patchValue({
              sire: animal.primary_id,
              sireID: +animal.id
            });

          } else {
            this.currentAnimalSireID = null;
            this.editAnimalForm.patchValue({
              sire: '',
              sireID: ''
            });

          }
        }

        // Reset modalOpen flag to false
        this.modalOpen = false;
      });

    return await modal.present();
  }

  /*
  * If the response is not "success", display alert to user regarding editing current animal
  * Else, an animal was edited successfully, navigate back to appropriate page
  */
  async editAlert(response: any) {
    if (response !== 'success') {
      const alert = await this.alertCtrl.create({
        header: 'Error Editing Animal',
        message: 'An error occurred while trying to apply edits to animal. Check your network connection and try again.',
        buttons: ['OK'],
        backdropDismiss: false
      });

      await alert.present();
    } else {
      // Navigate back depending on which page we came from
      if (this.page === 'manageAnimals' && !this.groupID && !this.groupName) {
        this.router.navigateByUrl(`/tabs/manage-animals`);
      } else if (this.page === 'groupAnimals' && this.groupID && this.groupName) {
        this.router.navigateByUrl(`/tabs/home/manage/group-animals/${this.groupID}/${this.groupName}/0`);
      } else {
        this.router.navigateByUrl(`/tabs/home`);
      }
    }
  }

  // Display alert to user if the required fields do not have value
  async missingAlert() {
    const alert = await this.alertCtrl.create({
      header: 'Missing Data',
      message: 'Make sure you have a value for required fields Primary ID, Sex, and Addition Type.',
      buttons: ['OK'],
      backdropDismiss: false
    });

    await alert.present();
  }

  /*
  * When the user taps `Edit` button
  * This enables the form and marks editing flag as `True`
  * This enables the `Save` button to take place of the `Edit` button
  */
  edit() {
    this.editFlag = true;
  }

  /*
  * Save changes for animals or groups when they are done editing
  * Updates data in database
  */
  save() {
    // Grab the current values for `primaryID`, `sex`, and `additionType` (required `editAnimalForm` fields)
    const primaryID = this.editAnimalForm.value.primaryID;
    const sex = this.editAnimalForm.value.sex;
    const additionType = this.editAnimalForm.value.additionType;

    // Make sure both form fields have a value since this data is required when creating an animal
    if (primaryID && sex && additionType) {
      const formData = new FormData();

      /*
      * Iterate through `editAnimalForm` object and append to form data object
      * Do not append `dam`, `sire`, or `breed` values to list since those are only names
      * Only care about the associated ID values (`dam_id`, `sire_id`, `breedID`)
      */
      Object.keys(this.editAnimalForm.controls).forEach(key => {
        if (key === 'dam' || key === 'sire' || key === 'breed') {
          return;
        } else {
          formData.append(key, this.editAnimalForm.get(key)?.value);
        }
      });

      // Make sure we are passing `formData` that has value
      if (formData) {
        this.dataService.editAnimals(formData)
          .then((response: any) => {
            this.editAlert(response);
          })
          .catch((error: any) => {
            this.editAlert(error);
          });
      } else {
        this.editAlert(null);
      }
    } else {
      this.missingAlert();
    }
  }

  /*
  * If the user taps `Cancel` on the page, check to see if edit mode has been enabled
  * If so, prompt them that they are leaving and any changes will be dismissed unless saved
  */
  async cancelChanges() {
    if (this.editFlag) {
      const alert = await this.alertCtrl.create({
        header: 'Cancel Changes',
        message: 'Are you sure you want to leave this page? ' +
          'Any changes you have entered will be lost!',
        buttons: [
          {
            text: 'Leave Page',
            cssClass: 'alertRed',
            handler: () => {
              /*
              * Navigate back depending on which page we came from
              * If the user navigates to the `Edit Animal` page from `Manage Animals`, then there will not be
                  a `groupID` or `groupName` because all the animals listed have concatenated data
              * So, if an animal exists in more than one group, they will have multiple group names and group IDs associated with them
              * In this case, `groupID` and `groupName` will be set to `null`
              * Else if the user navigated from the `Group Animals` page
                  then `groupID` and `groupName` values should have been passed and set
              * Else, navigate back to `Home` page
              */
              if (this.page === 'manageAnimals' && !this.groupID && !this.groupName) {
                this.router.navigateByUrl(`/tabs/manage-animals`);
              } else if (this.page === 'groupAnimals' && this.groupID && this.groupName) {
                this.router.navigateByUrl(`/tabs/home/manage/group-animals/${this.groupID}/${this.groupName}/0`);
              } else {
                this.router.navigateByUrl(`/tabs/home`);
              }
            }
          },
          {
            text: 'Dismiss',
            role: 'cancel',
            cssClass: 'alertGreen',
          }
        ],
        backdropDismiss: false
      });

      await alert.present();
    } else {
      // Navigate back depending on which page we came from
      if (this.page === 'manageAnimals' && !this.groupID && !this.groupName) {
        this.router.navigateByUrl(`/tabs/manage-animals`);
      } else if (this.page === 'groupAnimals' && this.groupID && this.groupName) {
        this.router.navigateByUrl(`/tabs/home/manage/group-animals/${this.groupID}/${this.groupName}/0`);
      } else {
        this.router.navigateByUrl(`/tabs/home`);
      }
    }
  }

  ionViewDidEnter() {
    // Initialize edit flag to false
    this.editFlag = false;

    // Possible for `groupID` to not be set ("null") if coming from `Manage Animals` page
    if (this.route.snapshot.paramMap.get('groupID') !== 'null') {
      this.groupID = this.route.snapshot.paramMap.get?.('groupID');
      this.groupID = +this.groupID;
    }

    // Possible for `groupName` to not be set ("null") if coming from `Manage Animals` page
    if (this.route.snapshot.paramMap.get('groupName') !== 'null') {
      this.groupName = this.route.snapshot.paramMap.get('groupName');
    }

    /*
    * `animalID` has to exist to be able to edit an animal
    * If there is not an `animalID` value, navigate back to `Home` page
    */
    if (this.route.snapshot.paramMap.get('animalID')) {
      this.animalID = this.route.snapshot.paramMap.get?.('animalID');
      this.animalID = +this.animalID;
    } else {
      this.router.navigateByUrl('/tabs/home');
    }
  }
}
