import { Component, OnInit } from '@angular/core';
import { DataService } from 'src/app/services/data.service';
import { ActivatedRoute, Router, NavigationExtras } from '@angular/router';
import { FormBuilder, FormGroup, Validators, FormControl } from '@angular/forms';
import { AlertController, ModalController } from '@ionic/angular';
import { CategoriesPage } from '../categories/categories.page';
import { Geolocation } from '@capacitor/geolocation';

@Component({
  selector: 'app-edit-group',
  templateUrl: './edit-group.page.html',
  styleUrls: ['./edit-group.page.scss'],
})
export class EditGroupPage implements OnInit {
  public editGroupForm: any;
  public groupID: any;
  public editFlag = false;
  private selected: any;

  constructor(
    private dataService: DataService,
    private route: ActivatedRoute,
    private router: Router,
    private alertCtrl: AlertController,
    private modalCtrl: ModalController,
    public formBuilder: FormBuilder
  ) {
    this.route.queryParams.subscribe(() => {
      if (this.router.getCurrentNavigation()?.extras.state) {
        this.selected = this.router.getCurrentNavigation()?.extras.state?.['data'];

        if (this.selected) {
          this.editGroupForm = formBuilder.group({
            groupID: new FormControl(this.selected.id, Validators.required),
            groupName: new FormControl(this.selected.group_name, [Validators.required, Validators.maxLength(255)]),
            groupCategory: new FormControl(this.selected.category_name, Validators.required),
            groupCategoryID: new FormControl(this.selected.category_id, Validators.required),
            locationName: new FormControl(this.selected.location_name, [Validators.maxLength(255)]),
            latitude: new FormControl(this.selected.latitude, [Validators.maxLength(10)]),
            longitude: new FormControl(this.selected.longitude, [Validators.maxLength(10)]),
            groupDescription: new FormControl(this.selected.groupDescription, [Validators.maxLength(255)])
          });

          // If there are any value changes in the form, set `editFlag` to true; display edit message
          this.editGroupForm.valueChanges.subscribe(() => {
            this.editFlag = true;
          });

        }
      }
    });
  }

  // Get the current location of the user (for setting new group location)
  async getCurrentLocation() {
    const coordinates = await Geolocation.getCurrentPosition();
    if (coordinates) {
      this.editGroupForm.patchValue({
        latitude: +coordinates.coords.longitude,
        longitude: +coordinates.coords.longitude
      });
    }
  }

  // Display alert related to the user attempting to edit the animal
  async editAlert(response: any) {
    if (response !== 'success') {
      const alert = await this.alertCtrl.create({
        header: 'Error Editing Group',
        message: 'An error occurred while trying to apply edits to group. Check your network connection and try again.',
        buttons: ['OK'],
        backdropDismiss: false
      });

      await alert.present();
    } else {
      const navigationExtras: NavigationExtras = {
        state: {
          viewType: 'settings'
        }
      };

      // Navigate to the add page for multiple animals and pass the current group id
      this.router.navigateByUrl('/tabs/home/manage', navigationExtras);
    }
  }

  // If the user is missing the required fields data, alert them
  async missingAlert() {
    const alert = await this.alertCtrl.create({
      header: 'Missing Data',
      message: 'Make sure you have a value for required fields Group Name and Group Category.',
      buttons: ['OK'],
      backdropDismiss: false
    });

    await alert.present();
  }

  /*
  * Save changes for animals or groups when they are done editing
  * Updates data in database
  */
  save() {

    // Grab the current values for group name and group category id (required form fields)
    const groupName = this.editGroupForm.value.groupName;
    const groupCategoryID = this.editGroupForm.value.groupCategoryID;

    // Make sure both form fields have a value since this data is required when creating an animal
    if (groupName && groupCategoryID) {
      const formData = new FormData();

      // Disable form then send data to service to update database record
      Object.keys(this.editGroupForm.controls).forEach(key => {
        if (key !== 'groupCategory') {
          formData.append(key, this.editGroupForm.get(key)?.value);
        }
      });

      if (formData) {
        this.dataService.editGroup(formData)
          .then((response: any) => {
            this.editAlert(response);
          })
          .catch((error: any) => {
            this.editAlert(error);
          });
      } else {
        this.editAlert(null);
      }
    } else {
      this.missingAlert();
    }
  }

  /*
  * If the user clicks 'cancel' on the page, check to see if edit mode has been enabled
  * If so, prompt them that they are leaving and any changes will be
  */
  async cancelChanges() {
    if (this.editFlag) {
      const alert = await this.alertCtrl.create({
        header: 'Cancel Changes',
        message: 'Are you sure you want to leave this page? ' +
          'Any changes you have entered will be lost!',
        buttons: [
          {
            text: 'Leave Page',
            cssClass: 'alertRed',
            handler: () => {
              const navigationExtras: NavigationExtras = {
                state: {
                  viewType: 'settings'
                }
              };
              this.router.navigateByUrl('/tabs/settings/manage', navigationExtras);
            }
          },
          {
            text: 'Dismiss',
            role: 'cancel',
            cssClass: 'alertGreen',
          }
        ],
        backdropDismiss: false
      });

      await alert.present();
    } else {
      const navigationExtras: NavigationExtras = {
        state: {
          viewType: 'settings'
        }
      };
      this.router.navigateByUrl('/tabs/settings/manage', navigationExtras);
    }

  }

  /*
  * Present modal that shows the user's group categories in a list they can select from
    once the user has dismissed the modal, check to see if any data is returned (category)
  * If so, set appropriate others, otherwise, set to empty string
  */
  async groupCategories(view: string) {
    const modal = await this.modalCtrl.create({
      component: CategoriesPage,
      componentProps: {
        viewType: view
      }
    });

    modal.onDidDismiss()
      .then((data: any) => {
        const category = data.data;
        if (category) {
          this.editGroupForm.patchValue({
            groupCategory: category.category_name,
            groupCategoryID: +category.id
          });
        } else {
          this.editGroupForm.patchValue({
            groupCategory: '',
            groupCategoryID: ''
          });
        }
      });

    return await modal.present();
  }

  ionViewDidEnter() {
    // Initialize edit flag to false
    this.editFlag = false;
  }

  ngOnInit() { }

}
