import { Component, OnInit } from '@angular/core';
import { DataService } from 'src/app/services/data.service';
import { AddService } from 'src/app/services/add.service';
import { ActivatedRoute, Router, NavigationExtras } from '@angular/router';
import { AlertController, ModalController } from '@ionic/angular';
import { SearchCriteriaPage } from '../search-criteria/search-criteria.page';


@Component({
  selector: 'app-group-animals',
  templateUrl: './group-animals.page.html',
  styleUrls: ['./group-animals.page.scss'],
})
export class GroupAnimalsPage implements OnInit {
  public appliedFilters: any = [];
  public filteredAnimals: any = []
  public viewType: any;
  public selectedLength = 0;
  public criteria: any;
  public animals: any = [];
  public groupName: any;
  /*
  * Initialize `selectAllFlag` to False to show 'Select All' text on button
  * Toggles to true whenever tapped and changed to 'Deselect All'
  */
  public selectAllFlag = 'unchecked';
  private groupID: any;
  private selected: any = [];
  private filter: any;

  // Set amount to be used when user is adding multiple animals, but they do not know the sexes
  private amount = 0;

  /*
    - Initialize variable for empty string related to sex ID
    - If the user is adding multiple animals and they do not know the number of animal's sexes
        then default to Unknown sex which corresponds to ID 4 in the sexes table of the database
    - This helps users when entering data so they do not have to continually select the sex of each animal
    - This request came per Isaac from notes on July 05, 2022 (20220705)
  */
  private sexID: any;

  constructor(
    private dataService: DataService,
    private addService: AddService,
    private route: ActivatedRoute,
    private router: Router,
    private alertCtrl: AlertController,
    private modalCtrl: ModalController
  ) { }

  // clear all applied filters
  // just reset filtered animals to list of all animals
  clearFilters() {
    this.filteredAnimals = this.animals;
    this.criteria = null;
  }

  applyFilterCriteria(criteria: any) {

    let myList: any;

    if (this.animals.length > 0) {
      myList = this.animals;
    }

    if (myList.length > 0) {
      // https://hackernoon.com/5-techniques-to-iterate-over-javascript-object-entries-and-their-performance-6602dcb708a8
      // for (const [key, value] of Object.entries(criteria)) {
      for (const key in criteria) {
        if (key === 'primaryID' && criteria[key]) {
          myList = myList.filter((item: any) => item.primary_id.toLowerCase().indexOf(criteria[key].toLowerCase()) > -1);
        }

        if (key === 'sexID' && criteria[key]) {
          myList = myList.filter((item: any) => (+item.sex_id === +criteria[key]));
        }

        if (key === 'birthStartDate' && criteria[key]) {
          const startDate = criteria[key];
          myList = myList.filter((item: any) => {
            if (item.date_birth >= startDate) {
              return item.date_birth;
            }
          });
        }

        if (key === 'birthEndDate' && criteria[key]) {
          const endDate = criteria[key];
          myList = myList.filter((item: any) => {
            if (item.date_birth <= endDate) {
              return item.date_birth;
            }
          });
        }

        if (key === 'purchaseStartDate' && criteria[key]) {
          const startDate = criteria[key];
          myList = myList.filter((item: any) => {
            if (item.date_purchase >= startDate) {
              return item.date_purchase;
            }
          });
        }

        if (key === 'purchaseEndDate' && criteria[key]) {
          const endDate = criteria[key];
          myList = myList.filter((item: any) => {
            if (item.date_purchase <= endDate) {
              return item.date_purchase;
            }
          });
        }
        if (key === 'groupIDs' && criteria[key]) {
          myList = myList.filter((item: any) => (+item.group_ids === +criteria[key]));
        }
        if (key === 'categoryIDs' && criteria[key]) {
          myList = myList.filter((item: any) => (+item.category_ids === +criteria[key]));
        }
      }

      this.filteredAnimals = myList;

    }
  }

  // open view to let the user define a search criteria based on multiple filters
  async searchCriteria() {
    const modal = await this.modalCtrl.create({
      component: SearchCriteriaPage,
      componentProps: {
        searchType: this.filter,
        page: 'group-animals'
      }
    });

    modal.onDidDismiss()
      .then((data: any) => {
        if (data.data) {
          this.filteredAnimals = this.animals;
          console.log(data.data);
          this.criteria = data.data;
          // this.criteria = data.data.criteria;
          console.log('modal dismiss criteria:', this.criteria);
          // this.order = data.data.order;
          // console.log('modal dismiss order:', this.order);

          // this.applyFilterCriteria(this.criteria, this.order);
          this.applyFilterCriteria(this.criteria);
        }
      })
      .catch((error: any) => {
        console.log('Actions Modal Search Criteria Error', error);
      });

    return await modal.present();
  }

  // Once the user has added some items into the temporary list, then they can continue to the next page which will allow them to select
  // from a list of actions
  actionsContinue() {

    console.log(`actionsContinue ${this.selected}`);

    // Pass temporary list of selected items to "Actions Continue" page
    // This page will allow the user to select an action to apply to the temporary list of selected items
    const navigationExtras: NavigationExtras = {
      state: {
        itemType: 'animals',
        sexID: 0,
        selected: this.selected
      }
    };

    this.router.navigate(['/tabs/home/actions/actions-continue'], navigationExtras);
  }

  // detect when a checkbox for a group has been selected/deselected; adjust our selected list as needed
  checkboxChanged(selection: any) {
    console.log('checkboxChanged selection:', selection);
    console.log('checkboxChanged selection id:', selection.id);
    console.log('checkboxChange selection checked status:', selection.isChecked);
    console.log('selected list BEFORE:', this.selected);

    console.log('is this in selected list?', this.selected.includes(selection));
    console.log('is this in selected list 2?', this.selected.some((s: any) => s.id === selection.id));


    // BACKWARDS DUE TO NO STARTING CHECKED VALUE???
    // isChecked = true means UNCHECKED
    // isChecked = false means CHECKED


    if (this.selected.includes(selection)) {
      // if we have items in our selected list and the current selected item already exists in the list,
      // then the user is "un-check"/"deselect"-ing the value, so we remove it from out list
      const index = this.selected.indexOf(selection);
      console.log('Index to splice from Selected list: ', index);
      this.selected.splice(index, 1);
    } else {
      console.log('push');
      // else, there are no items in our selected list or the current selected item is not in the list
      // push the selection to our list of selected items
      this.selected.push(selection);
    }


    console.log('select list AFTER:', this.selected);

    // update current length of selected (used to enable/disable 'Continue' button)
    this.selectedLength = this.selected.length;
  }

  // select all current list items
  selectAll(event: any) {
    console.log('selectAll event checked:', event.target.checked);

    const items = this.filteredAnimals;

    /*
    * True/false for event checked is backwards???
    * False means it is checked, true means it is unchecked
    * If, Select All is checked/tapped (false); set current as true; append to list
    * Else, Select All is unchecked (true); unchecked all and clear list
    */
    if (!event.target.checked) {
      this.selectAllFlag = 'checked';

      // iterate through items and set all to checked
      // push to list of selected items if they are not already in the list
      for (const i of items) {
        i.isChecked = true;
        if (!this.selected.includes(i)) {
          this.selected.push(i);
        }
      }


    } else if (event.target.checked) {
      // else if the user taps "Deselect All", all value are removed from list
      this.selectAllFlag = 'unchecked';

      // un-check/deselect all items
      for (const i of items) {
        i.isChecked = false;
      }

      // set list of selected items to empty
      this.selected = [];

    }

    console.log('selected', this.selected);

    // update current length of selected (used to enable/disable 'Continue' button)
    this.selectedLength = this.selected.length;
  }


  editAnimal(index: number) {
    /*
    * When the user taps on an individual animal in the group animals list,
        open view for that specific animal so the user can view all details and edit if desired
    */

    const navigationExtras: NavigationExtras = {
      state: {
        data: this.filteredAnimals[index],
        page: 'groupAnimals'
      }
    };

    /*
    Passed navigation data: `...edit-animal/group/{groupID}/{groupName}/animal/{animalID}
    * If the user is editing the animal from the Search tab (list of all animals), then there is not
        a specific group ID or group name value to pass
    * Else, if the user is editing an animal from inside the Herd Inventory --> Groups (View) --> Group Animals
        then pass appropriate group data
        - i.e., `...edit-animal/group/41/feb2022/animal/543
    */
    this.router.navigate(
      [`/tabs/home/manage/edit-animal/group/${this.groupID}/${this.groupName}/animal/${this.filteredAnimals[index].id}`],
      navigationExtras
    );
  }

  async multipleAnimals() {
    console.log(`group-animals.page.ts multipleAnimals()`);

    // Prompt when adding multiple animals
    // When the user DOES NOT know the sex of animals
    const alert = await this.alertCtrl.create({
      header: 'Amount of Animals',
      message: 'How many animals are you adding?',
      inputs: [
        {
          name: 'amount',
          type: 'number',
          placeholder: 'Enter amount of animals',
          min: 0,
          max: 100
        },
      ],
      buttons: [
        {
          text: 'Cancel',
          role: 'cancel',
          cssClass: 'alertRed'
        },
        {
          text: 'Continue',
          cssClass: 'alertGreen',
          handler: (data) => {
            /*
            * Check to see that the user input an amount of groups they are wanting to add
            * If, there is an amount, continue to add group page and pass amount
            * Else, continue but default the amount to 1
            */
            if (+data.amount) {
              if (+data.amount > 100) {
                // Limit to user to only add a maximum of 100 groups at a time to prevent potential abuse
                this.addService.displayAlert('Too Many Animals',
                  'You can only add a maximum of 100 animals at a time.');
              } else if (+data.amount == 1) {
                // Prevent user from using multiple animal functionality for just a single (1) animal
                this.addService.displayAlert('Too Few Animals',
                  'Enter more than 1 animal when adding multiple. Otherwise use Add Individual Animal.');
              } else {
                this.amount = +data.amount;
                console.log(`We are adding ${this.amount} animal(s)`);
                // Pass false because the user DOES NOT know the number of animal sexes
                this.multipleAdditionTypes(false);
              }
            }
          }
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  async multipleAdditionTypes(knowSexes: boolean) {
    console.log('multipleAdditionTypes()');
    console.log(`knowSexes: ${knowSexes}`);
    /*
    * Pass boolean (true/false) depending on if the user knows the number of animal sexes when adding multiple animals
    * True = yes, they do know the number of animal sexes
    * False = no, they do not know the number of animal sexes
    */

    const alert = await this.alertCtrl.create({
      header: 'Addition Type of Animals',
      message: '',
      buttons: [
        {
          text: 'Birth',
          cssClass: 'alertGreen',
          handler: () => {
            console.log('Adding animals as "Birth"');
            this.multipleEntry(knowSexes, 1);
          }
        },
        {
          text: 'Purchase',
          cssClass: 'alertGreen',
          handler: () => {
            console.log('Adding animals as "Purchase"');
            this.multipleEntry(knowSexes, 2);
          }
        },
        {
          text: 'Cancel',
          role: 'cancel',
          cssClass: 'alertRed',
        }
      ],
      backdropDismiss: false
    });
    await alert.present();
  }

  async multipleEntry(knowSexes: boolean, additionType: number) {
    // Prompt the user for input regarding a prefix and the number of animal (sexes) they will be adding

    console.log('multipleEntry');
    console.log(`knowSexes: ${knowSexes}`);
    console.log(`additionType: ${additionType}`);

    // Initialize alert, set value based on addition type
    let alert: any;

    if (knowSexes) {
      // If, variable `knowSexes` is true, prompt user to enter input for number of each animal as well as prefix and date
      if (additionType == 1) {
        console.log(`knowSexes (TRUE), additionType (1)`);
        // if additionType is 1 (birth), display birth date option
        alert = await this.alertCtrl.create({
          header: 'Animals Information',
          message: '(Required) Enter a prefix to identify the animals, the amount per sex you are adding ' +
            'and a birth date (optional) to be applied for all animals. ' +
            '<br><br>' +
            'The animals will be automatically added once you tap "Save"',
          inputs: [
            {
              name: 'prefix',
              type: 'text',
              placeholder: 'Enter identification prefix'
            },
            {
              name: 'numberBulls',
              type: 'number',
              placeholder: 'Enter number of bulls',
              min: 0,
              max: 100
            },
            {
              name: 'numberSteers',
              type: 'number',
              placeholder: 'Enter number of steers',
              min: 0,
              max: 100
            },
            {
              name: 'numberCows',
              type: 'number',
              placeholder: 'Enter number of cows',
              min: 0,
              max: 100
            },
            {
              name: 'birthDate',
              type: 'date',
            }
          ],
          buttons: [
            {
              text: 'Cancel',
              role: 'cancel',
              cssClass: 'alertRed'
            },
            {
              text: 'Save',
              cssClass: 'alertGreen',
              handler: (data: any) => {
                // to create multiple animals, the user needs to define a prefix for the animals, otherwise we cannot keep track of them
                if (data.prefix) {

                  // if none of the sexes have a value or they are not numeric value (number) set value to 0
                  if (!data.numberBulls || isNaN(+data.numberBulls)) {
                    data.numberBulls = 0;
                  }

                  if (!data.numberSteers || isNaN(+data.numberSteers)) {
                    data.numberSteers = 0;
                  }

                  if (!data.numberCows || isNaN(+data.numberCows)) {
                    data.numberCows = 0;
                  }

                  // Calculate the total of all sexes; user limited to adding maximum of 100 total at a time
                  const total = +data.numberBulls + +data.numberSteers + +data.numberCows;

                  if (total == 1) {
                    this.addService.displayAlert('Too Few Animals',
                      'Enter more than 1 animal when adding multiple. Otherwise use Add Individual Animal.');
                  } else {
                    /*
                    * If some sexes of animals have a value greater than 0 and our total animals to be created is less than or equal to 100
                        then create multiple animals with amount of sexes; pass empty string for purchase date since not defined in this instance
                    */
                    if ((+data.numberBulls > 0 || +data.numberSteers > 0 || +data.numberCows > 0) && total <= 100) {
                      this.addService.createMultiple(this.groupID, this.groupName, additionType,
                        data.prefix, +data.numberBulls, +data.numberSteers, +data.numberCows, data.birthDate, '');
                    } else if (+data.numberBulls === 0 && +data.numberSteers === 0 && +data.numberCows === 0) {
                      // Else if, all the sexes have an amount of 0, show alert to user that nothing was created due to all amounts being 0
                      this.addService.displayAlert('No Animals Created',
                        'No animals were created since amounts for bull, steer, and cow were all 0.');
                    } else if (total > 100) {
                      // Else if, the user is attempting to add more than 100 animals, show alert when attempting to add too many animals
                      this.addService.displayAlert('Too Many Animals',
                        'You can only add a maximum of 100 animals at a time.');
                    } else {
                      // Else, someone else happens, show error alert
                      this.addService.displayAlert('Error Alert',
                        'An error occurred while trying to create animals. Check your network connection and try again.');
                    }
                  }
                } else {
                  this.addService.displayAlert('No Prefix Defined',
                    'No prefix for the animals to be created was defined. Animals cannot be added unless a prefix is provided.');
                }
              }
            }
          ],
          backdropDismiss: false
        });
      } else if (additionType == 2) {
        console.log(`knowSexes (TRUE), additionType (2)`);
        // if additionType is 2 (purchase), display both birth and purchase date options
        alert = await this.alertCtrl.create({
          header: 'Animals Information',
          message: '(Required) Enter a prefix to identify the animals, the amount per sex you are adding ' +
            'and a birth date (optional) and purchase date (optional) to be applied for all animals. ' +
            '<br><br>' +
            'The animals will be automatically added once you tap "Save"',
          inputs: [
            {
              name: 'prefix',
              type: 'text',
              placeholder: 'Enter identification prefix'
            },
            {
              name: 'numberBulls',
              type: 'number',
              placeholder: 'Enter number of bulls',
              min: 0,
              max: 100
            },
            {
              name: 'numberSteers',
              type: 'number',
              placeholder: 'Enter number of steers',
              min: 0,
              max: 100
            },
            {
              name: 'numberCows',
              type: 'number',
              placeholder: 'Enter number of cows',
              min: 0,
              max: 100
            },
            {
              name: 'birthDate',
              type: 'date',
            },
            {
              name: 'purchaseDate',
              type: 'date',
            }
          ],
          buttons: [
            {
              text: 'Cancel',
              role: 'cancel',
              cssClass: 'alertRed'
            },
            {
              text: 'Save',
              cssClass: 'alertGreen',
              handler: (data: any) => {
                // to create multiple animals, the user needs to define a prefix for the animals, otherwise we cannot keep track of them
                if (data.prefix) {

                  // if none of the sexes have a value or they are not numeric value (number) set value to 0
                  if (!data.numberBulls || isNaN(+data.numberBulls)) {
                    data.numberBulls = 0;
                  }

                  if (!data.numberSteers || isNaN(+data.numberSteers)) {
                    data.numberSteers = 0;
                  }

                  if (!data.numberCows || isNaN(+data.numberCows)) {
                    data.numberCows = 0;
                  }

                  // Calculate the total of all sexes; user limited to adding maximum of 100 total at a time
                  const total = +data.numberBulls + +data.numberSteers + +data.numberCows;

                  /*
                  * If some sexes of animals have a value greater than 0 and our total animals to be created is less than or equal to 100
                      then create multiple animals with amount of sexes
                  */
                  if ((+data.numberBulls > 0 || +data.numberSteers > 0 || +data.numberCows > 0) && total <= 100) {
                    this.addService.createMultiple(this.groupID, this.groupName, additionType,
                      data.prefix, +data.numberBulls, +data.numberSteers, +data.numberCows, data.birthDate, data.purchaseDate);
                  } else if (+data.numberBulls === 0 && +data.numberSteers === 0 && +data.numberCows === 0) {
                    // Else if, all the sexes have an amount of 0, show alert to user that nothing was created due to all amounts being 0
                    this.addService.displayAlert('No Animals Created',
                      'No animals were created since amounts for bull, steer, and cow were all 0.');
                  } else if (total > 100) {
                    // Else if, the user is attempting to add more than 100 animals, show alert when attempting to add too many animals
                    this.addService.displayAlert('Too Many Animals',
                      'You can only add a maximum of 100 animals at a time.');
                  } else {
                    // Else, someone else happens, show error alert
                    this.addService.displayAlert('Error Alert',
                      'An error occurred while trying to create animals. Check your network connection and try again.');
                  }
                } else {
                  this.addService.displayAlert('No Prefix Defined',
                    'No prefix for the animals to be created was defined. Animals cannot be added unless a prefix is provided.');
                }
              }
            }
          ],
          backdropDismiss: false
        });
      }
    } else {
      // else, variable `knowSexes` is false, prompt user to only enter prefix and date
      if (additionType === 1) {
        console.log(`knowSexes (FALSE), additionType (1)`);
        // if additionType is 1 (birth), display birth date option
        alert = await this.alertCtrl.create({
          header: 'Animals Information',
          message: ' Enter a prefix (required) to identify the animals and a birth date (optional) to be applied for all animals. ' +
            '<br><br>' +
            'You will proceed with entering information for the animals after you tap "Save"',
          inputs: [
            {
              name: 'prefix',
              type: 'text',
              placeholder: 'Enter identification prefix'
            },
            {
              name: 'birthDate',
              type: 'date',
            }
          ],
          buttons: [
            {
              text: 'Cancel',
              role: 'cancel',
              cssClass: 'alertRed'
            },
            {
              text: 'Save',
              cssClass: 'alertGreen',
              handler: (data: any) => {
                console.log(`data.prefix ${data.prefix}`);
                console.log(`data.birthDate ${data.birthDate}`);
                console.log(`additionType ${additionType}`);
                console.log(`this.groupID ${this.groupID}`);
                console.log(`this.groupName: ${this.groupName}`);
                console.log(`this.sexID: ${this.sexID}`);
                // To create multiple animals, the user needs to define a prefix for the animals, otherwise we cannot keep track of them
                if (data.prefix) {
                  const navigationExtras: NavigationExtras = {
                    state: {
                      prefix: data.prefix,
                      birthDate: data.birthDate,
                      additionType: additionType.toString(),
                      groupID: this.groupID,
                      groupName: this.groupName,
                      sexID: this.sexID.toString()
                    }
                  };

                  console.log(`navigationExtras prefix: ${navigationExtras.state?.['prefix']}`);
                  console.log(`navigationExtras birthDate: ${navigationExtras.state?.['birthDate']}`);
                  console.log(`navigationExtras additionType: ${navigationExtras.state?.['additionType']}`);
                  console.log(`navigationExtras groupID ${navigationExtras.state?.['groupID']}`);
                  console.log(`navigationExtras groupName: ${navigationExtras.state?.['groupName']}`);
                  console.log(`navigationExtras sexID: ${navigationExtras.state?.['sexID']}`);

                  // Navigate to the add page for multiple animals and pass the current group ID
                  this.router.navigateByUrl(`/tabs/home/manage/add-animal/multiple/${this.amount}`, navigationExtras);

                } else {
                  this.addService.displayAlert('No Prefix Defined',
                    'No prefix for the animals to be created was defined. Animals cannot be added unless a prefix is provided.');
                }
              }
            }
          ],
          backdropDismiss: false
        });
      } else if (additionType === 2) {
        console.log(`knowSexes (FALSE), additionType (2)`);
        // if additionType is 2 (purchase), display both birth and purchase date options
        alert = await this.alertCtrl.create({
          header: 'Animals Information',
          message: ' Enter a prefix (required) to identify the animals and a birth date (optional) and purchase date (optional) to be applied for all animals. ' +
            '<br><br>' +
            'You will proceed with entering information for the animals after you tap "Save"',
          inputs: [
            {
              name: 'prefix',
              type: 'text',
              placeholder: 'Enter identification prefix'

            },
            {
              name: 'birthDate',
              type: 'date',
            },
            {
              name: 'purchaseDate',
              type: 'date',
            }
          ],
          buttons: [
            {
              text: 'Cancel',
              role: 'cancel',
              cssClass: 'alertRed'
            },
            {
              text: 'Save',
              cssClass: 'alertGreen',
              handler: (data: any) => {
                console.log(`data ${data}`);
                console.log(`sexID: ${this.sexID}`);
                // To create multiple animals, the user needs to define a prefix for the animals, otherwise we cannot keep track of them
                if (data.prefix) {
                  const navigationExtras: NavigationExtras = {
                    state: {
                      prefix: data.prefix,
                      birthDate: data.birthDate,
                      purchaseDate: data.purchaseDate,
                      additionType: additionType.toString(),
                      groupID: this.groupID,
                      groupName: this.groupName,
                      sexID: this.sexID.toString()
                    }
                  };

                  console.log(`navigationExtras prefix: ${navigationExtras.state?.['prefix']}`);
                  console.log(`navigationExtras birthDate: ${navigationExtras.state?.['birthDate']}`);
                  console.log(`navigationExtras purchaseDate: ${navigationExtras.state?.['purchaseDate']}`);
                  console.log(`navigationExtras additionType: ${navigationExtras.state?.['additionType']}`);
                  console.log(`navigationExtras groupID ${navigationExtras.state?.['groupID']}`);
                  console.log(`navigationExtras groupName: ${navigationExtras.state?.['groupName']}`);
                  console.log(`navigationExtras sexID: ${navigationExtras.state?.['sexID']}`);

                  // Navigate to the add page for multiple animals and pass the current group ID
                  this.router.navigateByUrl(`/tabs/home/manage/add-animal/multiple/${this.amount}`, navigationExtras);

                } else {
                  this.addService.displayAlert('No Prefix Defined',
                    'No prefix for the animals to be created was defined. Animals cannot be added unless a prefix is provided.');
                }
              }
            }
          ],
          backdropDismiss: false
        });
      }

    }

    await alert.present();
  }

  /*
  * Prompt user to determine if they know the number of sexes for the animals they are adding
    this helps automatically created X number of animals per sex since a primary id and sex is required
    to add an animal to a user's account
  */
  async multipleSexes() {
    console.log(`multipleSexes`);
    const alert = await this.alertCtrl.create({
      // Number of Animal's Sexes
      header: 'Number of Animal\'s Sexes',
      message: 'Do you know how many animals of each sex (i.e., steer, bull, cow) you are adding?',
      buttons: [
        {
          text: 'Yes',
          cssClass: 'alertGreen',
          handler: () => {
            console.log('We DO know the number of animal sexes');
            // Pass true because the user DOES know the number of animal sexes
            this.multipleAdditionTypes(true);
          }
        },
        {
          text: 'No',
          cssClass: 'alertGreen',
          handler: () => {
            console.log('We DO NOT know the number of animal sexes, set sex to "Unknown" (ID=4)');
            // The user does not know the number of animal sexes, so set sex as "Unknown" (ID = 4)
            this.sexID = 4;
            // Go to another function to enter # of animals
            this.multipleAnimals();
          }
        },
        {
          text: 'Cancel',
          role: 'cancel',
          cssClass: 'alertRed',
        }
      ],
      backdropDismiss: false
    });
    await alert.present();
  }

  async addAnimal() {
    // Display alert (prompt) for user to decide on adding animals (individual, multiple, existing)
    const alert = await this.alertCtrl.create({
      header: 'Add Animal',
      message: 'Would you like to add an individual animal, multiple animals or existing animals?',
      buttons: [
        {
          text: 'Individual Animal',
          cssClass: 'alertGreen',
          handler: () => {
            const navigationExtras: NavigationExtras = {
              state: {
                groupID: this.groupID,
                groupName: this.groupName,
              }
            };
            this.router.navigateByUrl('/tabs/home/manage/add-animal/individual/1', navigationExtras);
          }
        },
        {
          text: 'Multiple Animals',
          cssClass: 'alertGreen',
          handler: () => {
            this.multipleSexes();
          }
        },
        {
          text: 'Existing Animals',
          cssClass: 'alertGreen',
          handler: () => {
            const navigationExtras: NavigationExtras = {
              state: {
                id: this.groupID,
                name: this.groupName
              }
            };

            this.router.navigateByUrl('/tabs/home/manage/add-existing', navigationExtras);
          }
        },
        {
          text: 'Cancel',
          role: 'cancel',
          cssClass: 'alertRed',
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  // Filter animals based on primary ID
  filterAnimals(event: any) {
    return this.filteredAnimals = this.animals.filter((animal: any) =>
      animal.primary_id.toLowerCase().indexOf(event.detail.value.toLowerCase()) > -1);
  }

  ionViewDidEnter() {
    if (this.groupID) {
      this.dataService.get('getGroupAnimals', this.groupID)
        .then((data: any) => {
          /*
          * If response contains a value and has a length > 0
              then there are some animals for the current group
          * Set `animals` and `filteredAnimals` variables with the result
          */
          if (data && data.length > 0) {
            this.dataService.presentLoading();
            this.animals = data;
            console.log(this.animals);
            this.filteredAnimals = this.animals;
          }
        })
        .catch((error: any) => {
          console.log(`Group Animals Error: ${error}`);
          this.addService.displayAlert('Group Animals Error',
            'An error occurred while retrieving group animals. Check your network connection and try again.');
        });
    } else {
      console.log('Group Animals Error: `groupID` unavailable.');
      this.addService.displayAlert('Group ID Error',
        'Unable to determine `Group ID`. Please try again.');
    }
  }

  ngOnInit() {
    if (
      this.route.snapshot.paramMap.get('groupID') &&
      this.route.snapshot.paramMap.get('groupName') &&
      this.route.snapshot.paramMap.get('view')
    ) {
      this.groupID = this.route.snapshot.paramMap.get?.('groupID');
      this.groupID = +this.groupID;
      this.groupName = this.route.snapshot.paramMap.get?.('groupName');
      this.viewType = this.route.snapshot.paramMap.get?.('view');
      console.log(`group-animals ngOnInit groupID: ${this.groupID}`);
      console.log(`group-animals ngOnInit groupName: ${this.groupName}`);
      console.log(`group-animals ngOnInit viewType: ${this.viewType}`);
    } else {
      this.router.navigateByUrl('/tabs/home');
    }
  }

}
