import { Component, OnInit } from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { FormBuilder, FormGroup, Validators, FormControl } from '@angular/forms';
import { HttpClient } from '@angular/common/http';
import { DataService } from 'src/app/services/data.service';
import { ActionsService } from 'src/app/services/actions.service';
import { ModalController } from '@ionic/angular';

@Component({
  selector: 'app-health',
  templateUrl: './health.page.html',
  styleUrls: ['./health.page.scss'],
})
export class HealthPage implements OnInit {
  public action: string = '';
  public actionID: number = 0;
  public actionType: string = '';
  public bodySystems: any;
  public bodySystemChecked = false;
  public conditions: any;
  public filteredProducts: any;
  public isProductAvailable = false;
  public isProductOther = false;
  public observationForm: FormGroup;
  public productSelected: any;
  public treatmentForm: FormGroup;

  public routes = [
    { id: 1, name: 'Subcutaneously (SQ)' },
    { id: 2, name: 'Intramuscularly (IM)' },
    { id: 3, name: 'Intravenously (IV)' },
    { id: 4, name: 'As an epidural' },
    { id: 5, name: 'Fat pad behind ear' },
    { id: 6, name: 'Each Nostril' },
    { id: 7, name: 'Eye' },
    { id: 8, name: 'Left Eye' },
    { id: 9, name: 'Right Eye' },
    { id: 10, name: 'Intradermally' },
    { id: 11, name: 'Intramammary' },
    { id: 12, name: 'Intranasally' },
    { id: 13, name: 'Intrauterine' },
    { id: 14, name: 'Orally' },
    { id: 15, name: 'Paravertebrally' },
    { id: 16, name: 'Rectally' },
    { id: 17, name: 'Topically' }
  ];

  // Set variable related to if the Health page is being served normally from the Actions page or from the Custom Form page as a modal
  public isModal = false;

  // number: 2 = Health Event; Reference database `action_types` table
  private actionTypeID: number = 2;
  private healthIDs: any = [];
  private itemType: string = '';
  private products: any;
  private productOtherName: any;
  public selected: any;
  private selectedAmount = '';
  private selectedBodySystem: any;
  private selectedBodyCondition: any;
  private selectedRoute: any;

  constructor(
    public formBuilder: FormBuilder,
    private route: ActivatedRoute,
    private router: Router,
    private httpClient: HttpClient,
    private dataService: DataService,
    private actionsService: ActionsService,
    private modalCtrl: ModalController
  ) {
    // Determine if the user is navigating to the Health page normally or via a modal from the Custom Form page in the Actions portion
    if (!this.isModal) {
      this.route.queryParams.subscribe(_ => {
        if (this.router.getCurrentNavigation()) {
          if (this.router.getCurrentNavigation()?.extras.state) {
            // `animals` or `groups`
            this.itemType = this.router.getCurrentNavigation()?.extras.state?.['itemType'];
            // e.g., `health`
            this.actionType = this.router.getCurrentNavigation()?.extras.state?.['actionType'];
            // e.g. `treatment` or `observation`
            this.action = this.router.getCurrentNavigation()?.extras.state?.['action'];
            // animals or groups selected by the user
            this.selected = this.router.getCurrentNavigation()?.extras.state?.['selected'];
            // `single` or `multiple`
            this.selectedAmount = this.router.getCurrentNavigation()?.extras.state?.['selectedAmount'];
          }
        }
      });
    }

    this.observationForm = formBuilder.group({
      system: new FormControl('', Validators.required),
      condition: new FormControl('', Validators.required),
      date: new FormControl('', Validators.required),
      weight: new FormControl(''),
      notes: new FormControl('')
    });

    this.treatmentForm = formBuilder.group({
      date: new FormControl('', Validators.required),
      volume: new FormControl(''),
      units: new FormControl(''),
      product: new FormControl('', Validators.required),
      productOther: new FormControl(''),
      route: new FormControl('', Validators.required),
      reason: new FormControl('', Validators.required),
      system: new FormControl('', Validators.required),
      condition: new FormControl('', Validators.required),
      weight: new FormControl(''),
      notes: new FormControl('')
    });
  }

  // If the user taps "Close", the modal will be dismissed
  dismiss() {
    this.modalCtrl.dismiss();
  }

  /*
  * If the user taps "Done" (required data entered), continue with processing the form
  */
  done(action: string) {
    // Initialize empty dictionary and form
    let formDictionary: any = {};
    let form: any;

    // Determine which type of form the user was using and set form variable accordingly
    if (action === 'observation') {
      form = this.observationForm;
    } else if (action === 'treatment') {
      form = this.treatmentForm;
    }

    // Iterate through the form data and add the key/value pairings to the dictionary
    Object.keys(form.controls).forEach(key => {
      // console.log(`key: ${key}`);
      // console.log(`form.get(key).value: ${form.get(key).value}`);
      formDictionary[key] = form.get(key).value;
      // console.log(formDictionary);
    });

    // If the formDictionary has data, continue with making call and passing data for database record insertion
    if (formDictionary) {
      // formDictionary example: (2, 'observation', 'animals', [31], 'single', {formData})

      /*
      * If the Health page was navigated to via modal from the Actions Custom Processing Form
      * Dismiss the modal and return the formDictionary
      */
      if (this.isModal) {
        // Determine if the current Action is for an Observation or Treatment and set returned data accordingly
        if (this.action === 'observation') {
          // If the Action is an Observation, then less data is returned (form dictionary, body system and body condition)
          this.modalCtrl.dismiss([formDictionary, this.selectedBodySystem, this.selectedBodyCondition]);
        } else if (this.action === 'treatment') {
          /*
          * If the Action is a Treatment, then more data is returned
          * Next, determine if the selected product is from the pre-defined list or the user chose Other and provided their own
          * Return data (form dictionary, body system, body condition, selected product, other product name, route of administration)
          */
          if (this.isProductOther) {
            // Grab name of "Other" product from the user input
            this.productOtherName = this.treatmentForm.controls['productOther'].value.trim();
            this.modalCtrl.dismiss([formDictionary, this.selectedBodySystem, this.selectedBodyCondition,
              this.productSelected, this.productOtherName, this.selectedRoute]);
          } else {
            this.modalCtrl.dismiss([formDictionary, this.selectedBodySystem, this.selectedBodyCondition,
              this.productSelected, null, this.selectedRoute]);
          }
        }
      } else {
        /*
        * Else, the user navigated to the Health page normally via Actions page
        * Call data service to insert Health event into database
        */
        this.dataService.addHealthEvent(this.actionTypeID, action,
          this.itemType, this.healthIDs, this.selectedAmount, formDictionary)
          .then((response: any) => {
            console.log(response);
            /*
            * If there is a response (successful), then reset the list
            * Display message to user and navigate back to Actions page
            */
            if (response) {
              this.actionsService.actionToast('health', this.itemType);
              this.router.navigateByUrl('/tabs/home');
            } else {
              // Else, there is not a response, display error message to user
              this.actionsService.actionError('Actions Health Event Response Error');
            }
          })
          .catch((error: any) => {
            // Else, there is an unexpected error, display error message to user
            this.actionsService.actionError(error);
          });
      }
    }
  }

  // When the user changes the Route of Administration, set variable to keep track of selected route
  routeChange(event: any) {
    if (event) {
      const filteredRoute = this.routes.filter((r: any) => r.id === +event.detail.value);
      if (filteredRoute) {
        this.selectedRoute = filteredRoute[0]?.name;
      }
    }
  }

  // Set product (drug) for treatment
  setProduct(product: any) {
    if (product) {
      this.productSelected = product;
      this.isProductAvailable = false;

      // Patch the form value for `product`
      this.treatmentForm.patchValue({
        product: this.productSelected
      });

      if (product === 'Other' && !this.isProductAvailable) {
        this.isProductOther = true;
      } else {
        this.isProductOther = false;
      }
    }
  }

  // Grab products (drugs) data from local JSON file
  initializeItems() {
    return this.httpClient.get('/assets/data/drugs.json');
  }

  // When the user begins searching (typing) a product (drug), filter results from `drugs.json`
  getProducts(event: any) {
    console.log(this.products);
    console.log(event);
    console.log(event.target.value);
    /*
    * If products (drugs) data has already been called and set into variable
      then only worry about filtering results based on user input
    */
    if (this.products) {
      // Set `inputValue` to the value of the search bar
      const inputValue = event.target.value;

      // If `inputValue` is an empty string then don't filter the items
      if (inputValue && inputValue.trim() !== '') {
        this.isProductAvailable = true;
        this.filteredProducts = this.products.filter((item: any) =>
          /*
          * Return any occurrence of the inputValue in the string
          * i.e., `anti` would show a result that includes: `ReBalance Antiprotozoal`
          */
          (item.toLowerCase().indexOf(inputValue.toLowerCase()) > -1)
        );

      } else {
        /*
        * Else, the `inputValue` is empty, either there is no input or the user cleared existing input
        * Reset variables
        */
        this.isProductAvailable = false;
        this.isProductOther = false;
        this.productSelected = null;
      }
    } else {
      /*
      * Else, no product (drug) data has been called
      * Get data from local `drugs.json` file
      * NOTE: This should be done on initial page load, however, if there is a problem
          attempt to retrieve the list of products (drugs) again
      */
      this.initializeItems().subscribe(resp => {
        if (resp) {
          this.products = resp;
          this.getProducts(event);
        } else {
          this.isProductAvailable = false;
        }
      });
    }
  }

  // If the user selects a new system, change the available body condition options
  systemChange(action: string, event: any) {
    // Check for value, since the form will detect change when the body systems are applied as form values when the page loads
    if (event) {
      if (action === 'observation') {
        // Reset value for conditions when body system select option is changed
        this.observationForm.patchValue({
          condition: ''
        });
      } else if (action === 'treatment') {
        // Reset value for conditions when body system select option is changed
        this.treatmentForm.patchValue({
          condition: ''
        });
      }

      this.bodySystemChecked = true;

      /*
      * If there is a list of body systems and the user has selected one
      * Filter the systems based on the ID that is equal to the passed selected value from the user
      */
      if (this.bodySystems && this.bodySystemChecked) {
        const filteredBodySystem = this.bodySystems.filter((s: any) => s.id === +event.detail.value);
        // If a result was found from filtering based on ID, set conditions list equal to that body system's conditions
        if (filteredBodySystem && filteredBodySystem[0] && filteredBodySystem[0].system) {
          this.selectedBodySystem = filteredBodySystem[0].system;
          this.conditions = filteredBodySystem[0].conditions;
        } else {
          this.selectedBodySystem = null;
          this.conditions = null;
        }
      }
    }
  }

  /*
  * When the user selects (changes) a condition, get type and set variable
  * Used to be passed back to modal from Custom Form page under Actions
  */
  conditionChange(event: any) {
    if (event && this.conditions) {
      const filteredBodyCondition = this.conditions.filter((c: any) => c.id === +event.detail.value);
      this.selectedBodyCondition = filteredBodyCondition[0].type;
    } else {
      this.selectedBodyCondition = null;
    }
  }

  /*
  * If the user decides to cancel applying actions to their selected items
  * Navigate back to the main `Home` page
  */
  cancelActions() {
    this.router.navigateByUrl('/tabs/home');
  }

  getBodySystems() {
    return this.httpClient.get('/assets/data/body_systems.json');
  }

  /*
  * When the `Health Event` page loads
  * Grab the `body_systems.json` file from local device
  * If not available, navigate back to `Home` page
  */
  ionViewDidEnter() {
    this.getBodySystems().subscribe(resp => {
      if (resp) {
        this.bodySystems = resp;
      } else {
        this.router.navigateByUrl('/tabs/home');
      }
    });

    // Determine if the user is inputting health data from the Custom Form page Treatment option
    if (!this.isModal) {
      /*
      * If there are selected values (animals or groups from the user)
      * Iterate through the selections and append the associated ID
          to the list of `healthIDs` used for writing records to database
      */
      if (this.selected) {
        for (const s of this.selected) {
          this.healthIDs.push(s.id);
        }
      } else {
        this.router.navigateByUrl('/tabs/home');
      }
    } else {
      console.log('Not a modal');
    }
  }

  ngOnInit() {
    // On initial page load, grab the list of products (drugs) from local device (JSON file)
    this.initializeItems().subscribe(resp => {
      this.dataService.presentLoading();
      if (resp) {
        this.products = resp;
      }
    });
  }

}
