import { Component, OnInit } from '@angular/core';
import { DataService } from 'src/app/services/data.service';
import { AddService } from 'src/app/services/add.service';
import { Router, NavigationExtras } from '@angular/router';
import { SearchCriteriaPage } from '../search-criteria/search-criteria.page';
import { ModalController } from '@ionic/angular';

@Component({
  selector: 'app-manage-animals',
  templateUrl: './manage-animals.page.html',
  styleUrls: ['./manage-animals.page.scss'],
})
export class ManageAnimalsPage implements OnInit {
  public animals: any = [];
  public filteredAnimals: any = [];
  public sexType = '';
  public filter: any;
  public criteria: any;
  public appliedFilters: any = [];

  constructor(
    private dataService: DataService,
    private addService: AddService,
    private router: Router,
    private modalCtrl: ModalController
  ) { }

  /*
  * When the user taps on an individual animal in the list
      open view for that specific animal
      so the user can view all details and edit if desired
  */
  editAnimal(index: number) {
    const navigationExtras: NavigationExtras = {
      state: {
        data: this.filteredAnimals[index],
        page: 'manageAnimals'
      }
    };

    /*
    Passed navigation data: `...edit-animal/group/{groupID}/{groupName}/animal/{animalID}
    * If the user is editing the animal from the Search tab (list of all animals), then there is not
        a specific group ID or group name value to pass
    * Else, if the user is editing an animal from inside the Herd Inventory --> Groups (View) --> Group Animals
        then pass appropriate group data
        - i.e., `...edit-animal/group/41/feb2022/animal/543
    */
    this.router.navigate(
      [`/tabs/manage-animals/edit-animal/group/null/null/animal/${index}`],
      navigationExtras
    );
  }

  /*
  * Clear all applied filters
  * Reset filtered animals to list of all animals
  */
  clearFilters() {
    this.filteredAnimals = this.animals;
    this.criteria = null;
  }

  // Apply filter criteria to list of animals or groups
  applyFilterCriteria(criteria: any) {

    // Reset list of `appliedFilters` to empty
    this.appliedFilters = [];

    // Initialize list of items (`myList`) to empty
    let myList = [];

    if (this.animals.length > 0) {

      // Set `myList` equal to the full list of animals (later filtered based on selected filter criteria)
      myList = this.animals;

      // If, `myList` has values (animals), iterate through the different criteria and filter list down to appropriate animals
      if (myList.length > 0) {
        for (const key in criteria) {
          if (key === 'statusID' && criteria[key]) {
            myList = myList.filter((item: any) => (+item.status_id === +criteria[key]));
            this.appliedFilters.push(` Status: ${criteria.status}`);

          }
          if (key === 'primaryID' && criteria[key]) {
            myList = myList.filter((item: any) =>
              item.primary_id.toLowerCase().indexOf(criteria[key].toLowerCase()) > -1);
            this.appliedFilters.push(` Primary ID Contains: ${criteria[key]}`);

          }
          if (key === 'sexID' && criteria[key]) {
            myList = myList.filter((item: any) => (+item.sex_id === +criteria[key]));
            this.appliedFilters.push(` Sex: ${criteria.sex}`);

          }

          if (key === 'birthStartDate' && criteria[key]) {
            const startDate = criteria[key];
            myList = myList.filter((item: any) => {
              if (item.date_birth >= startDate) {
                return item.date_birth;
              }
            });
            this.appliedFilters.push(` Birth Start Date >= ${criteria[key]};`);
          }

          if (key === 'birthEndDate' && criteria[key]) {
            const endDate = criteria[key];
            myList = myList.filter((item: any) => {
              if (item.date_birth <= endDate) {
                return item.date_birth;
              }
            });
            this.appliedFilters.push(` Birth End Date <= ${criteria[key]};`);
          }

          if (key === 'purchaseStartDate' && criteria[key]) {
            const startDate = criteria[key];
            myList = myList.filter((item: any) => {
              if (item.date_purchase >= startDate) {
                return item.date_purchase;
              }
            });
            this.appliedFilters.push(` Purchase Start Date >= ${criteria[key]};`);
          }

          if (key === 'purchaseEndDate' && criteria[key]) {
            const endDate = criteria[key];
            myList = myList.filter((item: any) => {
              if (item.date_purchase <= endDate) {
                return item.date_purchase;
              }
            });
            this.appliedFilters.push(` Purchase End Date <= ${criteria[key]};`);
          }

          if (key === 'groupIDs' && criteria[key]) {
            myList = myList.filter((item: any) => (criteria[key].includes(+item.group_ids)));
            this.appliedFilters.push(` Group IDs: ${criteria[key]}`);

          }
          if (key === 'categoryIDs' && criteria[key]) {
            myList = myList.filter((item: any) => (+item.category_ids === +criteria[key]));
            this.appliedFilters.push(` Category IDs: ${criteria[key]}`);
          }
        }

        if (myList) {
          // Set list of filtered animals to list of animals with selected filtered applied
          this.filteredAnimals = myList;
        }

      }
    } else {
      console.log('No animals!');
    }
  }

  // Open Search Criteria page to let the user define a filter for available items
  async searchCriteria() {
    const modal = await this.modalCtrl.create({
      component: SearchCriteriaPage,
      componentProps: {
        itemType: 'animals',
        page: 'manage-animals'
      }
    });

    modal.onDidDismiss()
      .then((data: any) => {
        if (data.data) {
          console.log(data);
          console.log(data.data);
          this.filteredAnimals = this.animals;
          console.log('filteredAnimals', this.filteredAnimals);
          this.criteria = data.data;
          console.log('criteria', this.criteria);

          this.applyFilterCriteria(this.criteria);
        }
      })
      .catch((error: any) => {
        console.log('Actions Modal Search Criteria Error', error);
      });

    return await modal.present();
  }

  ionViewDidEnter() {
    this.dataService.get('getAllAnimals', null)
      .then((data: any) => {
        /*
        * Returns an object array
        * If response contains a value and has a length > 0
            then there are some animals for the current group
            set our `animals` and `filteredAnimals` variables with the result
        */
        if (data && data.length > 0) {
          this.dataService.presentLoading();
          this.animals = data;
          this.filteredAnimals = this.animals;

        } else {
          this.addService.displayAlert('Get All Animals Error',
            'An error occurred while retrieving all animals. Check your network connection and try again.');
        }
      })
      .catch((error: any) => {
        console.log(`Get All Animals Error: ${error}`);
      });
  }

  ngOnInit() {
    this.ionViewDidEnter();
  }

}
