import { Component, OnInit } from '@angular/core';
import { DataService } from 'src/app/services/data.service';
import { AddService } from 'src/app/services/add.service';
import { ActionsService } from 'src/app/services/actions.service';
import { ActivatedRoute, Router, NavigationExtras } from '@angular/router';
import { AlertController } from '@ionic/angular';

@Component({
  selector: 'app-manage',
  templateUrl: './manage.page.html',
  styleUrls: ['./manage.page.scss'],
})
export class ManagePage implements OnInit {
  public filteredGroups: any = [];
  public view = 'manage';
  public groups: any = [];
  public selected: any = [];
  // Passed `itemType` will be string: 'groups' or 'animals'
  private itemType = '';
  private passedSelected: any;

  constructor(
    private dataService: DataService,
    private addService: AddService,
    private router: Router,
    private route: ActivatedRoute,
    private alertCtrl: AlertController,
    private actionsService: ActionsService
  ) {
    this.route.queryParams.subscribe(() => {
      if (this.router.getCurrentNavigation()?.extras.state) {
        this.view = this.router.getCurrentNavigation()?.extras.state?.['viewType'];
        this.itemType = this.router.getCurrentNavigation()?.extras.state?.['itemType'];
        this.passedSelected = this.router.getCurrentNavigation()?.extras.state?.['passedSelected'];
      }
    });
  }

  pushItemIDs(groupIndex: number, currentID: number, itemIDs: any) {
    /*
    If the amount of items in the itemIDs list is greater than or equal to
        the index, then that means there is already a list
    */
    if (itemIDs.length > groupIndex) {
      itemIDs[groupIndex].push(currentID);
    } else {
      // Else, create a new list with the current Item ID (Animal ID)
      itemIDs.push([currentID]);
    }
  }

  // Move Group Action
  moveGroup() {
    // Initialize lists to hold current item IDs (`itemIDs`) and associated Group IDs (`itemGroupIDs`)
    let itemIDs: any = [];
    let itemGroupIDs: any = [];

    // If there is a passed value of the selected item(s)
    if (this.passedSelected) {

      // Iterate through the selected items and push the IDs and Group IDs to lists
      for (const p of this.passedSelected) {
        if (this.itemType === 'animals') {
          /*
          * If a comma (',') exists in the current animal's `group_ids` key,
              then the animal exists in multiple groups
          * Split the string apart (on comma) into a list of group IDs
          */
          if (p.group_ids) {
            if (p.group_ids.includes(',')) {
              // Split Group IDs based on comma separator and convert into integer values
              const splitIDs = p.group_ids.split(',').map((x: any) => +x);

              // Iterate through the split Group IDs
              for (const gID of splitIDs) {
                /*
                * Check to see if the current Group ID already exists in the
                    list of Group IDs for selected items (animals)
                * Returned index of -1 if Group ID is not present in list
                */
                if (itemGroupIDs.indexOf(gID) === -1) {
                  // Push the current Group ID to the list of Group IDs
                  itemGroupIDs.push(+gID);
                  // Grab the index of the Group ID that was just added to the list of Group IDs
                  const groupIndex = itemGroupIDs.indexOf(+gID);
                  /*
                  * Using the `index` value, current item ID (Animal ID), and the list of item IDs
                  * Append the item to the appropriate portion of the item IDs list based on the index value
                      of the associated Group ID from the Group ID List
                  */
                  this.pushItemIDs(groupIndex, +p.id, itemIDs);
                }
              }
            } else {
              // If the current Group ID associated with the Current Animal ID is not in the list
              // Push that Group ID into the list of Item Group IDs
              if (!itemGroupIDs.includes(+p.group_ids)) {
                itemGroupIDs.push(+p.group_ids);
              }

              const groupIndex = itemGroupIDs.indexOf(+p.group_ids);
              this.pushItemIDs(groupIndex, +p.id, itemIDs);
            }
          } else {
            return;
          }
        } else if (this.itemType === 'groups') {
          itemGroupIDs.push(+p.id);
        }
      }
    }

    // If there is an `itemType`, a selected item (`selected.id`) and both IDs lists have values
    if (this.itemType && this.selected.id && itemIDs.length > 0 && itemGroupIDs.length > 0) {
      this.actionsService.moveGroup(this.itemType, itemGroupIDs, this.selected.id, itemIDs)
        .then((response: any) => {
          /*
          * If there is a response and the response number is equal to the length of IDs
          * Navigate back to the actions page
          */
          if (response && response === 'success') {
            this.actionsService.actionToast('move', this.itemType);
            this.router.navigateByUrl('/tabs/home');
          } else {
            this.actionsService.actionError('Actions Manage Move Response Error');
          }
        })
        .catch((error: any) => {
          console.log(`Manage Actions Move Group Error: ${error}`);
          this.actionsService.actionError('Actions Manage Move Service Error');
        });
    } else {
      this.actionsService.actionError('Actions Manage No Values Set Error');
    }
  }

  /*
  * NOTE: Is this a bug with Ionic/Capacitor???
  * EXPLANATION: The `group` being selected (checked) is not aligned with the `isChecked` value
  * I am assuming the program wants an initial checked value to be present when the page opens, thus
      when you check an item to begin with, you are "un-checking" it in the mind of the code
  * Keep in mind we need to check the opposite in this case
  * `group.isChecked` FALSE to begin with, even after checking group on list
  * `group.isChecked` TRUE when you un-check group on list
  * There might be a `prevent default` settings, but at the time of this writing there was not one
  * Functionality of code works as intended, but find better solution in long term
  */
  checkboxChanged(group: any) {
    if (this.selected) {
      if (group.id === this.selected.id) {
        if (this.selected.isChecked) {
          this.selected = null;
        } else if (!this.selected.isChecked) {
          this.selected = group;
        }
      } else {
        this.selected.isChecked = false;
        this.selected = group;
      }
    } else {
      this.selected = group;
    }
  }

  // If the user is in the `settings` view and chooses to edit a group
  editGroup(group: any) {
    const navigationExtras: NavigationExtras = {
      state: {
        data: group
      }
    };

    // Navigate to the add page for multiple animals and pass the current group ID
    this.router.navigateByUrl('/tabs/settings/manage/edit-group', navigationExtras);
  }

  // Navigate to new page view that displays selected group's animals for current user
  viewGroupAnimals(groupID: number, groupName: string) {
    console.log(`viewGroupAnimals() groupID: ${groupID}`);
    console.log(`viewGroupAnimals() groupName: ${groupName}`);
    this.router.navigate([`/tabs/home/manage/group-animals/${groupID}/${groupName}/0`]);
  }

  // Filter groups based on group name
  filterGroups(event: any) {
    const searchTerm = event.target.value;
    
    if (searchTerm) {
      console.log('filterGroups searchTerm', searchTerm);
      console.log(this.groups);
      return this.filteredGroups = this.groups.filter((group: any) =>
        group.group_name.toLowerCase().indexOf(searchTerm.toLowerCase()) > -1);
    } else {
      return this.filteredGroups = this.groups;
    }
  }

  /*
  * Prompt user when adding multiple groups
  * Allow user input; default to 1 if null
  */
  async multipleGroups() {
    const alert = await this.alertCtrl.create({
      header: 'Amount of Groups',
      message: 'How many groups are you adding?',
      inputs: [
        {
          name: 'amount',
          type: 'number',
          placeholder: 'enter amount of groups',
          min: 0,
          max: 100
        },
      ],
      buttons: [
        {
          text: 'Cancel',
          role: 'cancel',
          cssClass: 'alertRed'
        },
        {
          text: 'Continue',
          cssClass: 'alertGreen',
          handler: (data) => {
            /*
            * If, the user input a value, convert string to integer and continue
            * If the amount entered is greater than 100, alert user
            * Else, continue with the amount entered
            */
            if (+data.amount && !isNaN(+data.amount)) {
              if (+data.amount > 100) {
                // Limit to user to only add a maximum of 100 groups at a time to prevent potential abuse
                this.addService.displayAlert('Too Many Groups',
                  'You can only add a maximum of 100 groups at a time.');
              } else {
                this.router.navigateByUrl(`/tabs/home/manage/add-group/multiple/${data.amount}`);
              }
            } else {
              this.router.navigateByUrl('/tabs/home/manage/add-group/multiple/1');
            }
          }
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  // Prompt to user to determine how many groups they are adding (individual or multiple)
  async addGroup() {
    const alert = await this.alertCtrl.create({
      header: 'Add Group',
      message: 'Would you like to add an individual group or multiple groups?',
      buttons: [
        {
          text: 'Individual Group',
          cssClass: 'alertGreen',
          handler: () => {
            this.router.navigateByUrl('/tabs/home/manage/add-group/individual/1');
          }
        },
        {
          text: 'Multiple Groups',
          cssClass: 'alertGreen',
          handler: () => {
            this.multipleGroups();
          }
        },
        {
          text: 'Cancel',
          role: 'cancel',
          cssClass: 'alertRed'
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  ionViewDidEnter() {
    this.dataService.get('getGroups', null)
      .then((data: any) => {
        if (data && data.length > 0) {
          this.dataService.presentLoading();
          this.groups = data;

          /*
          * If we are dealing with the `/actions/manage` view
              remove any groups represented in our passed animals selected list from the view
          */
          if (this.view === 'actions' && this.itemType && this.groups && this.passedSelected) {
            for (const p of this.passedSelected) {
              if (this.itemType === 'animals') {
                let currentGroupIDs = [];

                /*
                * If a comma (',') exists in the current animal's `group_ids` key, then the animal exists in multiple groups
                * Split the string apart (on comma) into a list of group IDs
                */
                if (p.group_ids) {
                  if (p.group_ids.includes(',')) {
                    currentGroupIDs = p.group_ids.split(',');
                  } else {
                    /*
                    * Else, there is only one (1) group ID associated with the animal, so the animal is only in one (1) group
                    * Set `currentGroupID` to the group ID of the animal (as a number)
                    */
                    currentGroupIDs.push(+p.group_ids);
                  }
                }

                if (currentGroupIDs.length > 0) {
                  for (const id of currentGroupIDs) {
                    /*
                    * Compare list of groups to passed animal's current group ID
                    * If there is a match, set variable `groupIndex`
                    */
                    const groupIndex = this.groups.findIndex((g: any) => g.id === +id);

                    // If `groupIndex` has a value, splice it from the full list of groups
                    if (groupIndex >= 0) {
                      this.groups.splice(groupIndex, 1);
                    }
                  }
                }
              } else if (this.itemType === 'groups') {
                /*
                * Compare list of groups to passed group's current ID
                * If there is a match, set variable `groupIndex`
                */
                const groupIndex = this.groups.findIndex((g: any) => g.id === p.id);

                // If `groupIndex` has a value, splice it from the full list of groups
                if (groupIndex >= 0) {
                  this.groups.splice(groupIndex, 1);
                }
              }
            }
          }

          // Set `filteredGroups` equal to `groups`
          this.filteredGroups = this.groups;
        }
      })
      .catch((error: any) => {
        console.log(`Manage Get Groups Error: ${error}`);
        this.addService.displayAlert('Get Groups Error',
          'An error occurred while retrieving groups. Check your network connection and try again.');
      });
  }

  ngOnInit() {
  }

}
