import { Component, OnInit } from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { FormBuilder, FormGroup, FormControl } from '@angular/forms';
import { AlertController, ModalController } from '@ionic/angular';
import { AddService } from 'src/app/services/add.service';
import { DataService } from 'src/app/services/data.service';
import { ReportsResultPage } from '../reports-result/reports-result.page';
import { ReportsModalPage } from '../reports-modal/reports-modal.page';
import { SearchCriteriaPage } from '../search-criteria/search-criteria.page';
import { ShareService } from '../../services/share.service';

@Component({
  selector: 'app-reports-criteria',
  templateUrl: './reports-criteria.page.html',
  styleUrls: ['./reports-criteria.page.scss'],
})
export class ReportsCriteriaPage implements OnInit {
  public groups: any = [];
  public animals: any = [];
  public filteredAnimals: any = [];
  public selected: any;
  public reportType: any;
  public reportsForm: FormGroup;
  public noResults = false;
  public filter: any;
  public criteria: any;
  public appliedFilters: any = [];
  private startingDate: any;
  private endingDate: any;
  private statusID: any;
  private weightType: any;
  private loading: any;

  constructor(
    public formBuilder: FormBuilder,
    private route: ActivatedRoute,
    private router: Router,
    private modalCtrl: ModalController,
    private dataService: DataService,
    private addService: AddService,
    private alertCtrl: AlertController,
    private shareService: ShareService
  ) {
    this.reportsForm = formBuilder.group({
      startDate: new FormControl(null),
      endDate: new FormControl(null),
      status: new FormControl('1'),
      weightType: new FormControl('0')
    });
  }

  async shareCSV() {
    try {
      if (this.reportType === 'inventory' && this.animals) {
        await this.shareService.generateAndShareCSV('Inventory', this.animals);
        console.log('Share successful');
      }
    } catch (error) {
      console.error('Error creating CSV and sharing:', error);
    }
  }

  // Open Search Criteria page to let the user define a filter for available items
  async searchCriteria() {
    const modal = await this.modalCtrl.create({
      component: SearchCriteriaPage,
      componentProps: {
        itemType: 'animals',
        page: 'reports-criteria'
      }
    });

    modal.onDidDismiss()
      .then((data: any) => {
        if (data.data) {
          console.log(`data: ${JSON.stringify(data)}`);
          console.log(`data.data: ${JSON.stringify(data.data)}`);
          this.filteredAnimals = this.animals;
          console.log(`filteredAnimals: ${JSON.stringify(this.filteredAnimals)}`);
          this.criteria = data.data;
          console.log(`criteria: ${JSON.stringify(this.criteria)}`);

          this.applyFilterCriteria(this.criteria);
        }
      })
      .catch((error: any) => {
        console.log('Actions Modal Search Criteria Error', error);
      });

    return await modal.present();
  }

  /*
  * Clear all applied filters
  * Reset filtered animals to list of all animals
  */
  clearFilters() {
    this.filteredAnimals = this.animals;
    this.criteria = null;
  }

  // Apply filter criteria to list of animals or groups
  applyFilterCriteria(criteria: any) {

    // Reset list of `appliedFilters` to empty
    this.appliedFilters = [];

    // Initialize list of items (`myList`) to empty
    let myList = [];

    if (this.animals.length > 0) {

      // Set `myList` equal to the full list of animals (later filtered based on selected filter criteria)
      myList = this.animals;

      // If, `myList` has values (animals), iterate through the different criteria and filter list down to appropriate animals
      if (myList.length > 0) {
        for (const key in criteria) {
          if (key === 'statusID' && criteria[key]) {
            myList = myList.filter((item: any) => (+item.status_id === +criteria[key]));
            this.appliedFilters.push(` Status: ${criteria.status}`);
          }

          if (key === 'primaryID' && criteria[key]) {
            myList = myList.filter((item: any) =>
              item.primary_id.toLowerCase().indexOf(criteria[key].toLowerCase()) > -1);
            this.appliedFilters.push(` Primary ID Contains: ${criteria[key]}`);
          }

          if (key === 'sexID' && criteria[key]) {
            myList = myList.filter((item: any) => (+item.sex_id === +criteria[key]));
            this.appliedFilters.push(` Sex: ${criteria.sex}`);
          }

          if (key === 'birthStartDate' && criteria[key]) {
            const startDate = criteria[key];
            myList = myList.filter((item: any) => {
              if (item.date_birth >= startDate) {
                return item.date_birth;
              }
            });
            this.appliedFilters.push(` Birth Start Date >= ${criteria[key]};`);
          }

          if (key === 'birthEndDate' && criteria[key]) {
            const endDate = criteria[key];
            myList = myList.filter((item: any) => {
              if (item.date_birth <= endDate) {
                return item.date_birth;
              }
            });
            this.appliedFilters.push(` Birth End Date <= ${criteria[key]};`);
          }

          if (key === 'purchaseStartDate' && criteria[key]) {
            const startDate = criteria[key];
            myList = myList.filter((item: any) => {
              if (item.date_purchase >= startDate) {
                return item.date_purchase;
              }
            });
            this.appliedFilters.push(` Purchase Start Date >= ${criteria[key]};`);
          }

          if (key === 'purchaseEndDate' && criteria[key]) {
            const endDate = criteria[key];
            myList = myList.filter((item: any) => {
              if (item.date_purchase <= endDate) {
                return item.date_purchase;
              }
            });
            this.appliedFilters.push(` Purchase End Date <= ${criteria[key]};`);
          }

          if (key === 'groupIDs' && criteria[key]) {
            myList = myList.filter((item: any) => (criteria[key].includes(+item.group_ids)));
            this.appliedFilters.push(` Group IDs: ${criteria[key]}`);
          }

          if (key === 'categoryIDs' && criteria[key]) {
            myList = myList.filter((item: any) => (+item.category_ids === +criteria[key]));
            this.appliedFilters.push(` Category IDs: ${criteria[key]}`);
          }
        }

        if (myList) {
          // Set list of filtered animals to list of animals with selected filtered applied
          this.filteredAnimals = myList;
        }

      }
    } else {
      console.log('No animals!');
    }
  }

  async reportsModal(searchType: 'groups' | 'animals') {
    console.log('reportsModal searchType:', searchType);

    try {
      const modal = await this.modalCtrl.create({
        component: ReportsModalPage,
        componentProps: { searchType }
      });

      await modal.present();

      const { data } = await modal.onDidDismiss();
      if (data) {
        console.log('reportsModal data:', data);
        if (searchType === 'groups') {
          this.groups = data;
          console.log('reportsModal groups:', this.groups);
        } else if (searchType === 'animals') {
          this.animals = data;
          console.log('reportsModal animals:', this.animals);
        }
      } else {
        // If there's no data, reset the corresponding array.
        if (searchType === 'groups') {
          this.groups = [];
        } else if (searchType === 'animals') {
          this.animals = [];
        }
        console.log(`reportsModal ${searchType}: No data returned`);
      }
    } catch (error) {
      console.error('Error in reportsModal:', error);
    }
  }

  async clearAlert() {
    const alert = await this.alertCtrl.create({
      header: 'Clear Current Form',
      message: 'Are you sure you want to clear the current form?',
      buttons: [
        {
          text: 'No',
          role: 'cancel',
          cssClass: 'alertRed',
          handler: () => false
        }, {
          text: 'Yes',
          cssClass: 'alertGreen',
          handler: () => {
            this.clearReport();
          }
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  clearReport() {
    // Reset variables
    this.statusID = null;
    this.startingDate = null;
    this.endingDate = null;
    this.weightType = null;
    this.groups = [];
    this.animals = [];

    // Reset (patch) the form values
    this.reportsForm.patchValue({
      status: '1',
      startDate: '',
      endDate: '',
      weightType: '0',
    });
  }

  async reportHelp() {
    const alert = await this.alertCtrl.create({
      header: 'Form Help',
      message: 'All form fields are optional. You may only chose groups or animals, not both. \
      If nothing is selected, then results from all of your groups over time will be returned.',
      buttons: [
        {
          text: 'OK',
          cssClass: 'alertGreen'
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  async openModal(response: any) {
    const modal = await this.modalCtrl.create({
      component: ReportsResultPage,
      componentProps: {
        reportType: this.reportType,
        response
      }
    });

    modal.onDidDismiss();

    return await modal.present();
  }

  reportError(num: string) {
    this.addService.displayAlert(`Reports Error ${num}`,
      'An error occurred because a valid report is not available. Check your network connection and try again.');
  }

  // Call this method when your report is done generating
  async dismissLoading() {
    if (this.loading) {
      await this.loading.dismiss();
    }
  }

  // Function to get form control value
  private getFormControlValue(controlName: string): any {
    return this.reportsForm.controls[controlName]?.value;
  }

  // Method to generate report based on selected items
  private reportWithSelectedItems(): Promise<any> {
    console.log('reportWithSelectedItems');
    // Call the 'generateReport' method on 'dataService' with selected items as parameters
    return this.dataService.generateReport(this.reportType, null, this.selected, '')
  }

  // Method to generate report based on date range and items (groups or animals)
  private reportWithDatesAndItems(): Promise<any> {

    console.log('reportWithDatesAndItems reportType:', this.reportType);
    console.log('reportWithDatesAndItems groups:', this.groups);
    console.log('reportWithDatesAndItems animals:', this.animals);

    // Prepare the data to be used in report generation
    const data = [this.startingDate, this.endingDate, this.statusID, this.weightType];

    // Determine if we are dealing with groups or animals
    // NOTE: A user cannot select both groups and animals when generating a report

    // If we are dealing with groups, then generate a report for selected group IDs
    if (this.groups?.length > 0) {
      return this.dataService.generateReport(this.reportType, data, this.groups, 'groups');
    }
    // If we are dealing with animals, then generate a report for selected animal IDs
    else if (this.animals?.length > 0) {
      return this.dataService.generateReport(this.reportType, data, this.animals, 'animals');
    }
    // If neither groups nor animals are selected, generate a report for all animals (which also means all groups)
    else if (this.groups?.length === 0 && this.animals?.length === 0) {
      return this.dataService.generateReport(this.reportType, data, null, 'all');
    }
    // If neither groups nor animals are selected, reject the Promise with an error
    else {
      return Promise.reject('No data for generating report');
    }
  }

  // Main method to generate the report
  generateReport() {
    // Start the loading spinner
    this.dataService.presentLoading();

    // Set variable based on form control values
    this.startingDate = this.getFormControlValue('startDate');
    this.endingDate = this.getFormControlValue('endDate');
    this.statusID = this.getFormControlValue('status');
    this.weightType = this.getFormControlValue('weightType');

    // Declare a variable to hold the Promise for the report generation process
    let reportPromise: Promise<any>;

    // Check if the report type requires dates and items, and if so, call the appropriate method
    if (['healthEvent', 'customProcessing', 'birth', 'weaning', 'ccr', 'pcr', 'cpr', 'growth', 'air', 'bior', 'nsbr'].includes(this.reportType)) {
      reportPromise = this.reportWithDatesAndItems();
    }
    // Check if the report type requires only selected items, and if so, call the appropriate method
    else if (['ped', 'unknown', 'inactive'].includes(this.reportType)) {
      // If no item is selected, display an alert and return
      if (!this.selected) {
        this.addService.displayAlert('Report Error', 'Please select an item.');
        return;
      }
      // If the report type is either 'unknown' or 'inactive', report an error and return
      else if (['unknown', 'inactive'].includes(this.reportType)) {
        this.reportError('7');
        return;
      }
      // Otherwise, call the method to generate report with selected items
      else {
        reportPromise = this.reportWithSelectedItems();
      }
    }
    // If the report type doesn't match any of the known types, report an error and return
    else {
      this.reportError('8');
      return;
    }

    // Process the report generation Promise
    reportPromise
      .then(response => {
        // Log the response of the report generation
        console.log('Generate Report response:', response);
        // If the response includes 'Error:', log it and report an error
        if (response.includes('Error: ')) {
          console.log('Response included "Error":', response);
          this.reportError('4');
        }
        // If no error is found in the response, open a modal with the response
        else {
          this.openModal(response);
        }
      })
      .catch(error => {
        // If there's an error in the report generation, log it and report an error
        console.log('Reports Item Catch Error:', error);
        this.reportError('6');
      })
      .finally(() => {
        // Regardless of the outcome, hide the loading spinner after the report generation
        this.dismissLoading();
      });
  }


  /*
  * NOTE: Is this a bug with Ionic/Capacitor???
  * EXPLANATION: The `item` being selected (checked) is not aligned with the `isChecked` value
  * I am assuming the program wants an initial checked value to be present when the page opens, thus
      when you check an item to begin with, you are "un-checking" it in the mind of the code
  * Keep in mind we need to check the opposite in this case
  * `item.isChecked` FALSE to begin with, even after checking item on list
  * `item.isChecked` TRUE when you un-check item on list
  * There might be a `prevent default` settings, but at the time of this writing there was not one
  * Functionality of code works as intended, but find better solution in long term
  */
  checkboxChanged(item: any) {
    // If there is already value for selected (previous selection)
    if (this.selected) {

      // If the item selected by the user is the same as our previously selected value
      if (item.id === this.selected.id) {

        // If the selected value is "checked" aka un-checked
        if (this.selected.isChecked) {

          // If so, set selected value to null
          this.selected = null;
        } else if (!this.selected.isChecked) {

          /*
          * Else if our selected value is "not checked" aka checked
          * Set our new selected value equal to the item the user selected
          */
          this.selected = item;
        }
      } else {

        /*
        * Else, the item selected by the user is not the same as our previously selected value
        * "Un-check" our current selected value and set new selected value to item
        */
        this.selected.isChecked = false;
        this.selected = item;
      }
    } else {

      /*
      * Else, there is not a selected value
      * Set it to the passed item (user selected)
      */
      this.selected = item;
    }
  }

  ionViewDidEnter() {
    this.noResults = false;

    console.log('Reports Criteria page');
    // Report (abbreviation) passed from "Reports" page
    if (this.route.snapshot.paramMap.get('report')) {
      this.reportType = this.route.snapshot.paramMap.get('report');
      console.log(`Reports Criteria; Report type: ${this.reportType}`);
    } else {
      console.log(`Reports Criteria; No passed parameter for Report.`);
      this.router.navigateByUrl('/tabs/reports');
    }

    if (this.reportType) {
      if (this.reportType === 'ped' || this.reportType === 'unknown' || this.reportType === 'inactive' || this.reportType === 'inventory') {
        /*
        * Else if, the report type is Pedigree ('ped'), Unknown sex ('unknown') or Inactive status ('inactive') report
        */
       console.log('reportType:', this.reportType);
       console.log('getAllAnimals');
        this.dataService.get('getAllAnimals', null)
          .then((data: any) => {
            if (data && data.length > 0) {
              this.dataService.presentLoading();

              // Initialize empty lists to hold animals based on type of report
              this.animals = [];
              this.filteredAnimals = [];

              if (this.reportType === 'ped') {
                // Pedigree report: Do NOT filter results; set animals as is
                this.animals = data;
              } else if (this.reportType === 'unknown') {
                // Unknown sex report: Filter the lists of animals down to only Unknown sex (sex_id === 4)
                this.animals = data.filter((s: any) => s.sex_id === 4);
              } else if (this.reportType === 'inactive') {
                // Inactive status report: Filter the lists of animals down to only Inactive status (status === 2)
                this.animals = data.filter((s: any) => s.status_id === 2);
              } else if (this.reportType === 'inventory') {
                this.animals = data;
              }

              this.filteredAnimals = this.animals;

              if (this.reportType === 'unknown' || this.reportType === 'inactive') {
                if (this.animals?.length > 0) {
                  console.log('Unknown or Inactive animals:', this.animals);
                  this.openModal(this.animals);
                } else {
                  this.noResults = true;
                }
              }

            } else {
              this.addService.displayAlert('Animals Error',
                'An error occurred while retrieving animals. Check your network connection and try again.');
            }
          })
          .catch((error: any) => {
            this.noResults = true;
            console.log(`Reports Criteria; Animals Error: ${error}`);
            console.log(`Reports Criteria; Animals Error typeof: ${typeof (error)}`);
          });
      }
    }
  }

  ngOnInit() {

  }

}
