import { Component, OnInit } from '@angular/core';
import { ModalController } from '@ionic/angular';
import { AddService } from 'src/app/services/add.service';
import { DataService } from 'src/app/services/data.service';

@Component({
  selector: 'app-select-bull',
  templateUrl: './select-bull.page.html',
  styleUrls: ['./select-bull.page.scss'],
})
export class SelectBullPage implements OnInit {
  public bulls: any;
  public filteredBulls: any;
  public selected: any;

  constructor(
    private modalCtrl: ModalController,
    private dataService: DataService,
    private addService: AddService
  ) { }

  /*
  * Dismiss `Select Bull` page modal
  * Pass back selected bull
  */
  done() {
    this.modalCtrl.dismiss(this.selected);
  }

  /*
  * Dismiss `Select Bull` page modal
  * If a bull was selected, pass value back otherwise pass NULL
  */
  dismiss() {
    if (this.selected) {
      this.modalCtrl.dismiss(this.selected);
    } else {
      this.modalCtrl.dismiss(null);
    }
  }

  /*
  * NOTE: Is this a bug with Ionic/Capacitor???
  * EXPLANATION: The `bull` being selected (checked) is not aligned with the `isChecked` value
  * I am assuming the program wants an initial checked value to be present when the page opens, thus
      when you check an item to begin with, you are "un-checking" it in the mind of the code
  * Keep in mind we need to check the opposite in this case
    - `bull.isChecked` FALSE to begin with, even after checking bull on list
    - `bull.isChecked` TRUE when you un-check bull on list
  * There might be a `prevent default` settings, but at the time of this writing there was not one
  * Functionality of code works as intended, but find better solution in long term
  */
  checkboxChanged(bull: any) {
    // If there is already a selected bull (previous selection)
    if (this.selected) {
      // If the bull selected by the user is the same as the previously selected bull
      if (bull.id === this.selected.id) {
        /*
        * Check to see if the selected value is "checked" aka un-checked
        * If so, set selected value to null
        */
        if (this.selected.isChecked) {
          this.selected = null;
        } else if (!this.selected.isChecked) {
          /*
          * Else, the selected value is "un-checked" aka checked
          * Set new selected value equal to the bull the user selected
          */
          this.selected = bull;
        }
      } else {
        /*
        * Else, the bull selected by the user is not the same as the previously selected value
        * "Un-check" current selected value and set new select value to bull selected by user
        */
        this.selected.isChecked = false;
        this.selected = bull;
      }
    } else {
      // Else, there is not a selected bull, set it to the passed bull (user selected)
      this.selected = bull;
    }
  }

  // Filter bulls based on primary ID
  filterAnimals(event: any) {
    this.filteredBulls = this.bulls.filter((bull: any) =>
      bull.primary_id.toLowerCase().indexOf(event.detail.value.toLowerCase()) > -1);
  }

  ionViewDidEnter() {
    this.dataService.get('getAllAnimals', null)
      .then((data: any) => {
        if (data && data.length > 0) {
          this.dataService.presentLoading();
          // Filter the lists of animals down to only Bulls (sex_id === 1)
          this.bulls = data.filter((s: any) => s.sex_id === 1);
          this.filteredBulls = this.bulls;
        }
      })
      .catch((error: any) => {
        console.log(`Bulls Error: ${error}`);
        this.addService.displayAlert('Bulls Error',
          'An error occurred while retrieving bulls. Check your network connection and try again.');
      });
  }

  ngOnInit() { }

}
