import { Component, OnInit } from '@angular/core';
import { DataService } from 'src/app/services/data.service';
import { AddService } from 'src/app/services/add.service';
import { ModalController } from '@ionic/angular';

@Component({
  selector: 'app-select-dam-sire',
  templateUrl: './select-dam-sire.page.html',
  styleUrls: ['./select-dam-sire.page.scss'],
})
export class SelectDamSirePage implements OnInit {
  public selected: any;
  public animals: any;
  public selectTypeCapital: any;
  public filteredAnimals: any;
  private selectType: any;
  private currentAnimalID: any;
  private currentAnimalDamID: any;
  private currentAnimalSireID: any;

  constructor(
    private modalCtrl: ModalController,
    private dataService: DataService,
    private addService: AddService
  ) { }

  /*
  * Dismiss `Select Dam Sire` page modal
  * Pass back selected animal
  */
  done() {
    this.modalCtrl.dismiss(this.selected);
  }

  /*
  * Dismiss `Select Dam Sire` page modal
  * User did not select a dam or sire animal
  */
  dismiss() {
    if (this.selected) {
      this.modalCtrl.dismiss(this.selected);
    } else {
      this.modalCtrl.dismiss(null);
    }
  }

  /*
  * NOTE: Is this a bug with Ionic/Capacitor???
  * EXPLANATION: The `animal` being selected (checked) is not aligned with the `isChecked` value
  * I am assuming the program wants an initial checked value to be present when the page opens, thus
      when you check an item to begin with, you are "un-checking" it in the mind of the code
  * Keep in mind we need to check the opposite in this case
    - `animal.isChecked` FALSE to begin with, even after checking animal on list
    - `animal.isChecked` TRUE when you un-check animal on list
  * There might be a `prevent default` settings, but at the time of this writing there was not one
  * Functionality of code works as intended, but find better solution in long term
  */
  checkboxChanged(animal: any) {
    // If there is already a selected animal (previous selection)
    if (this.selected) {
      // If the animal selected by the user is the same as the previously selected animal
      if (animal.id === this.selected.id) {
        /*
        * Check to see if the selected value is "checked" aka un-checked
        * If so, set selected value to null
        */
        if (this.selected.isChecked) {
          this.selected = null;
        } else if (!this.selected.isChecked) {
          /*
          * Else, the selected value is "un-checked" aka checked
          * Set new selected value equal to the animal the user selected
          */
          this.selected = animal;
        }
      } else {
        /*
        * Else, the animal selected by the user is not the same as the previously selected value
        * "Un-check" current selected value and set new select value to animal selected by user
        */
        this.selected.isChecked = false;
        this.selected = animal;
      }
    } else {
      // Else, there is not a selected animal, set it to the passed animal (user selected)
      this.selected = animal;
    }
  }

  // Filter animals based on primary ID
  filterAnimals(event: any) {
    this.filteredAnimals = this.animals.filter((animal: any) =>
      animal.primary_id.toLowerCase().indexOf(event.detail.value.toLowerCase()) > -1);
  }

  setAnimals() {
    // Make select type (dam or sire) capitalized, so it can be set as page title in header
    this.selectTypeCapital = this.selectType.charAt(0).toUpperCase() + this.selectType.slice(1);

    /*
    * If there is a value for the animal, then the user is editing an animal (animal already in database)
    * Otherwise, the user is adding an animal (animal not in database)
    */
    if (this.currentAnimalID) {
      /*
      * Find index of current animal we are editing and remove from list of animals
          so the user cannot select an animal as it's own dam/sire
      */
      const animalIndex = this.animals.findIndex((d: any) => d.id === this.currentAnimalID);
      // Remove animal from array
      this.animals.splice(animalIndex, 1);
    }

    // If dealing with selection of `Dam`
    if (this.selectType === 'dam') {

      // Remove all animals that do not have `sex` of `Cow` (`sex_id` = 3)
      this.animals = this.animals.filter((animal: any) => animal.sex_id === 3);

      // If the sire ID has a value, remove that animal from the list of animals that can be selected for Sire
      if (this.currentAnimalSireID) {
        const sireIndex = this.animals.findIndex((d: any) => d.id === this.currentAnimalSireID);
        // -1 means there is not result (index)
        if (sireIndex > -1) {
          this.animals.splice(sireIndex, 1);
        }
      }

      /*
      * If there is a value for the Dam ID, then there is already a previous animal the user selected
      * Find that animal (index) in our list of animals and set it as checked and our current selected value
      */
      if (this.currentAnimalDamID) {
        const damIndex = this.animals.findIndex((d: any) => d.id === this.currentAnimalDamID);
        this.animals[damIndex].isChecked = true;
        this.selected = this.animals[damIndex];
      }
    } else if (this.selectType === 'sire') {
      // If dealing with selection of `Sire`

      // Remove all animals that do not have `sex` of `Bull` (`sex_id` = 1)
      this.animals = this.animals.filter((animal: any) => animal.sex_id === 1);

      // If the Dam ID has a value, remove that animal from the list of animals that can be selected for Dam
      if (this.currentAnimalDamID) {
        const damIndex = this.animals.findIndex((d: any) => d.id === this.currentAnimalDamID);
        // -1 means there is not result (index)
        if (damIndex > -1) {
          this.animals.splice(damIndex, 1);
        }
      }

      /*
      * If there is a value for the Sire ID, then there is already a previous animal the user selected
      * Find that animal (index) in our list of animals and set it as checked and our current selected value
      */
      if (this.currentAnimalSireID) {
        const sireIndex = this.animals.findIndex((d: any) => d.id === this.currentAnimalSireID);
        this.animals[sireIndex].isChecked = true;
        this.selected = this.animals[sireIndex];
      }
    } else {
      this.addService.displayAlert('Dam/Sire Error',
        'An error occurred determining selection type. Please try again.');
    }

    this.filteredAnimals = this.animals;
  }

  /*
  * When the page enters the view
  * Get all animals associated with the current user from the database
  */
  ionViewDidEnter() {
    this.dataService.get('getAllAnimals', null)
      .then((data: any) => {
        if (data && data.length > 0) {
          this.dataService.presentLoading();
          this.animals = data;
          this.setAnimals();
        }
      })
      .catch((error: any) => {
        console.log(`Select Dam Sire Error: ${error}`);
        this.addService.displayAlert('Dam/Sire Error',
          'An error occurred while retrieving dam/sire. Check your network connection and try again.');
      });
  }

  ngOnInit() { }

}
